/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.directories.mgmt.range;

import com.google.common.collect.Range;

import java.time.LocalDate;
import java.time.ZoneOffset;

/**
 * Diese Klasse repräsentiert eine beliebige laufende Kalenderwoche. Intern werden die Anzahl Wochen seit der Woche
 * gezählt, die am 01.01.2017 um 0:00 UTC startet. Dies ist ein Sonntag. Im Sinne dieses Programms starten Wochen immer Sonntags,
 * da so der Umschaltzeitpunkt von einer Woche auf die nächste auf das "ruhige" Wochenende gelegt werden kann.
 * <p>
 * Das Jahr 2017 wurde willkürlich gewählt, da es an einem Sonntag beginnt.
 * <p>
 * UTC und Sonntag sind fest einprogrammiert, um Probleme zu umgehen, die sich durch sich ändernde Systemeinstellungen
 * oder Zeitzoneneinstellungen usw. ergeben könnten.
 *
 * @param internalWeekNumber laufende Wochennummer seit dem 01.01.2017. Die Woche, die am 01.01.2017 startet,
 *                           bekommt die Nummer 0, die Woche die am 08.01.2017 startet die 1, usw. negative Zahlen
 *                           sind für Wochen vor 2017 möglich.
 */
public record Week(long internalWeekNumber) implements TimeRange<Week> {

	static final LocalDate root = LocalDate.of(2017, 1, 1);

	// Die Grenzwerte sind so gestaltet, dass Milli-Zeitstempel noch in gültige Wochen umgewandelt werden können.
	// (falls z. B. jemand alle Daten haben will und eine Abfrage vom minimalen bis zum maximalen Zeitstempel durchführt)
	private static final long MAX_WEEK_NUMBER = 15250282000L;
	private static final long MIN_WEEK_NUMBER = -15250286905L;

	/**
	 * Erstellt eine neue Woche.
	 *
	 * @param internalWeekNumber laufende Wochennummer
	 */
	public Week {
		// Sicherheitsabfrage um irgendwo Überläufe oder komplett unsinnige Daten zu verhindern.
		if (internalWeekNumber > MAX_WEEK_NUMBER || internalWeekNumber < MIN_WEEK_NUMBER) {
			throw new IllegalArgumentException("internalWeekNumber: " + internalWeekNumber);
		}
	}

	@Override
	public int compareTo(Week o) {
		return Long.compare(internalWeekNumber, o.internalWeekNumber);
	}

	public LocalDate getFirstDay() {
		return root.plusDays(internalWeekNumber * 7);
	}


	public LocalDate getLastDay() {
		return root.plusDays(6 + internalWeekNumber * 7);
	}

	@Override
	public Range<LocalDate> getDayRange() {
		return Range.closed(getFirstDay(), getLastDay());
	}

	public long getFirstEpochMillis() {
		return getFirstDay().atStartOfDay().atOffset(ZoneOffset.UTC).toInstant().toEpochMilli();
	}

	public long getLastEpochMillis() {
		return getLastDay().plusDays(1).atStartOfDay().atOffset(ZoneOffset.UTC).toInstant().toEpochMilli();
	}

	@Override
	public Range<Long> getEpochMillisRange() {
		return Range.closedOpen(getFirstEpochMillis(), getLastEpochMillis());
	}

	@Override
	public Week next() {
		return new Week(internalWeekNumber + 1);
	}

	@Override
	public Week previous() {
		return new Week(internalWeekNumber - 1);
	}

	@Override
	public long stepsUntil(Week other) {
		return other.internalWeekNumber - internalWeekNumber;
	}

}
