/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.directories.mgmt.util;

import de.bsvrz.ars.ars.persistence.directories.mgmt.range.TimeDomain;
import de.bsvrz.ars.ars.persistence.directories.mgmt.range.TimeRange;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

/**
 * Diese Klasse empfängt über die {@link #update(long)}-Methode laufend (monoton steigende) Archivzeiten.
 * Diese Klasse detektiert, sobald eine neue Woche (oder andere {@link TimeRange}) angefangen hat,
 * wodurch das Archivverzeichnis dann anfangen kann, ein neues Persistenzverzeichnis zu beschreiben.
 *
 * @param <T> Zeitbereichs-Typ, z. B. {@link de.bsvrz.ars.ars.persistence.directories.mgmt.range.Week}
 */
public class RangeUpdater<T extends TimeRange<T>> {

	private final TimeDomain<T> domain;
	private T lastRange;

	/**
	 * Erstellt einen neuen RangeUpdater.
	 *
	 * @param domain Zeitbereichs-Klasse (z. B. {@link de.bsvrz.ars.ars.persistence.directories.mgmt.range.WeekDomain})
	 */
	public RangeUpdater(TimeDomain<T> domain) {
		this.domain = domain;
	}

	/**
	 * Wird aufgerufen, sobald eine neue Archivzeit "gemessen" wurde.
	 *
	 * @param monotoneArchiveTime monoton steigende Archivzeit in Epoch-Millis.
	 * @return ein Zeitbereich (eine Woche) wenn eine neue Woche angefangen hat, sonst null.
	 */
	@Nullable
	public T update(long monotoneArchiveTime) {
		T currentRange = getRange(monotoneArchiveTime);
		if (lastRange == null || lastRange.compareTo(currentRange) < 0) {
			lastRange = currentRange;
			return currentRange;
		}
		return null;
	}

	/**
	 * Gibt zu einer Archivzeit die zugehörige Woche zurück.
	 *
	 * @param monotoneArchiveTime Archivzeit in Epoch-Millis
	 * @return zu einer Archivzeit die zugehörige Woche
	 */
	@NotNull
	public T getRange(long monotoneArchiveTime) {
		return domain.ofEpochMillis(monotoneArchiveTime);
	}


}
