/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.gap;

import de.bsvrz.ars.ars.mgmt.tasks.DataIndexRangeSet;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.ars.ars.persistence.PersistenceManager;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

/**
 * Interface für eine Lücken-Datei
 *
 * @author Kappich Systemberatung
 */
public interface GapFile {

	/**
	 * Ermittelt das {@link GapFile} für eine Datenidentifikation
	 *
	 * @param persistenceManager Persistenzschicht
	 * @param dataIdentification Datenidentifikation
	 * @return Daten mit Lückeninformation für das Nachfordern
	 */
	static GapFile getInstance(PersistenceManager persistenceManager, IdDataIdentification dataIdentification) {
		Path path = persistenceManager.getGapFilePath(dataIdentification);
		return new GapFileImpl(path);
	}

	/**
	 * Liest die vorhandenen bekannten Datenlücken aus der Datei und fügt dieser der übergebenen Map hinzu 
 	 * @param map Map mit Datenlücken je Archivsystem
	 */
	void readGaps(Map<? super RemoteArchive, DataIndexRangeSet> map);

	/**
	 * Speichert die übergebenen Datenlücken in die Datei (vorhandene Einträge werden gelöscht)
	 * @param map Map mit Datenlücken je Archivsystem
	 */
	void writeGaps(Map<? extends RemoteArchive, ? extends DataIndexRangeSet> map);

	/**
	 * Gibt die Archivsysteme zurück, die im angegebenen Zeitbereich in der Daten stehen
	 * @param fromIndex Start-Datenindex
	 * @param toIndex   End-Datenindex
	 * @return Archivsystem-Pids
	 */
	default String[] getEntry(long fromIndex, long toIndex) {
		HashMap<RemoteArchive, DataIndexRangeSet> map = new HashMap<>();
		readGaps(map);
		ArrayList<String> result = new ArrayList<>();
		for (Map.Entry<RemoteArchive, DataIndexRangeSet> entry : map.entrySet()) {
			if (entry.getValue().contains(Util.dIdxNoModBits(fromIndex) + 1, Util.dIdxNoModBits(toIndex))) {
				result.add(entry.getKey().getPid());
			}
		}
		return result.toArray(String[]::new);
	}
}
