/*
 * Copyright 2019 by Kappich Systemberatung Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.gap;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.archive.ArchiveRequestManager;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

import java.util.Objects;

/**
 * Kapselung eines anderen Archivsystems. Referenzierung hier nur über die PID, damit auch Archivsysteme in Gap-Dateien referenziert werden können, die kein Systemobjekt (mehr) haben oder die temporär nicht verfügbar sind.
 *
 * @author Kappich Systemberatung
 */
public class RemoteArchive implements Comparable<RemoteArchive>{

	private static final Debug _debug = Debug.getLogger();
	
	private final String _pid;

	private RemoteArchive(final String pid) {
		_pid = Objects.requireNonNull(pid);
	}

	public static RemoteArchive open(String pid) {
		return new RemoteArchive(pid);
	}

	public static RemoteArchive open(final SystemObject object) {
		return new RemoteArchive(object.getPid());
	}

	@Nullable
	public ArchiveRequestManager createRequestManager(ClientDavInterface connection) {
		SystemObject object = connection.getDataModel().getObject(_pid);
		if(object != null) {
			try {
				return connection.getArchive(object);
			}
			catch(Exception e) {
				_debug.fine("Das Archiv mit der PID '" + _pid + "' ist ungültig.", e);
				return null;
			}
		}
		_debug.fine("Das Archiv mit der PID '" + _pid + "' ist ungültig.");
		return null;
	}

	public String getPid() {
		return _pid;
	}

	@Override
	public int compareTo(final RemoteArchive o) {
		return _pid.compareTo(o.getPid());
	}

	@Override
	public boolean equals(final Object o) {
		if(this == o) return true;
		if(o == null || getClass() != o.getClass()) return false;

		final RemoteArchive that = (RemoteArchive) o;

		return _pid.equals(that._pid);
	}

	@Override
	public int hashCode() {
		return _pid.hashCode();
	}

	@Override
	public String toString() {
		return _pid;
	}
}
