/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.index;

import de.bsvrz.ars.ars.mgmt.datatree.IndexImpl;
import de.bsvrz.ars.ars.persistence.index.backend.management.AbstractIndex;
import de.bsvrz.ars.ars.persistence.index.backend.management.ColumnType;
import de.bsvrz.ars.ars.persistence.index.backend.management.IndexContentDescriptor;

import java.nio.file.Path;

/**
 * Diese Klasse implementiert einen monoton steigenden Index. Jeder Eintrag des Index besteht aus Datenindex-Minimum, Datenindex-Maximum, Archivzeit-Minimum,
 * Archivzeit-Maximum, Datenzeit-Minimum, Datenzeit-Maximum und der Container-ID. In dieser Klasse ist gefordert, dass die Archivzeit monoton und die
 * ContainerID streng monoton über alle Eintraege hinweg steigend sind. Datenindex und Datenzeit duerfen Rückspruenge aufweisen (benoetigt für nachgeforderte
 * Datensätze). Daher kann diese Klasse nur nach Archivzeit suchen.
 * <p>
 * Zudem muss in jedem Eintrag das Maximum Größer oder gleich dem Minimum sein.
 * <p>
 * Wenn die Zahl der Einträge die Pufferkapazität im Speicher übersteigt, werden die Einträge in die Indexdatei geschrieben. Je größer die Pufferkapazität im
 * Speicher, umso schneller ist die Suche, weil keine Plattenzugriffe notwendig sind. Die Suche nach ContainerIDs ist als Binärsuche implementiert, die
 * automatisch im Speicher und in der Datei sucht.
 *
 * @author beck et al. projects GmbH
 * @author Phil Schrettenbrunner
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class ArchiveTimeIndexImpl extends AbstractIndex<IndexValues> implements ArchiveTimeIndex {

	/** Der Dateiname, unter dem alle Indices dieser Art gespeichert werden */
	public static final String IDX_FILENAME = IndexImpl.ArchiveTime.getFileName();

	private static final IndexContentDescriptor<IndexValues> INDEX_CONTENT_DESCRIPTOR;

	static {
		INDEX_CONTENT_DESCRIPTOR = new IndexContentDescriptor<>(IndexValues.class);
		INDEX_CONTENT_DESCRIPTOR.addRange(8, IndexValues.DataIndexMin, ColumnType.Unordered, IndexValues.DataIndexMax, ColumnType.Unordered);
		INDEX_CONTENT_DESCRIPTOR.addRange(6, IndexValues.ArchiveTimeMin, ColumnType.Increasing, IndexValues.ArchiveTimeMax, ColumnType.Increasing);
		INDEX_CONTENT_DESCRIPTOR.addRange(6, IndexValues.DataTimeMin, ColumnType.Unordered, IndexValues.DataTimeMax, ColumnType.Unordered);
		INDEX_CONTENT_DESCRIPTOR.addColumn(5, IndexValues.ContainerId, ColumnType.StrictlyIncreasing);
	}

	@Deprecated
	public enum Key {
		DI,
		AT,
		DT
	}

	/**
	 * Erzeugt einen neuen monoton steigenden Index.
	 * <p>
	 * Bei startmode==APPEND_REUSE_LAST wird der Ringpuffer mit den aktuellsten Daten aus einer vorhandenen Indexdatei gefüllt. Zusätzlich wird der letzte Eintrag
	 * in der Indexdatei zum currentEntry. Damit kann ein Container fortgesetzt und der reaktivierte Eintrag weiter aktualisiert werden.
	 * <p>
	 * Bei startmode==CREATE_NEW wird ein leerer Index erzeugt und die bisherige Indexdatei (sofern vorhanden) geloescht.
	 *
	 * @param bufferSize Anzahl der Einträge, die der die der Index im RAM halten soll
	 * @param file       Datei
	 * @throws CorruptIndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	public ArchiveTimeIndexImpl(int bufferSize, Path file) throws CorruptIndexException {
		this(bufferSize, file, INDEX_CONTENT_DESCRIPTOR);
	}

	/**
	 * Erzeugt einen neuen monoton steigenden Index.
	 * <p>
	 * Bei startmode==APPEND_REUSE_LAST wird der Ringpuffer mit den aktuellsten Daten aus einer vorhandenen Indexdatei gefüllt. Zusätzlich wird der letzte Eintrag
	 * in der Indexdatei zum currentEntry. Damit kann ein Container fortgesetzt und der reaktivierte Eintrag weiter aktualisiert werden.
	 * <p>
	 * Bei startmode==CREATE_NEW wird ein leerer Index erzeugt und die bisherige Indexdatei (sofern vorhanden) geloescht.
	 *
	 * @param bufferSize Anzahl der Einträge, die der die der Index im RAM halten soll
	 * @param file Datei
	 *
	 * @param contentDescriptor Beschreibung der Spalten (wegen Subklasse)
	 * @throws CorruptIndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	public ArchiveTimeIndexImpl(int bufferSize, Path file, final IndexContentDescriptor<IndexValues> contentDescriptor) throws CorruptIndexException {
		super(contentDescriptor, bufferSize, file);
	}


	/**
	 * Fügt einen neuen Eintrag in den Buffer ein. Bei Bedarf wird der Puffer auf Platte geschrieben.
	 *
	 * @param dIMin    Min-Wert des Datenindex
	 * @param dIMax    Max-Wert des Datenindex
	 * @param aTMin    Min-Wert der Archivzeit
	 * @param aTMax    Max-Wert der Archivzeit
	 * @param dTMin    Min-Wert der Datenzeit
	 * @param dTMax    Max-Wert der Datenzeit
	 * @param cID      Container-ID des Neueintrags
	 *
	 * @throws IndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	public void add(long dIMin, long dIMax, long aTMin, long aTMax, long dTMin, long dTMax, long cID) throws IndexException {
		checkAdd(dIMin, dIMax, aTMin, aTMax, dTMin, dTMax, cID);
		setInsertValue(IndexValues.DataIndexMin, dIMin);
		setInsertValue(IndexValues.DataIndexMax, dIMax);
		setInsertValue(IndexValues.ArchiveTimeMin, aTMin);
		setInsertValue(IndexValues.ArchiveTimeMax, aTMax);
		setInsertValue(IndexValues.DataTimeMin, dTMin);
		setInsertValue(IndexValues.DataTimeMax, dTMax);
		setInsertValue(IndexValues.ContainerId, cID);
		append();
	}

	protected void checkAdd(long dIMin, long dIMax, long aTMin, long aTMax, long dTMin, long dTMax, long cID) throws IndexException {
		// Neue Werte prüfen:
		try {
			if(dIMin < 0 || dIMax < 0 || aTMin < 0 || aTMax < 0 || dTMin < 0 || dTMax < 0 || cID <= 0) {
				throw new IllegalArgumentException("Neueintrag im monotonen Index ungültig:");
			}
		}
		catch(Exception e) {
			throw new CorruptIndexException(
					e.getMessage() + " (" + "dIMin=" + dIMin + ", dIMax=" + dIMax + ", aTMin=" + aTMin + ", aTMax=" + aTMax + ", dTMin=" + dTMin + ", dTMax="
							+ dTMax + ", contID=" + cID + ")", getFile(),
					e
			);
		}
	}

	/**
	 * Liefert den kleinsten Archivzeit-Minwert, der jemals in diesem Index gespeichert wird.
	 *
	 * @return Kleinster Archivzeitwert
	 *
	 * @throws IndexException Bei Zugriffsproblemen mit der IndexDatei
	 */
	public long getAbsoluteATMin() throws IndexException {
		return getFirst(IndexValues.DataTimeMin);
	}

	/**
	 * Gibt die anzahl Bytes zurück, die ein Indexeintrag belegt.
	 * @return Anzahl bytes
	 */
	public static int entrySize() {
		return INDEX_CONTENT_DESCRIPTOR.getEntryLengthBytes();
	}
}
