/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.index.backend.management;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.io.IOException;

/**
 * Ein SpaltenIndex innerhalb eines {@link AbstractIndex}, hiermit kann über eine Spalte der kleinste oder größte Wert bestimmt werden,
 * wenn der Index über diese Spalte sortiert abgespeichert ist.
 *
 * @author Kappich Systemberatung
 */
public interface IndexImplementation {

	/**
	 * Fragt die Indizes (Zeilennummern) eines Wertebereichs ab. Falls die Spalte, die dieser Index implementiert bspw. Datenzeitwerte enthält,
	 * wird {@code query(new LongRange(1000, 2000))} den Zeilenbereich (bspw 12...45) zurückliefern, in denen die Werte in diesem Bereich stehen.
	 * In dem Fall könnte z. B. Zeile 12 den Wert 999 enthalten (wenn Zeile 13 den Wert 1002 enthält) und Zeile 45 den Wert 1985 (wenn Zeile 46 den Wert 2002 enthält).
	 * <p>
	 * Das Verhalten entspricht hierbei Archivanfragen, d. h. der erste Datensatz vor Intervallanfang wird ebenfalls mit zurückgegeben,
	 * wenn auf dem Intervallanfang kein Datensatz liegt.
	 *
	 * @param inputRange Wertebereich
	 * @return Zeilenbereich oder null
	 * @throws IOException IO-Fehler beim Lesen des Index
	 */
	@Nullable
	LongRange query(LongRange inputRange) throws IOException;

	/**
	 * Findet den Element-Index des Eintrags zu einem Wert in diesem Index
	 * @param searchValue Zu suchender Wert
	 * @param tmp Byte-Array zum ablegen von temporären Daten. (Mindestgröße: eine Index-Zeile)
	 * @return Der Index (Zeilennummer) an dem dieser Wert vorkommt.
	 * <ul><li>Falls der Wert mehrmals vorkommt (bei nicht streng monoton ansteigenden Werten) wird
	 * <b>eine beliebige</b> passende Zeilennummer zurückgegeben.</li><li>Kommt der Wert nicht direkt vor, wird analog zu {@link java.util.Arrays#binarySearch(int[], int)}
	 * die Zahl {@code (-(<i>Einfügeposition</i>) - 1)} zurückgegeben, wobei Einfügeposition die ist, an der der Eintrag "fehlt" und eingefügt werden sollte,
	 * falls das beabsichtigt ist.</li></ul>
	 */
	long findEntryIndex(long searchValue, byte[] tmp) throws IOException;

	/**
	 * Die von diesem Objekt indexierte Spalte
	 * @return Spalte
	 */
	IndexContentDescriptor.IndexColumn getColumn();
}
