/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.index.backend.storage;

import java.io.IOException;

/**
 * Puffert den lesenden Zugriff auf ein darunterliegendes {@link FileIndexStorage}
 *
 * @author Kappich Systemberatung
 */
public class BufferedIndexStorage implements IndexStorage, AutoCloseable {

	private final FileIndexStorage _delegate;
	private final byte[] _cache;
	private final int _cacheSize;
	private long _cacheIndex = -1;
	private int _bufferedCount;

	/**
	 * Erstellt ein neues BufferedIndexStorage
	 * @param delegate {@link FileIndexStorage}
	 * @param cacheSize Größe des Puffers in Bytes
	 *
	 */
	public BufferedIndexStorage(FileIndexStorage delegate, final int cacheSize) {
		_delegate = delegate;
		_cacheSize = cacheSize;
		_cache = new byte[_cacheSize * delegate.entryByteSize()];
	}

	@Override
	public void getEntries(final long entryIndex, final int numEntries, final byte[] result, final int destPos) throws IOException {
		if(numEntries > _cacheSize) {
			// Zuviele Daten, passen nicht in cache
			_delegate.getEntries(entryIndex, numEntries, result, destPos);
			return;
		}
		// Daten in Cache laden, falls nötig
		cacheValues(entryIndex, numEntries);

		// Daten aus Cache holen
		System.arraycopy(_cache, (int)(entryIndex - _cacheIndex) * entryByteSize(), result, destPos, numEntries * entryByteSize());
	}

	/**
	 * Speichert Daten in Cache
	 * @param entryIndex Zu cachender Index. Die Funktion Cacht nach Möglichkeit weitere Daten in der Umgebung.
	 * @param numEntries Anzahl zu cachende Einträge ab entryIndex
	 * @throws IOException IO-Fehler beim Lesen der Datei
	 */
	private void cacheValues(final long entryIndex, final int numEntries) throws IOException {
		if(!isCached(entryIndex, numEntries)) {
			// Anzahl zu speichernder Elemente, normalerweise _cacheSize, außer es gibt weniger in der Datei
			int size = (int) Math.min(_cacheSize, numEntries());
			if(size == 0) {
				throw new IOException("Ungültiger Index: " + entryIndex);
			}

			_bufferedCount = size;

			// Normalerweise wird ab EntryIndex gecacht, außer das Dateiende ist im weg. Dann wird der Index nach vorne geschoben.
			_cacheIndex = Math.min(entryIndex, numEntries() - size);

			// Daten laden
			_delegate.getEntries(_cacheIndex, size, _cache, 0);
		}
	}

	/**
	 * Gibt {@code true} zurück, wenn der Index im Cache liegt
	 * @return {@code true}, wenn der Index im Cache liegt, sonst {@code false}
	 */
	private boolean isCached(final long entryIndex, final int numEntries) {
		return _cacheIndex >= 0 && entryIndex >= _cacheIndex && entryIndex + numEntries <= _cacheIndex + _cacheSize;
	}

	/**
	 * leert den Cache
	 */
	private void eraseCache() {
		_cacheIndex = -1;
		_bufferedCount = 0;
	}

	@Override
	public void setEntries(final long entryIndex, final int numEntries, final byte[] data, final int fromPos) throws IOException {
		_delegate.setEntries(entryIndex, numEntries, data, fromPos);
		eraseCache();
	}

	@Override
	public void insertEntries(final long entryIndex, final int numEntries, final byte[] data, final int fromPos) throws IOException {
		_delegate.insertEntries(entryIndex, numEntries, data, fromPos);
		eraseCache();
	}

	@Override
	public long numEntries() throws IOException {
		return _delegate.numEntries();
	}

	@Override
	public int entryByteSize() {
		return _delegate.entryByteSize();
	}

	@Override
	public void deleteEntryAtIndex(final long entryIndex) throws IOException {
		_delegate.deleteEntryAtIndex(entryIndex);
		eraseCache();
	}

	public void flush() throws IOException {
		_delegate.flush();
	}

	@Override
	public String toString() {
		return _delegate.toString();
	}

	@Override
	public void close() throws IOException {
		_delegate.close();
		eraseCache();
	}

	/**
	 * Gibt zurück, wie viele Einträge im lesepuffer gespeichert sind
	 * @return Anzahl Einträge
	 */
	public int getBufferedCount() {
		return _bufferedCount;
	}
}
