/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.index.backend.storage;

import de.bsvrz.dav.daf.util.BufferedRandomAccessFile;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;

/**
 * Implementierung des {@link IndexStorage}-Interfaces für Dateien im Dateisystem
 *
 * @author Kappich Systemberatung
 */
public final class FileIndexStorage implements IndexStorage, AutoCloseable {

	private final BufferedRandomAccessFile _file;

	private final int _entryByteSize;

	/**
	 * Erstellt ein neues FileIndexStorage
	 *
	 * @param file          Datei
	 * @param entryByteSize Größe eines Eintrags in Bytes
	 */
	public FileIndexStorage(Path file, int entryByteSize) throws IOException {
		Path parentDir = file.getParent();
		if(parentDir != null && !Files.isDirectory(parentDir)) {
			Files.createDirectories(parentDir);
		}
		_file = new BufferedRandomAccessFile(file.toFile());
		_entryByteSize = entryByteSize;
		long length = _file.length();
		if(length % entryByteSize != 0) {
			throw new IOException("Dateilänge ist ungültig: " + length + ". Erwarte Vielfaches von " + entryByteSize + ".");
		}
	}


	@Override
	public void getEntries(final long entryIndex, final int numEntries, final byte[] result, final int destPos) throws IOException {
		_file.position(entryIndex * _entryByteSize);
		_file.readFully(result, destPos, numEntries * _entryByteSize);
	}

	@Override
	public void setEntries(final long entryIndex, final int numEntries, final byte[] data, final int fromPos) throws IOException {
		_file.position(entryIndex * _entryByteSize);
		_file.write(data, fromPos, numEntries * _entryByteSize);
		_file.flush();
	}

	@Override
	public void insertEntries(final long entryIndex, final int numEntries, final byte[] data, final int fromPos) throws IOException {
		shiftFileRight(entryIndex * _entryByteSize, numEntries * _entryByteSize);
		setEntries(entryIndex, numEntries, data, fromPos);
	}

	private void shiftFileRight(final long fromPosition, final int shiftDistance) throws IOException {
		if(shiftDistance == 0) return;
		if(shiftDistance < 0) throw new IllegalArgumentException();

		final byte[] buffer = new byte[1024*8];

		long readPosUpperBound = _file.length();
		if(readPosUpperBound % entryByteSize() != 0) {
			throw new IOException("Dateilänge ist ungültig: " + readPosUpperBound + ". Erwarte Vielfaches von " + entryByteSize() + ".");
		}
		while(true) {
			long readPosLowerBound = Math.max(readPosUpperBound - buffer.length, fromPosition);
			if(readPosLowerBound == readPosUpperBound) break;

			_file.position(readPosLowerBound);
			int len = (int) (readPosUpperBound - readPosLowerBound);
			_file.readFully(buffer, 0, len);

			if(len == _entryByteSize) {
				// Workaround für Bug in Kernsoftware
				// https://gitlab.nerz-ev.de/ERZ/SWE_de.bsvrz.kernsoftware/issues/7
				_file.position(0);
			}

			_file.position(readPosLowerBound + shiftDistance);
			_file.write(buffer, 0, len);

			readPosUpperBound = readPosLowerBound;
		}
	}

	@Override
	public void deleteEntryAtIndex(final long entryIndex) throws IOException {
		shiftFileLeft((1 + entryIndex) * _entryByteSize, _entryByteSize);
	}

	public void flush() throws IOException {
		_file.flush();
	}

	private void shiftFileLeft(final long fromPosition, final int shiftDistance) throws IOException {
		if(shiftDistance == 0) return;
		if(shiftDistance < 0) throw new IllegalArgumentException();

		final byte[] buffer = new byte[1024*8];
		long fileLength = _file.length();
		if(fileLength % entryByteSize() != 0) {
			throw new IOException("Dateilänge ist ungültig: " + fileLength + ". Erwarte Vielfaches von " + entryByteSize() + ".");
		}
		long readPosLowerBound = fromPosition;
		while(true) {
			long readPosUpperBound = Math.min(readPosLowerBound + buffer.length, fileLength);
			if(readPosLowerBound == readPosUpperBound) break;

			_file.position(readPosLowerBound);
			int len = (int) (readPosUpperBound - readPosLowerBound);
			_file.readFully(buffer, 0, len);

			_file.position(readPosLowerBound - shiftDistance);
			_file.write(buffer, 0, len);

			readPosLowerBound = readPosUpperBound;
		}
		_file.setLength(_file.length() - _entryByteSize);
	}

	@Override
	public long numEntries() throws IOException {
		long length = _file.length();
		if(length % entryByteSize() != 0) {
			throw new IOException("Dateilänge ist ungültig: " + length + ". Erwarte Vielfaches von " + entryByteSize() + ".");
		}
		return length / _entryByteSize;
	}

	@Override
	public int entryByteSize() {
		return _entryByteSize;
	}

	@Override
	public String toString() {
		return getClass().getSimpleName();
	}

	@Override
	public void close() throws IOException {
		_file.close();
	}
}
