/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.index.backend.storage;

import java.io.IOException;

/**
 * Interface für die Low-Level-Speicherungsschicht für Indexe. Ein Index ist eine (sortierte) Liste von Index-Einträgen einer festen Byte-Größe.
 * <p>
 * Einen Index kann man sich vorstellen wie eine Excel-Tabelle, wobei Implementierungen dieses Interfaces keine Informationen über die einzelnen Spalten benötigen
 * oder wie die Bytes einer Zeile auf die einzelnen Spalten abgebildet werden.
 * <p>
 * Klassen, die dieses Interface implementieren, speichern eine Menge von diesen Byte-Einträgen ab und ermöglichen einen wahlfreien Zugriff sowie
 * das Hinzufügen von Einträgen.
 *
 * @author Kappich Systemberatung
 */
public interface IndexStorage {

	/**
	 * Gibt den ersten (obersten) Eintrag des Index zurück
	 * @param result Byte-Array in das das Ergebnis kopiert wird, muss mindestens {@link #entryByteSize()} Bytes groß sein.
	 * @throws IOException IO-Fehler (z. B. bei Dateizugriff)
	 */
	default void getFirst(byte[] result) throws IOException {
		getEntries(0, 1, result, 0);
	}

	/**
	 * Gibt den letzten (untersten) Eintrag des Index zurück
	 * @param result Byte-Array in das das Ergebnis kopiert wird, muss mindestens {@link #entryByteSize()} Bytes groß sein.
	 * @throws IOException IO-Fehler (z. B. bei Dateizugriff)
	 */
	default void getLast(byte[] result) throws IOException {
		getEntries(numEntries() - 1, 1, result, 0);
	}

	/**
	 * Gibt Einträge aus dem Index zurück
	 *
	 * @param entryIndex Nullbasierte Position des ersten Eintrags, der angefragt werden soll. Gemeint ist hier die "Zeilennummer" im Index, nicht der byte-Offset
	 * @param numEntries Anzahl Einträge, die angefragt werden sollen
	 * @param result     Byte-Array, in das das Ergebnis kopiert werden soll, muss mindestens destPos + (numEntries * {@link #entryByteSize()}) Bytes groß sein.
	 * @param destPos    Ziel-Position im Byte-Array. Standardmäßig 0.
	 * @throws IOException IO-Fehler (z. B. bei Dateizugriff)
	 */
	void getEntries(long entryIndex, int numEntries, byte[] result, int destPos) throws IOException;

	/**
	 * Gibt Einträge aus dem Index zurück
	 * @param entryIndex Nullbasierte Position des ersten Eintrags, der angefragt werden soll. Gemeint ist hier die "Zeilennummer" im Index, nicht der byte-Offset
	 * @param numEntries Anzahl Einträge, die angefragt werden sollen
	 * @param result     Byte-Array, in das das Ergebnis kopiert werden soll, muss mindestens destPos + (numEntries * {@link #entryByteSize()}) Bytes groß sein.
	 * @throws IOException IO-Fehler (z. B. bei Dateizugriff)
	 */
	default void getEntries(long entryIndex, int numEntries, byte[] result) throws IOException {
		getEntries(entryIndex, numEntries, result, 0);
	}

	/**
	 * Setzt Einträge im Index
	 * @param entryIndex Nullbasierte Position des ersten Eintrags, der gesetzt werden soll. Gemeint ist hier die "Zeilennummer" im Index, nicht der byte-Offset
	 * @param numEntries Anzahl Einträge, die überschrieben werden sollen
	 * @param data     Byte-Array, aus dem die Daten kopiert werden sollen, muss mindestens fromPos + (numEntries * {@link #entryByteSize()}) Bytes groß sein.
	 * @param fromPos    Von-Position im Byte-Array. Standardmäßig 0.
	 * @throws IOException IO-Fehler (z. B. bei Dateizugriff)
	 */
	void setEntries(long entryIndex, int numEntries, byte[] data, int fromPos) throws IOException;

	/**
	 * Setzt Einträge im Index
	 * @param entryIndex Nullbasierte Position des ersten Eintrags, der gesetzt werden soll. Gemeint ist hier die "Zeilennummer" im Index, nicht der byte-Offset
	 * @param numEntries Anzahl Einträge, die überschrieben werden sollen
	 * @param data     Byte-Array, aus dem die Daten kopiert werden sollen, muss mindestens fromPos + (numEntries * {@link #entryByteSize()}) Bytes groß sein.
	 * @throws IOException IO-Fehler (z. B. bei Dateizugriff)
	 */
	default void setEntries(long entryIndex, int numEntries, byte[] data) throws IOException {
		setEntries(entryIndex, numEntries, data, 0);
	}

	/**
	 * Fügt Einträge im Index ein, anders als bei {@link #setEntries(long, int, byte[])} werden die dort bestehenden Daten aber nicht überschrieben, sondern nach hinten geschoben.
	 * @param entryIndex Nullbasierte Position an der Eingefügt werden soll (neuer Index des ersten eingefügten Eintrags). Gemeint ist hier die "Zeilennummer" im Index, nicht der byte-Offset
	 * @param numEntries Anzahl Einträge, die eingefügt werden sollen
	 * @param data     Byte-Array, aus dem die Daten kopiert werden sollen, muss mindestens (numEntries * {@link #entryByteSize()}) Bytes groß sein.
	 * @throws IOException IO-Fehler (z. B. bei Dateizugriff)
	 */
	default void insertEntries(long entryIndex, int numEntries, byte[] data) throws IOException {
		insertEntries(entryIndex, numEntries, data, 0);
	}

	/**
	 * Fügt Einträge im Index ein, anders als bei {@link #setEntries(long, int, byte[])} werden die dort bestehenden Daten aber nicht überschrieben, sondern nach hinten geschoben.
	 * @param entryIndex Nullbasierte Position an der Eingefügt werden soll (neuer Index des ersten eingefügten Eintrags). Gemeint ist hier die "Zeilennummer" im Index, nicht der byte-Offset
	 * @param numEntries Anzahl Einträge, die eingefügt werden sollen
	 * @param data     Byte-Array, aus dem die Daten kopiert werden sollen, muss mindestens (numEntries * {@link #entryByteSize()}) Bytes groß sein.
	 * @param fromPos    Von-Position im Byte-Array. Standardmäßig 0.
	 * @throws IOException IO-Fehler (z. B. bei Dateizugriff)
	 */
	void insertEntries(long entryIndex, int numEntries, byte[] data, int fromPos) throws IOException;

	/**
	 * Gibt die Anzahl Einträge zurück
	 * @return die Anzahl Einträge
	 */
	long numEntries() throws IOException;

	/**
	 * Gibt die Byte-Größe eines Eintrags zurück
	 * @return die Byte-Größe eines Eintrags
	 */
	int entryByteSize();

	/**
	 * Fügt Einträge am Ende ein.
	 * @param numEntries Anzahl Einträge, die eingefügt werden sollen
	 * @param data     Byte-Array, aus dem die Daten kopiert werden sollen, muss mindestens (numEntries * {@link #entryByteSize()}) Bytes groß sein.
	 * @throws IOException IO-Fehler (z. B. bei Dateizugriff)
	 */
	default void addLast(byte[] data, int numEntries) throws IOException {
		insertEntries(numEntries(), numEntries, data);
	}

	/**
	 * Fügt einen Eintrag ein.
	 * @param data     Byte-Array, aus dem die Daten kopiert werden sollen, muss mindestens {@link #entryByteSize()} Bytes groß sein. Weitere Bytes werden ignoriert.
	 * @throws IOException IO-Fehler (z. B. bei Dateizugriff)
	 */
	default void addLast(byte[] data) throws IOException {
		addLast(data, 1);
	}

	/**
	 * Gibt alle Einträge als ein großes byte-Array zurück. Diese Methode verbraucht möglicherweise viel Speicher und hat eine fragwürdige Performance, ist daher
	 * hauptsächlich für Unit-Tests gedacht.
	 * @return Alle gespeicherten Daten.
	 * @throws IOException IO-Fehler (z. B. bei Dateizugriff)
	 */
	default byte[] getAll() throws IOException {
		byte[] result = new byte[Math.toIntExact(entryByteSize() * numEntries())];
		getEntries(0, Math.toIntExact(numEntries()), result);
		return result;
	}

	/**
	 * Löscht den Eintrag am Index entryIndex
	 * @param entryIndex Index, der gelöscht werden soll
	 */
	void deleteEntryAtIndex(long entryIndex) throws IOException;

}
