/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.index.result;

import de.bsvrz.ars.ars.persistence.index.backend.management.ColumnType;
import de.bsvrz.ars.ars.persistence.index.backend.management.IndexContentDescriptor;
import java.util.ArrayList;
import java.util.List;

/**
 * Diese Klasse repräsentiert das Ergebnis einer Indexanfrage aus der Klasse {@link de.bsvrz.ars.ars.persistence.index.backend.management.AbstractIndex}.
 * <p>
 * Dieses Ergebnis kann man sich wie den Teilbereich einer {@link de.bsvrz.ars.ars.persistence.index.backend.management.AbstractIndex}-Klasse vorstellen,
 * der das Ergebnis einer Bereichsanfrage enthält. Wie die dort genannte Klasse kann man sich diese Klasse wie eine Excel- oder Datenbanktabelle vorstellen.
 * <p>
 * Der generische Parameter E gibt die möglicherweise enthaltenen Daten/Spaltenköpfe an.
 *
 * @param <E> Spaltentyp
 */
public class BinaryIndexResult<E extends Enum<E>> implements IndexResult<E> {

	/**
	 * Enthaltene Datensätze
	 */
	private final List<byte[]> _dataSets = new ArrayList<>();

	/**
	 * Definition der Indexdatei
	 */
	private final IndexContentDescriptor<E> _indexContentDescriptor;

	/**
	 * Der erste Index-Datensatz hinter den Anfragebereich (sofern vorhanden)
	 */
	private byte[] _next;

	/**
	 * Erstellt ein neues IndexResult
	 *
	 * @param indexContentDescriptor Index-Datei-Definition (siehe {@link de.bsvrz.ars.ars.persistence.index.backend.management.AbstractIndex})
	 */
	public BinaryIndexResult(final IndexContentDescriptor<E> indexContentDescriptor) {
		_indexContentDescriptor = indexContentDescriptor;
	}

	/**
	 * Fügt einen Datensatz hinzu
	 * @param data Binär kodierter einzelner Datensatz (wie in IndexDatei kodiert)
	 */
	public void add(final byte[] data) {
		_dataSets.add(data);
	}

	/**
	 * Setzt den Datensatz hinter dem Anfragebereich
	 * @param data Binär kodierter einzelner Datensatz (wie in IndexDatei kodiert)
	 */
	public void setNext(final byte[] data) {
		_next = data;
	}

	/**
	 * Gibt die Anzahl Datensätze zurück. Der {@link #getNext(Enum) Datensatz hinter dem Anfragebereich} wird nicht mitgezählt.
	 * @return die Anzahl Datensätze
	 */
	@Override
	public int size() {
		return _dataSets.size();
	}

	/**
	 * Gibt eine einzelne Information als Long-Wert zurück
	 * @param index Zeilen/Datensatzindex im Ergebnis. 0 gibt den ersten Eintrag zurück, 1 den zweiten usw.
	 * @param value Wert/Spalte, der abgefragt werden soll
	 *
	 * @return Information als Long-Wert oder -1 wenn der entsprechende Wert nicht indiziert wurde.
	 *
	 * @throws IllegalArgumentException bei ungültigen Anfragen außerhalb des Bereichs
	 */
	@Override
	public long get(int index, E value) {
		if(_indexContentDescriptor == null) {
			throw new IllegalArgumentException("Index: " + index + " Size: " + size());
		}
		IndexContentDescriptor.IndexColumn column = _indexContentDescriptor.getColumn(value);

		if(column == null) {
			return -1; 
		}
		else if(column.getType() == ColumnType.String) {
			throw new IllegalArgumentException("Spalte " + column + " kann nicht als Long-Wert abgerufen werden.");
		}

		if(index < 0 || index >= _dataSets.size()) {
			throw new IllegalArgumentException("Index: " + index + " Size: " + size());
		}
		return column.readLong(_dataSets.get(index));
	}

	/**
	 * Gibt eine einzelne Information als Integer-Wert mit Vorzeichen zurück
	 * @param index Zeilen/Datensatzindex im Ergebnis. 0 gibt den ersten Eintrag zurück, 1 den zweiten usw.
	 * @param value Wert/Spalte, der abgefragt werden soll
	 *
	 * @return Information als Integer-Wert oder -1 wenn der entsprechende Wert nicht indiziert wurde.
	 *
	 * @throws IllegalArgumentException bei ungültigen Anfragen außerhalb des Bereichs
	 */
	@Override
	public int getSignedInt(int index, E value) {
		 return (int)get(index, value);
	}

	/**
	 * Gibt eine einzelne Information als String-Wert zurück
	 * @param index Zeilen/Datensatzindex im Ergebnis. 0 gibt den ersten Eintrag zurück, 1 den zweiten usw.
	 * @param value Wert/Spalte, der abgefragt werden soll
	 *
	 * @return Information als String-Wert oder `null` wenn der entsprechende Wert nicht indiziert wurde.
	 *
	 * @throws IllegalArgumentException bei ungültigen Anfragen außerhalb des Bereichs
	 */
	@Override
	public String getString(int index, E value) {
		if(_indexContentDescriptor == null) {
			throw new IllegalArgumentException("Index: " + index + " Size: " + size());
		}
		IndexContentDescriptor.IndexColumn column = _indexContentDescriptor.getColumn(value);

		if(column == null) {
			return null;
		}
		else if(column.getType() != ColumnType.String) {
			throw new IllegalArgumentException("Spalte " + column + " kann nur als Long-Wert abgerufen werden.");
		}

		if(index < 0 || index >= _dataSets.size()) {
			throw new IllegalArgumentException("Index: " + index + " Size: " + size());
		}
		return column.readString(_dataSets.get(index));
	}

	/**
	 * Gibt eine einzelne Information als Boolean-Wert zurück
	 * @param index Zeilen/Datensatzindex im Ergebnis. 0 gibt den ersten Eintrag zurück, 1 den zweiten usw.
	 * @param value Wert/Spalte, der abgefragt werden soll
	 *
	 * @return Information als Boolean-Wert oder false wenn der entsprechende Wert nicht indiziert wurde.
	 *
	 * @throws IllegalArgumentException bei ungültigen Anfragen außerhalb des Bereichs
	 */
	@Override
	public boolean getBoolean(final int index, final E value) {
		return "1".equals(getString(index, value));
	}

	/**
	 * Gibt {@code true} zurück, wenn ein nächster Datensatz nach dem Anfragebereich ermittelt wurde
	 * @return {@code true}, wenn ein nächster Datensatz nach dem Anfragebereich ermittelt wurde, sonst {@code false}
	 */
	@Override
	public boolean hasNext() {
		return _next != null;
	}

	/**
	 * Gibt einen Wert aus dem ersten nach dem Anfragebereich liegenden Datensatz zurück
	 * @param value Spalte
	 * @return einen Wert aus dem nächsten Datensatz
	 */
	@Override
	public long getNext(E value) {
		IndexContentDescriptor.IndexColumn column = _indexContentDescriptor.getColumn(value);
		return column.readLong(_next);
	}

	/**
	 * Gibt {@code true} zurück, wenn das Ergebnis leer ist
	 * @return {@code true}, wenn das Ergebnis leer ist, sonst {@code false}
	 */
	@Override
	public boolean isEmpty() {
		return size() == 0;
	}

	/**
	 * Gibt den minimalen Wert einer Spalte innerhalb des Ergebnisses zurück
	 * @param value Spalte
	 * @return den minimalen Wert einer Spalte innerhalb des Ergebnisses
	 */
	@Override
	public long getMin(final E value) {
		IndexContentDescriptor.IndexColumn column = _indexContentDescriptor.getColumn(value);
		if(column == null || isEmpty()) throw new IllegalStateException();
		if(column.getType() == ColumnType.String) {
			throw new IllegalArgumentException("Strings können aktuell nicht verglichen werden.");
		}
		if(column.getType() == ColumnType.Increasing || column.getType() == ColumnType.StrictlyIncreasing) {
			return get(0, value);
		}
		//noinspection OptionalGetWithoutIsPresent 		// isEmpty wurde oben getestet
		return _dataSets.stream().mapToLong(column::readLong).min().getAsLong();
	}

	/**
	 * Gibt den maximalen Wert einer Spalte innerhalb des Ergebnisses zurück
	 * @param value Spalte
	 * @return den maximalen Wert einer Spalte innerhalb des Ergebnisses
	 */
	@Override
	public long getMax(final E value) {
		IndexContentDescriptor.IndexColumn column = _indexContentDescriptor.getColumn(value);
		if(column == null || isEmpty()) throw new IllegalStateException();
		if(column.getType() == ColumnType.String) {
			throw new IllegalArgumentException("Strings können aktuell nicht verglichen werden.");
		}
		if(column.getType() == ColumnType.Increasing || column.getType() == ColumnType.StrictlyIncreasing) {
			return get(size() - 1, value);
		}

		//noinspection OptionalGetWithoutIsPresent 		// isEmpty wurde oben getestet
		return _dataSets.stream().mapToLong(column::readLong).max().getAsLong();
	}

	@Override
	public List<IndexContentDescriptor.IndexColumn> getColumns() {
		return _indexContentDescriptor.getColumns();
	}

	@Override
	public String toString() {
		return "IndexResult[" + size() + "]";
	}
}
