/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.index.result;

import de.bsvrz.ars.ars.persistence.index.backend.management.IndexContentDescriptor;

import java.util.List;

/**
 * Ergebnis einer Index-Abfrage, entspricht einer Tabellen-Darstellung mit dem Typ-Parameter E als "Spaltenkopf".
 * <p>
 * Werte können mit {@link #get(int, Enum)} abgefragt werden, wobei der erste parameter die Zeile und der zweite die Spalte ist.
 * </p>
 *
 * @author Kappich Systemberatung
 */
public interface IndexResult<E extends Enum<E>> {

	/**
	 * Gibt die Anzahl Datensätze zurück. Der {@link #getNext(Enum) Datensatz hinter dem Anfragebereich} wird nicht mitgezählt.
	 * @return die Anzahl Datensätze
	 */
	int size();

	/**
	 * Gibt eine einzelne Information als Long-Wert zurück
	 * @param index Zeilen/Datensatzindex im Ergebnis. 0 gibt den ersten Eintrag zurück, 1 den zweiten usw.
	 * @param value Wert/Spalte, der abgefragt werden soll
	 *
	 * @return Information als Long-Wert oder -1 wenn der entsprechende Wert nicht indiziert wurde.
	 *
	 * @throws IllegalArgumentException bei ungültigen Anfragen außerhalb des Bereichs
	 */
	long get(int index, E value);

	/**
	 * Gibt eine einzelne Information als Integer-Wert mit Vorzeichen zurück
	 * @param index Zeilen/Datensatzindex im Ergebnis. 0 gibt den ersten Eintrag zurück, 1 den zweiten usw.
	 * @param value Wert/Spalte, der abgefragt werden soll
	 *
	 * @return Information als Integer-Wert oder -1 wenn der entsprechende Wert nicht indiziert wurde.
	 *
	 * @throws IllegalArgumentException bei ungültigen Anfragen außerhalb des Bereichs
	 */
	int getSignedInt(int index, E value);

	/**
	 * Gibt eine einzelne Information als String-Wert zurück
	 * @param index Zeilen/Datensatzindex im Ergebnis. 0 gibt den ersten Eintrag zurück, 1 den zweiten usw.
	 * @param value Wert/Spalte, der abgefragt werden soll
	 *
	 * @return Information als String-Wert oder `null` wenn der entsprechende Wert nicht indiziert wurde.
	 *
	 * @throws IllegalArgumentException bei ungültigen Anfragen außerhalb des Bereichs
	 */
	String getString(int index, E value);

	/**
	 * Gibt eine einzelne Information als Boolean-Wert zurück
	 * @param index Zeilen/Datensatzindex im Ergebnis. 0 gibt den ersten Eintrag zurück, 1 den zweiten usw.
	 * @param value Wert/Spalte, der abgefragt werden soll
	 *
	 * @return Information als Boolean-Wert oder false wenn der entsprechende Wert nicht indiziert wurde.
	 *
	 * @throws IllegalArgumentException bei ungültigen Anfragen außerhalb des Bereichs
	 */
	boolean getBoolean(int index, E value);

	/**
	 * Gibt {@code true} zurück, wenn ein nächster Datensatz nach dem Anfragebereich ermittelt wurde
	 * @return {@code true}, wenn ein nächster Datensatz nach dem Anfragebereich ermittelt wurde, sonst {@code false}
	 */
	boolean hasNext();

	/**
	 * Gibt einen Wert aus dem ersten nach dem Anfragebereich liegenden Datensatz zurück
	 * @param value Spalte
	 * @return einen Wert aus dem nächsten Datensatz
	 */
	long getNext(E value);

	/**
	 * Gibt {@code true} zurück, wenn das Ergebnis leer ist
	 * @return {@code true}, wenn das Ergebnis leer ist, sonst {@code false}
	 */
	boolean isEmpty();

	/**
	 * Gibt den minimalen Wert einer Spalte innerhalb des Ergebnisses zurück
	 * @param value Spalte
	 * @return den minimalen Wert einer Spalte innerhalb des Ergebnisses
	 */
	long getMin(E value);

	/**
	 * Gibt den maximalen Wert einer Spalte innerhalb des Ergebnisses zurück
	 * @param value Spalte
	 * @return den maximalen Wert einer Spalte innerhalb des Ergebnisses
	 */
	long getMax(E value);

	/** 
	 * Gibt die Spalten des Ergebnisses zurück. Dies entspricht den im Index definierten Spalten.
	 * @return die Spalten des Ergebnisses
	 */
	List<IndexContentDescriptor.IndexColumn> getColumns();

	@Override
	String toString();
}
