/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.iter;

import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationFailedException;
import de.bsvrz.ars.ars.persistence.ContainerDataResult;
import de.bsvrz.ars.ars.persistence.ContainerManagementData;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.ars.ars.persistence.PersistenceException;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.main.archive.ArchiveOrder;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.NoSuchElementException;

/**
 * Iteriert wie der {@link SequentialContainerIterator} über eine Menge von Containern. Diese Klasse muss benutzt werden, wenn sich die Containerdateien
 * zeitlich bzw. datenindexmäßig überschneiden.
 * <p>
 * In diesem Fall, werden die Container-Dateien in Buckets sortiert, wobei ein Bucket immer eine Menge an sich überschneidenden Containerdateien enthält. Die
 * Buckets selbst werden aufsteigend nach Datenindex sortiert.
 * <p>
 * Diese Klasse erzeugt die Buckets nicht selbst, sondern sie werden im Konstruktor übergeben. Diese Klasse übernimmt das Iterieren und Sortieren von den
 * einzelnen Datensätzen.
 *
 * @author Kappich Systemberatung
 */
public class BucketContainerIterator implements DataIterator {

	private final List<List<QueryDataSequence.IndexedContainer>> _containerFiles;
	private final ArchiveDataKind _adk;
	private DataIterator _currentDelegate;
	private int _index;
	private final ArchiveOrder _order;
	private final IdDataIdentification _idDataIdentification;
	private final DataGapManager gapManager;

	/**
	 * Erstellt einen neuen BucketContainerIterator
	 *
	 * @param containerFiles       Vorsortierte Container-Dateien. Eine Liste von Container-Dateien-Gruppen, die sich
	 *                             jeweils zeitlich überschneiden. Die äußere Liste ist nach Datenindex sortiert. 
	 *                             Die Liste darf nicht leer sein.
	 * @param adk                  Archivdatenart
	 * @param order                Sortierung
	 * @param idDataIdentification Datenidentifikation
	 * @param gapManager           Verwaltung von Datenlücken
	 * @throws PersistenceException Lesefehler beim Lesen von Daten
	 */
	public BucketContainerIterator(final List<List<QueryDataSequence.IndexedContainer>> containerFiles,
	                               final ArchiveDataKind adk,
	                               final ArchiveOrder order,
	                               final IdDataIdentification idDataIdentification, DataGapManager gapManager) throws PersistenceException {
		_containerFiles = containerFiles;
		_idDataIdentification = idDataIdentification;
		_adk = adk;
		_order = order;
		this.gapManager = gapManager;
		if (containerFiles.isEmpty()) throw new IllegalArgumentException("containerFiles");
		openBucket(_index);
	}


	private void openBucket(final int index) throws PersistenceException {
		List<QueryDataSequence.IndexedContainer> bucket = _containerFiles.get(index);
		if (bucket.size() == 1) {
			_currentDelegate = new SequentialContainerIterator(bucket, _adk, null, _idDataIdentification, gapManager);
		} else {
			final List<DataIterator> iterators = new ArrayList<>();
			for (QueryDataSequence.IndexedContainer indexedContainer : bucket) {
				iterators.add(new SequentialContainerIterator(Collections.singletonList(indexedContainer), _adk, null,
						_idDataIdentification, gapManager));
			}
			_currentDelegate = new CombineDataIterator(iterators, _order);
		}
	}

	private void closeBucket() {
		_currentDelegate.close();
	}

	@Override
	public void peek(final ContainerDataResult result) throws PersistenceException {
		_currentDelegate.peek(result);
	}

	@Override
	public ContainerDataResult peekNext() {
		if (!isEmpty()) {
			throw new IllegalStateException();
		}

		return null;
	}

	@Override
	public long peekDataIndex() {
		return _currentDelegate.peekDataIndex();
	}

	@Override
	public long peekDataTime() {
		return _currentDelegate.peekDataTime();
	}

	@Override
	public long peekArchiveTime() {
		return _currentDelegate.peekArchiveTime();
	}

	@Override
	public void remove() throws PersistenceException, SynchronizationFailedException {
		_currentDelegate.remove();
		if (_currentDelegate.isEmpty() && _index < _containerFiles.size() - 1) {
			closeBucket();
			_index++;
			openBucket(_index);
		}
	}

	@Override
	public boolean isEmpty() {
		return _currentDelegate.isEmpty();
	}

	@Override
	public void close() {
		_currentDelegate.close();
	}

	@NotNull
	@Override
	public ContainerManagementData getContainerManagementData() throws PersistenceException {
		if (_currentDelegate == null) throw new NoSuchElementException();
		return _currentDelegate.getContainerManagementData();
	}
}
