/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.iter;

import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationFailedException;
import de.bsvrz.ars.ars.persistence.ContainerDataResult;
import de.bsvrz.ars.ars.persistence.ContainerManagementData;
import de.bsvrz.ars.ars.persistence.PersistenceException;
import de.bsvrz.dav.daf.main.archive.ArchiveOrder;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

import java.util.*;

/**
 * Diese Klasse kombiniert mehrere parallel laufende {@link DataIterator Iteratoren}
 * um daraus einen einzelnen Iterator zu erstellen, der alle Daten enthält.
 * <p>
 * Die Daten werden dabei entweder anhand ihrer Datenzeit oder ihres Datenindex einsortiert.
 *
 * @author Kappich Systemberatung
 */
public class CombineDataIterator implements DataIterator {
	
	@Nullable
	private DataIterator _currentDelegate;
	
	final List<DataIterator> _iterators;
	
	private final ArchiveOrder _order;
	
	private boolean _closed;

	/**
	 * Konstruktor zum Kombinieren von mehreren {@link DataSequence}-Objekten
	 *
	 * @param containerSequences Die Sequenzen der verschiedenen Datenarten
	 * @param order              Sortierung der Sequenzen
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 * @throws SynchronizationFailedException Synchronisierung fehlgeschlagen
	 */
	public CombineDataIterator(final Collection<DataSequence> containerSequences, ArchiveOrder order) throws PersistenceException, SynchronizationFailedException {
		_order = order;
		final List<DataIterator> iterators = new ArrayList<>(containerSequences.size());
		for(DataSequence sequence : containerSequences) {
			DataIterator iterator = sequence.iterator();
			iterators.add(iterator);
		}
		_iterators = iterators;
		updateDelegate();
	}

	/**
	 * Konstruktor, der intern vom {@link BucketContainerIterator} benutzt wird.
	 * @param iterators Die Iteratoren der verschiedenen Datenarten
	 * @param order Reihenfolge
	 */
	CombineDataIterator(final List<DataIterator> iterators, ArchiveOrder order) {
		_iterators = iterators;
		_order = order;
		updateDelegate();
	}

	@Override
	public void peek(final ContainerDataResult result) throws PersistenceException {
		if(_currentDelegate == null) throw new NoSuchElementException();
		_currentDelegate.peek(result);
	}

	@Override
	public long peekDataIndex() {
		if(_currentDelegate == null) throw new NoSuchElementException();
		return _currentDelegate.peekDataIndex();
	}

	@Override
	public long peekDataTime() {
		if(_currentDelegate == null) throw new NoSuchElementException();
		return _currentDelegate.peekDataTime();
	}

	@Override
	public long peekArchiveTime() {
		if(_currentDelegate == null) throw new NoSuchElementException();
		return _currentDelegate.peekArchiveTime();
	}

	@Override
	public void remove() throws PersistenceException, SynchronizationFailedException {
		if(_closed) throw new IllegalStateException("closed == true");
		if(_currentDelegate == null) throw new NoSuchElementException();
		_currentDelegate.remove();
		updateDelegate();
	}

	protected final void updateDelegate() {
		if(_order == ArchiveOrder.BY_INDEX) {
			_currentDelegate = _iterators.stream().filter(it -> !it.isEmpty()).min(Comparator.comparing(DataIterator::peekDataIndex)).orElse(null);
		}
		else {
			_currentDelegate = _iterators.stream().filter(it -> !it.isEmpty()).min(Comparator.comparing(DataIterator::peekDataTime).thenComparing(DataIterator::peekDataIndex)).orElse(null);
		}
	}

	@Override
	public boolean isEmpty() {
		return _currentDelegate == null;
	}

	@Override
	public ContainerDataResult peekNext() throws PersistenceException, SynchronizationFailedException {
		if (!isEmpty()) throw new IllegalStateException();
		final List<ContainerDataResult> results = new ArrayList<>();
		for(DataIterator iterator : _iterators) {
			ContainerDataResult next = iterator.peekNext();
			if(next != null) {
				results.add(next);
			}
		}
		if(results.isEmpty()) return null;
		if(_order == ArchiveOrder.BY_INDEX) {
			return Collections.min(results, Comparator.comparing(ContainerDataResult::getDataIndex));
		}
		else {
			return Collections.min(results, Comparator.comparing(ContainerDataResult::getDataTime));
		}
	}

	@Override
	public void close() {
		if(_closed) {
			return;
		}
		_closed = true;
		for(DataIterator iterator : _iterators) {
			iterator.close();
		}
	}

	@NotNull
	@Override
	public ContainerManagementData getContainerManagementData() throws PersistenceException {
		if(_currentDelegate == null) throw new NoSuchElementException();
		return _currentDelegate.getContainerManagementData();
	}

	/** 
	 * Gibt die Sortierung zurück
	 * @return die Sortierung
	 */
	public ArchiveOrder getOrder() {
		return _order;
	}
}
