/*
 *
 * Copyright 2017-2019 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.iter;

import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationFailedException;
import de.bsvrz.ars.ars.persistence.ContainerDataResult;
import de.bsvrz.ars.ars.persistence.ContainerManagementData;
import de.bsvrz.ars.ars.persistence.PersistenceException;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

/**
 * Ein Iterator über die Datensätze von mehreren Containern. Der Iterator verhält sich anders als der Standard-Java-iterator und mehr wie
 * eine Queue. {@link #peek(ContainerDataResult) peek} ruft den aktuellen Datensatz am Kopf der Queue ab, {@link #remove() remove} entfernt diesen,
 * sodass der nächste Datensatz abgerufen werden kann (falls vorhanden). {@link #poll(ContainerDataResult)} gibt den aktuellen Datensatz zurück und entfernt ihn in einem Schritt.
 * <p>
 * Die Iteration mit dieser Klasse erfolg wie beispielsweise folgt:
 * <pre>{@code
 *	 ContainerDataResult result = new ContainerDataResult()
 *	 while(!iter.isEmpty()) {
 *		 iter.poll(result);
 *		 // tue was mit result
 *     }
 *	 iter.close()
 * }</pre>
 *
 * oder alternative Vorgehensweise zum Ausgeben aller Datenindizes:
 *
 * <pre>{@code
 * 		while(!iterator.isEmpty()) {
 *		  System.out.println(iterator.peekDataIndex());
 *		  iterator.remove();
 *      }
 * }</pre>
 *
 * @author Kappich Systemberatung
 */
public interface DataIterator extends AutoCloseable {

	/**
	 * Kopiert die Daten des aktuellen Elements in das angegebene Result-Objekt. Dies ist eine Optimierung, damit nicht ständig neue Result-Objekt angelegt werden müssen.
	 * @param result Ziel-Element
	 * @throws java.util.NoSuchElementException falls kein Element mehr Verfügbar (isEmpty == true)
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	void peek(ContainerDataResult result) throws PersistenceException;

	/**
	 * Gibt Daten des auf die gültige Sequenz folgenden Datensatzes zurück, damit kann dann unter anderem bestimmt werden, wie lange der Datensatz gültig ist.
	 * Diese Methode darf nur aufgerufen werden, wenn {@link #isEmpty()} true zurückliefert.
	 * @return Ziel-Element oder null, falls es keinen nächsten Datensatz gibt
	 * @throws IllegalStateException falls isEmpty != true
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 * @throws SynchronizationFailedException Synchronisierung fehlgeschlagen
	 */
	@Nullable
	ContainerDataResult peekNext() throws PersistenceException, SynchronizationFailedException;

	/**
	 * Ruft den aktuellen Datenindex ab. Entspricht
	 * <pre>{@code
	 *	 ContainerDataResult result = new ContainerDataResult()
	 *	 this.peek(result)
	 *	 return result.getDataIndex()
	 * }</pre>
	 * ist aber wahrscheinlich performanter.
	 * @return Datenindex des aktuellen Elements
	 * @throws java.util.NoSuchElementException falls kein Element mehr Verfügbar (isEmpty == true)
	 */
	long peekDataIndex();

	/**
	 * Ruft die aktuelle Datenzeit ab. Entspricht
	 * <pre>{@code
	 *	 ContainerDataResult result = new ContainerDataResult()
	 *	 this.peek(result)
	 *	 return result.getDataTime()
	 * }</pre>
	 * ist aber wahrscheinlich performanter.
	 * @return Datenzeit des aktuellen Elements
	 * @throws java.util.NoSuchElementException falls kein Element mehr Verfügbar (isEmpty == true)
	 */
	long peekDataTime();

	/**
	 * Ruft die aktuelle Archivzeit ab. Entspricht
	 * <pre>{@code
	 * ContainerDataResult result = new ContainerDataResult()
	 * this.peek(result)
	 * return result.getArchiveTime()
	 * }</pre>
	 * ist aber wahrscheinlich performanter.
	 * @return Archivzeit des aktuellen Elements
	 * @throws java.util.NoSuchElementException falls kein Element mehr Verfügbar (isEmpty == true)
	 */
	long peekArchiveTime();

	/**
	 * Entspricht dem Aufruf von `peek(result)´ gefolgt von `remove()`.
	 * @param result Objekt, in dem das Ergebnis abgelegt wird.
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 * @throws SynchronizationFailedException Synchronisierung fehlgeschlagen
	 */
	default void poll(ContainerDataResult result) throws PersistenceException, SynchronizationFailedException {
		peek(result);
		remove();
	}

	/**
	 * Entfernt/Überspringt das aktuelle Element. Der Container wird nicht modifiziert, nur der Iterator.
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 * @throws SynchronizationFailedException Synchronisierung fehlgeschlagen
	 */
	void remove() throws PersistenceException, SynchronizationFailedException;

	/**
	 * Gibt {@code true} zurück, wenn der Iterator erschöpft ist. Ist das der Fall liefern alle anderen Funktionen außer close() {@link java.util.NoSuchElementException}s.
	 * @return {@code true}, wenn der Iterator erschöpft ist, sonst {@code false}
	 */
	boolean isEmpty();

	/**
	 * Terminiert das iterieren. Hierdurch werden alle ggf. noch offenen Containerdateien geschlossen.
	 */
	@Override
	void close();

	/**
	 * Gibt einen leeren DataIterator zurück
	 * @return leerer DataIterator (singleton)
	 */
	static DataIterator empty() {
		return EmptyDataIterator.getInstance();
	}

	/**
	 * Gibt die Containerheaderdaten des Containers des aktuell betrachteten Datensatzes zurück
	 * @return ContainerManagementData
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	@NotNull
	ContainerManagementData getContainerManagementData() throws PersistenceException;
}
