/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.iter;

import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SyncKey;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationFailedException;
import de.bsvrz.ars.ars.persistence.*;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.index.IndexException;
import de.bsvrz.ars.ars.persistence.index.IndexValues;
import de.bsvrz.ars.ars.persistence.index.result.LocatedIndexResult;
import de.bsvrz.dav.daf.main.archive.*;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

import java.util.*;

/**
 * Klasse zum Iterieren über Datensätze einer Datenidentifikation. Diese Klasse kümmert sich darum, die benötigten Container zu suchen und bei Bedarf aus diesen die richtigen
 * Datensätze zu extrahieren.
 *
 * @author Kappich Systemberatung
 */
public final class QueryDataSequence implements DataSequence {
	private final ArchiveDataKindCombination _archiveDataKinds;
	@NotNull
	private final ArchiveTimeSpecification _archiveTimeSpecification;
	private final PersistenceManager _persMgr;
	private final IdDataIdentification _idDataIdentification;
	private final List<DataSequence> _containerSequences = new ArrayList<>();
	private final ArchiveOrder _order;

	/**
	 * Gibt Maximum für die Anzahl von Datensätzen an, die mit einer Relativanfrage zu bekommen sind.
	 */
	private static final int MAX_INTERVAL = 16000;

	/**
	 * Erstellt eine neue QueryDataSequence. Dies ist eine allgemeine Anfrageschnittstelle, über die für eine Datenidentifikation und einen Zeitbereich sowie eine Angabe der
	 * Datenarten ein {@link DataIterator Iterator} über alle in diesem Bereich vorliegenden Daten erstellt werden kann.
	 *
	 * @param persistenceManager           Archivmanager
	 * @param archiveDataKinds         Archivdatenarten
	 * @param archiveTimeSpecification Archivzeitspezifikation (Wenn null wird der gesamte Zeitbereich betrachtet)
	 * @param order                    Einsortierungs-Reihenfolge von nachgelieferten Daten
	 * @param idDataIdentification     Datenidentifikation
	 */
	public QueryDataSequence(PersistenceManager persistenceManager,
	                         ArchiveDataKindCombination archiveDataKinds,
	                         @Nullable ArchiveTimeSpecification archiveTimeSpecification,
	                         final ArchiveOrder order,
	                         final IdDataIdentification idDataIdentification) throws PersistenceException {
		_persMgr = persistenceManager;
		_archiveDataKinds = archiveDataKinds;
		_order = order;
		_idDataIdentification = idDataIdentification;

		try (SyncKey<IdDataIdentification> indexLock = _persMgr.lockIndex(idDataIdentification)) {
			// Aus der Zeitspezifikation eine gültige absolute Angabe machen
			if (archiveTimeSpecification == null) {
				// Null heißt, den kompletten Datenbereich ermitteln
				_archiveTimeSpecification = new ArchiveTimeSpecification(TimingType.DATA_INDEX, false, 0, Long.MAX_VALUE);
			}
			else if(archiveTimeSpecification.isStartRelative()) {
				// Relativanfragen müssen in absolute Anfragen übersetzt werden
				try {
					_archiveTimeSpecification = createAbsoluteTimeSpecification(indexLock, archiveTimeSpecification);
				}
				catch(IndexException e) {
					throw new PersistenceException("Fehler beim Übersetzen der Relativanfrage", e);
				}
			}
			else {
				// Absolute Anfragen bleiben beibehalten
				_archiveTimeSpecification = archiveTimeSpecification;
			}

			if(archiveDataKinds.isOnline()) {
				DataSequence containerSequence = getContainerSequence(new LockedContainerDirectory(indexLock, ArchiveDataKind.ONLINE));
				if(containerSequence != null) {
					_containerSequences.add(containerSequence);
				}
			}
			if(archiveDataKinds.isOnlineDelayed()) {
				DataSequence containerSequence = getContainerSequence(new LockedContainerDirectory(indexLock, ArchiveDataKind.ONLINE_DELAYED));
				if(containerSequence != null) {
					_containerSequences.add(containerSequence);
				}
			}
			if(archiveDataKinds.isRequested()) {
				DataSequence containerSequence = getContainerSequence(new LockedContainerDirectory(indexLock, ArchiveDataKind.REQUESTED));
				if(containerSequence != null) {
					_containerSequences.add(containerSequence);
				}
			}
			if(archiveDataKinds.isRequestedDelayed()) {
				DataSequence containerSequence = getContainerSequence(new LockedContainerDirectory(indexLock, ArchiveDataKind.REQUESTED_DELAYED));
				if(containerSequence != null) {
					_containerSequences.add(containerSequence);
				}
			}
		}
		catch(Exception e) {
			throw new PersistenceException("Fehler beim Auflisten der Container für die Iteration", e);
		}
	}

	private QueryDataSequence(final ArchiveDataKindCombination archiveDataKinds,
	                          @NotNull final ArchiveTimeSpecification archiveTimeSpecification,
	                          final PersistenceManager persistenceManager,
	                          final ArchiveOrder order,
	                          final Collection<? extends DataSequence> sequences,
	                          final IdDataIdentification idDataIdentification) {
		_archiveDataKinds = archiveDataKinds;
		_archiveTimeSpecification = archiveTimeSpecification;
		_persMgr = persistenceManager;
		_order = order;
		_containerSequences.addAll(sequences);
		_idDataIdentification = idDataIdentification;
	}

	@Nullable
	private DataSequence getContainerSequence(LockedContainerDirectory containerDirectory)
			throws Exception {
		LocatedIndexResult<IndexValues> indexResult = _persMgr.getIndexResult(containerDirectory, _archiveTimeSpecification);
		return getDataSequence(containerDirectory, indexResult, 0, indexResult.size() - 1);
	}

	@Nullable
	private DataSequence getDataSequence(LockedContainerDirectory containerDirectory, final LocatedIndexResult<IndexValues> indexResult, final int lowerBound, final int upperBound) {
		if(indexResult.isEmpty()) return null;
		ArchiveDataKind adk = containerDirectory.archiveDataKind();

		ArrayList<IndexedContainer> indexedContainers = new ArrayList<>(indexResult.size());
		for(int i = lowerBound; i <= upperBound; i++) {
			long containerId = indexResult.get(i, IndexValues.ContainerId);
			indexedContainers.add(
					new IndexedContainer(adk, containerId, indexResult.get(i, IndexValues.DataIndexMin), indexResult.get(i, IndexValues.DataIndexMax),
							indexResult.get(i, IndexValues.DataTimeMin), indexResult.get(i, IndexValues.DataTimeMax), indexResult.getPersistenceDirectory(i)));
		}

		IndexedContainer nextContainer = null;
		if(indexResult.hasNext()) {
			long containerId = indexResult.getNext(IndexValues.ContainerId);
			nextContainer =
					new IndexedContainer(adk, containerId, indexResult.getNext(IndexValues.DataIndexMin), indexResult.getNext(IndexValues.DataIndexMax),
							indexResult.getNext(IndexValues.DataTimeMin), indexResult.getNext(IndexValues.DataTimeMax), indexResult.getNextPersistenceDirectory());
		}

		if(adk.isDelayed() && _order == ArchiveOrder.BY_DATA_TIME) {
			// Die Container sind nicht nach Datenzeit sortiert, es kann Überschneidungen geben
			// Container, sie sich in der Datenzeit überschneiden müssen daher zusammen geöffnet und betrachtet werden
			// dazu werden sie in Buckets einsortiert, die nacheinander abgearbeitet werden
			return new BucketContainerSequence(containerDirectory, indexedContainers);
		}
		else if(adk.isRequested()) {
			// Die Container sind nicht zwingenderweise nach Datenindex sortiert, es kann Überschneidungen geben
			// Container, die sich vom Index her überschneiden müssen daher zusammen geöffnet und betrachtet werden
			// dazu werden sie in Buckets einsortiert, die nacheinander abgearbeitet werden
			return new BucketContainerSequence(containerDirectory, indexedContainers);
		}
		else {
			indexedContainers.sort(Comparator.comparing(IndexedContainer::getDataIndexMinimum));
			// Einfacher Fall: Die Container sind monoton nach Datenindex sortiert
			return new SimpleContainerSequence(indexedContainers, nextContainer, adk);
		}
	}

	/**
	 * Wandelt eine relative in eine absolute Anfrage um
	 *
	 * @param lock Objekt, das den synchronisierten Zugriff auf den Index erlaubt
	 * @param ats  Relative Zeitspezifikation
	 * @return Absolute Zeitspezifikation
	 * @throws IndexException       Lesefehler der Indexe (z. B. korrupt)
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	private ArchiveTimeSpecification createAbsoluteTimeSpecification(final SyncKey<IdDataIdentification> lock, final ArchiveTimeSpecification ats) throws IndexException, PersistenceException, SynchronizationFailedException {
		long numDs = ats.getIntervalStart();
		if(numDs > MAX_INTERVAL) {
			numDs = MAX_INTERVAL;
		}
		TimingType timingType = ats.getTimingType();
		if(numDs <= 0) return new ArchiveTimeSpecification(timingType, false, ats.getIntervalEnd(), ats.getIntervalEnd());
		ArchiveTimeSpecification timeSpecification = new ArchiveTimeSpecification(timingType, false, 0, ats.getIntervalEnd());

		final Collection<DataSequence> sequences = new ArrayList<>();


		for(ArchiveDataKind adk : _archiveDataKinds) {

			LockedContainerDirectory containerDirectory = new LockedContainerDirectory(lock, adk);

			LocatedIndexResult<IndexValues> indexResult = _persMgr.getIndexResult(containerDirectory, timeSpecification);
			int lastIndex = indexResult.size() - 1;

			// In lastIndex steht jetzt der letzte relevante Container
			// jetzt über die Datensatzanzahl den ersten Container suchen
			long remainingDs = numDs;
			int tmpIndex = lastIndex;
			if(tmpIndex > 0) {
				tmpIndex--;
				while(true) {
					long containerId = indexResult.get(tmpIndex, IndexValues.ContainerId);
					remainingDs -= indexResult.getPersistenceDirectory(tmpIndex)
							.getContainerHeaders(containerDirectory, containerId).getContainerHeaderParamAsInt(ContainerManagementInformation.CHP_ANZ_DS);
					if(remainingDs <= 0 || tmpIndex == 0) {
						break;
					}
					tmpIndex--;
				}
			}

			// Im Container, den tmpIndex angibt, steht jetzt der erste Container
			int firstIndex = tmpIndex;
			DataSequence sequence = getDataSequence(containerDirectory, indexResult, firstIndex, lastIndex);
			if (sequence != null) {
				sequences.add(sequence);
			}
		}

		QueryDataSequence sequence = new QueryDataSequence(_archiveDataKinds, timeSpecification, _persMgr, _order, sequences,
				_idDataIdentification);
		// Sequence enthält jetzt alle in Frage kommenden Datensätze, aber wahrscheinlich etwas zu viel.

		// Anfrage durchführen und den relevanten Indexbereich ermitteln, sodass die genaue Anzahl Datensätze ermittelt wird.
		final ArrayDeque<Long> tmpDeque;
		try(DataIterator iterator = sequence.iterator()) {
			tmpDeque = new ArrayDeque<>((int) numDs);
			while(!iterator.isEmpty()) {
				if(tmpDeque.size() == numDs) {
					tmpDeque.removeFirst();
				}
				if(timingType == TimingType.DATA_INDEX) {
					tmpDeque.addLast(iterator.peekDataIndex());
				}
				else if(timingType == TimingType.DATA_TIME) {
					tmpDeque.addLast(iterator.peekDataTime());
				}
				else if(timingType == TimingType.ARCHIVE_TIME) {
					tmpDeque.addLast(iterator.peekArchiveTime());
				}
				iterator.remove();
			}
		}
		if(tmpDeque.isEmpty()) {
			// Es konnten keine Daten gefunden werden
			return new ArchiveTimeSpecification(timingType, false, ats.getIntervalEnd(), ats.getIntervalEnd());
		}
		return new ArchiveTimeSpecification(timingType, false, tmpDeque.peekFirst(), tmpDeque.peekLast());
	}

	@Override
	public DataIterator iterator() throws PersistenceException, SynchronizationFailedException {
		// wenn ein Zeitbereich angegeben wurde, das Ergebnis so Filtern, dass es auf diesen Zeitbereich beschränkt ist.
		return new TimeSpecificationCombineDataIterator(_containerSequences, _order, _archiveTimeSpecification);
	}

	public DataIterator iteratorWithoutTimeFilter() throws PersistenceException, SynchronizationFailedException {
		return new CombineDataIterator(_containerSequences, _order);
	}

	private class SimpleContainerSequence implements DataSequence {
		private final List<IndexedContainer> _containerFiles;
		private final ArchiveDataKind _adk;
		private final IndexedContainer _nextContainer;

		public SimpleContainerSequence(final List<IndexedContainer> containerFiles, @Nullable final IndexedContainer nextContainer, final ArchiveDataKind adk) {
			_containerFiles = containerFiles;
			_nextContainer = nextContainer;
			_adk = adk;
		}

		@Override
		public DataIterator iterator() throws PersistenceException {
			return new SequentialContainerIterator(_containerFiles, _adk, _nextContainer, _idDataIdentification, _persMgr);
		}
	}

	private final class BucketContainerSequence implements DataSequence {
		// Gruppen von sich zeitlich überschneidenden Containern aufsteigend nach Datenindex sortiert
		private final List<List<IndexedContainer>> _containerFiles = new ArrayList<>();
		private final LockedContainerDirectory directory;

		public BucketContainerSequence(final LockedContainerDirectory directory, final List<IndexedContainer> containerFiles) {
			this.directory = directory;
			if(_order == ArchiveOrder.BY_INDEX) {
				initByIndex(containerFiles);
			}
			else {
				initByDataTime(directory.lock(), containerFiles);
			}
		}

		private void initByIndex(final List<IndexedContainer> containerFiles) {
			ArrayList<IndexedContainer> tmpBlock = new ArrayList<>();
			ArrayList<IndexedContainer> tmp = new ArrayList<>(containerFiles);
			tmp.sort(Comparator.comparing(IndexedContainer::getDataIndexMinimum).reversed());
			// tmp ist absteigend nach minimalem Datenindex sortiert, da vom Ende einer ArrayList entfernen performanter ist.
			while(!tmp.isEmpty()) {
				IndexedContainer last = tmp.remove(tmp.size() - 1);
				tmpBlock.add(last);
				long min = last.getDataIndexMinimum();
				long max = last.getDataIndexMaximum();
				while(true) {
					boolean doContinue = false;
					for(Iterator<IndexedContainer> iterator = tmp.iterator(); iterator.hasNext(); ) {
						final IndexedContainer indexedContainer = iterator.next();
						if(overlapIndex(indexedContainer, min, max)) {
							iterator.remove();
							tmpBlock.add(indexedContainer);
							min = Math.min(min, indexedContainer.getDataIndexMinimum());
							max = Math.max(max, indexedContainer.getDataIndexMaximum());
							doContinue = true;
						}
					}
					if(!doContinue) break;
				}
				_containerFiles.add(new ArrayList<>(tmpBlock));
				tmpBlock.clear();
			}
		}

		public void initByDataTime(final SyncKey<IdDataIdentification> lock, final List<IndexedContainer> containerFiles) {
			ArrayList<IndexedContainer> tmpBlock = new ArrayList<>();
			ArrayList<IndexedContainer> tmp = new ArrayList<>(containerFiles);
			tmp.sort(Comparator.comparing((IndexedContainer indexedContainer1) -> indexedContainer1.getDataTimeMinimum(lock)).reversed());
			// tmp ist absteigend nach minimaler Datenzeit sortiert, da vom Ende einer ArrayList entfernen performanter ist.
			while(!tmp.isEmpty()) {
				IndexedContainer last = tmp.remove(tmp.size() - 1);
				tmpBlock.add(last);
				long min = last.getDataTimeMinimum(lock);
				long max = last.getDataTimeMaximum(lock);
				while(true) {
					boolean doContinue = false;
					for(Iterator<IndexedContainer> iterator = tmp.iterator(); iterator.hasNext(); ) {
						final IndexedContainer indexedContainer = iterator.next();
						if(overlapDataTime(lock, indexedContainer, min, max)) {
							iterator.remove();
							tmpBlock.add(indexedContainer);
							min = Math.min(min, indexedContainer.getDataTimeMinimum(lock));
							max = Math.max(max, indexedContainer.getDataTimeMaximum(lock));
							doContinue = true;
						}
					}
					if(!doContinue) break;
				}
				_containerFiles.add(new ArrayList<>(tmpBlock));
				tmpBlock.clear();
			}
		}

		private boolean overlapIndex(final IndexedContainer indexedContainer, final long min, final long max) {
			return indexedContainer.getDataIndexMinimum() <= max && indexedContainer.getDataIndexMaximum() >= min;
		}

		private boolean overlapDataTime(final SyncKey<IdDataIdentification> lock, final IndexedContainer indexedContainer, final long min, final long max) {
			return indexedContainer.getDataTimeMinimum(lock) <= max && indexedContainer.getDataTimeMaximum(lock) >= min;
		}


		@Override
		public DataIterator iterator() throws PersistenceException {
			return new BucketContainerIterator(_containerFiles, directory.archiveDataKind(), _order,
					_idDataIdentification, _persMgr);
		}
	}

	/**
	 * Diese Klasse speichert eine Referenz auf einen Container, aus dem möglicherweise Archivdaten gelesen werden sollen.
	 * <p>
	 * Dabei werden soviele Informationen wie möglich bereits hier zusammen mti der Container-ID abgelegt, damit diese später nicht erneut ausgelesen
	 * werden müssen.
	 */
	public static class IndexedContainer {
		private final ArchiveDataKind _adk;
		private final long _containerId;
		private final long _dataIndexMinimum;
		private final long _dataIndexMaximum;
		private long _dataTimeMinimum;
		private long _dataTimeMaximum;
		private final PersistenceDirectory _directory;

		/**
		 * Erstellt eine neue Instanz
		 *
		 * @param adk              Archivdatenart
		 * @param containerId      Container-ID
		 * @param dataIndexMinimum Datenindexminimum des Containers (aus Indexdatei)
		 * @param dataIndexMaximum Datenindexmaximum des Containers (aus Indexdatei)
		 * @param dataTimeMinimum  Datenzeitminimum des Containers (aus Indexdatei) oder -1 falls nicht bekannt (je nach benutztem Index)
		 * @param dataTimeMaximum  Datenzeitmaximum des Containers (aus Indexdatei) oder -1 falls nicht bekannt (je nach benutztem Index)
		 * @param directory        Verzeichnis des Containers
		 */
		public IndexedContainer(final ArchiveDataKind adk,
		                         final long containerId,
		                         final long dataIndexMinimum,
		                         final long dataIndexMaximum,
		                         final long dataTimeMinimum,
		                         final long dataTimeMaximum,
		                         final PersistenceDirectory directory) {
			_adk = adk;
			_containerId = containerId;
			_dataIndexMinimum = dataIndexMinimum;
			_dataIndexMaximum = dataIndexMaximum;
			_dataTimeMinimum = dataTimeMinimum;
			_dataTimeMaximum = dataTimeMaximum;
			_directory = directory;
		}

		/**
		 * Gibt die Container-ID zurück
		 *
		 * @return die Container-ID
		 */
		public long getContainerId() {
			return _containerId;
		}

		/**
		 * Gibt das Minimum des Datenindex zurück
		 *
		 * @return das Minimum des Datenindex
		 */
		public long getDataIndexMinimum() {
			return _dataIndexMinimum;
		}

		/**
		 * Gibt das Maximum des Datenindex zurück
		 *
		 * @return das Maximum des Datenindex
		 */
		public long getDataIndexMaximum() {
			return _dataIndexMaximum;
		}

		/**
		 * Gibt das Datenzeit-Minimum zurück. Diese Methode sieht im Verwaltungsdatenindex nach, falls der bisher benutzte Index diese Information nicht enthält.
		 *
		 * @return das Datenzeit-Minimum zurück
		 * @param lock Objekt, das den synchronisierten Zugriff auf den Index erlaubt
		 */
		public long getDataTimeMinimum(final SyncKey<IdDataIdentification> lock) {
			if(_dataIndexMinimum == -1) {
				try {
					_dataTimeMinimum = _directory.getContainerHeaders(new LockedContainerDirectory(lock, _adk), _containerId).getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_DATA_TIME_MIN);
				}
				catch(IndexException e) {
					// Sollte nicht auftreten, Indexe wurden eben angelegt
					throw new AssertionError(e);
				}
			}
			return _dataTimeMinimum;
		}


		/**
		 * Gibt das Datenzeit-Maximum zurück. Diese Methode sieht im Verwaltungsdatenindex nach, falls der bisher benutzte Index diese Information nicht enthält.
		 *
		 * @return das Datenzeit-Minimum zurück
		 * @param lock Objekt, das den synchronisierten Zugriff auf den Index erlaubt
		 */
		public long getDataTimeMaximum(final SyncKey<IdDataIdentification> lock) {
			if(_dataIndexMaximum == -1) {
				try {
					_dataTimeMaximum = _directory.getContainerHeaders(new LockedContainerDirectory(lock, _adk), _containerId).getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_DATA_TIME_MAX);
				}
				catch(IndexException e) {
					// Sollte nicht auftreten, Indexe wurden eben angelegt
					throw new AssertionError(e);
				}
			}
			return _dataTimeMaximum;
		}

		public PersistenceDirectory getDirectory() {
			return _directory;
		}
	}
}
