/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.layout;

import de.bsvrz.ars.ars.persistence.IdDataIdentification;

import java.io.IOException;
import java.nio.file.Path;

/**
 * Dieses Interface kann implementiert werden, um in einem {@link DirectoryLayoutWalk} über die gefundenen
 * Elemente eines einzelnen Persistenzverzeichnisses (Wochenverzeichnisses) informiert zu werden.
 *
 * @see PersistenceDirectoryLayout
 */
public interface DataIdentificationListener {

	/**
	 * Wird vom {@link DirectoryLayoutWalk} aufgerufen, wenn ein Ordner für eine Datenidentifikation gefunden wurde.
	 * Dieser Ordner enthält unterordner wie "oa", "nn" usw, die jeweils die Containerdateien enthalten.
	 *
	 * @param dataIdentification Datenidentifikation
	 * @param dir                Verzeichnis auf dem Dateisystem
	 * @throws AbortWalkException Diese Exception kann von der Implementierung geworfen werden, wenn der
	 *                            Durchlauf des Persistenzverzeichnisses komplett abgebrochen werden soll.
	 */
	void foundDataIdentification(IdDataIdentification dataIdentification, Path dir) throws AbortWalkException;

	/**
	 * Wird vom {@link DirectoryLayoutWalk} aufgerufen, wenn ein nicht identifizierbares ("fremdes") Verzeichnis
	 * in der Ordnerstruktur des Persistenzverzeichnisses gefunden wurde. So ein Ordner kann normalerweise
	 * gefahrlos gelöscht oder ignoriert werden.
	 *
	 * @param dir Verzeichnis auf dem Dateisystem
	 * @throws AbortWalkException Diese Exception kann von der Implementierung geworfen werden, wenn der
	 *                            Durchlauf des Persistenzverzeichnisses komplett abgebrochen werden soll.
	 */
	void foundForeignDirectory(Path dir) throws AbortWalkException;

	/**
	 * Wird vom {@link DirectoryLayoutWalk} aufgerufen, wenn eine nicht identifizierbares ("fremde") Datei
	 * in der Ordnerstruktur des Persistenzverzeichnisses gefunden wurde. So eine Datei kann normalerweise
	 * gefahrlos gelöscht oder ignoriert werden.
	 *
	 * @param file Dateipfad auf dem Dateisystem
	 * @throws AbortWalkException Diese Exception kann von der Implementierung geworfen werden, wenn der
	 *                            Durchlauf des Persistenzverzeichnisses komplett abgebrochen werden soll.
	 */
	void foundForeignFile(Path file) throws AbortWalkException;

	/**
	 * Wird vom {@link DirectoryLayoutWalk} aufgerufen, wenn ein unerwarteter Fehler beim Durchsuchen des
	 * Persistenzverzeichnisses aufgetreten ist. Dieser Fehler kann evtl. die Bearbeitung von
	 * Bereichen des Persistenzverzeichnisses (und damit bestimmter Datenidentifikationen) verhindern.
	 *
	 * @param file      Dateipfad auf dem Dateisystem
	 * @param exception Aufgetretender IO-Fehler
	 * @throws AbortWalkException Diese Exception kann von der Implementierung geworfen werden, wenn der
	 *                            Durchlauf des Persistenzverzeichnisses komplett abgebrochen werden soll.
	 */
	void ioException(Path file, IOException exception) throws AbortWalkException;
}
