/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.layout;

import de.bsvrz.ars.ars.persistence.PersistenceException;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKindCombination;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

/**
 * Hilfsklasse für die Verwaltung von Archivdatenarten im Persistenzverzeichnis
 */
public class DataKinds {

	private DataKinds() {
	}

	/**
	 * Gibt alle Archivdatenarten zurück, die auf Dateisystemebene in einem Datenidentifikationsverzeichnis
	 * vorliegen. Die Unterverzeichnisse haben Namen wie "oa" oder "nn".
	 *
	 * @param dataIdentificationDir Verzeichnis
	 * @return alle Archivdatenarten
	 * @throws PersistenceException IO-Fehler
	 */
	public static ArchiveDataKindCombination getAvailableDataKinds(Path dataIdentificationDir) throws PersistenceException {
		try (DirectoryStream<Path> paths = Files.newDirectoryStream(dataIdentificationDir)) {
			final List<ArchiveDataKind> dataKindList = new ArrayList<>();
			for (Path path : paths) {
				if (isValidDataKindSuffix(path.getFileName().toString())) {
					dataKindList.add(getArchiveDataKind(path));
				}
			}
			return new ArchiveDataKindCombination(dataKindList);
		} catch (NoSuchFileException e) {
			// Verzeichnis existiert nicht, also gibt es keine
			return new ArchiveDataKindCombination(List.of());
		} catch (IOException e) {
			throw new PersistenceException(e);
		}
	}


	/**
	 * Gibt {@code true} zurück, wenn es sich um einen gültigen Datenart-Ordnernamen handelt
	 *
	 * @param dirName Ordnername
	 * @return {@code true}, wenn es sich um einen gültigen Datenart-Ordnernamen handelt, sonst {@code false}
	 */
	public static boolean isValidDataKindSuffix(String dirName) {
		return switch (dirName) {
			case "oa", "on", "na", "nn" -> true;
			default -> false;
		};
	}

	/**
	 * Extrahiert aus einem Pfadnamen die Archovdatenart, bei einem Pfad mit dem Namen "oa", wird z. B. {@link ArchiveDataKind#ONLINE}
	 * zurückgegeben.
	 *
	 * @param path Verzeichnis
	 * @return Archivdatenart
	 * @throws IllegalArgumentException Wenn das Verzeichnis keinen zuordbaren Namen hat.
	 */
	public static ArchiveDataKind getArchiveDataKind(Path path) {
		ArchiveDataKind kind = getArchiveDataKindOptional(path);
		if (kind == null) {
			throw new IllegalArgumentException("Ungültiges Verzeichnis: " + path);
		}
		return kind;
	}

	/**
	 * Wie {@link #getArchiveDataKind(Path)}, gibt bei Fehler aber {@code null} zurück, statt eine Exception zu werfen.
	 *
	 * @param path Verzeichnis
	 * @return Archivdatenart oder null
	 */
	public static ArchiveDataKind getArchiveDataKindOptional(Path path) {
		String dirName = path.getFileName().toString();

		return switch (dirName) {
			case "oa" -> ArchiveDataKind.ONLINE;
			case "on" -> ArchiveDataKind.ONLINE_DELAYED;
			case "na" -> ArchiveDataKind.REQUESTED;
			case "nn" -> ArchiveDataKind.REQUESTED_DELAYED;
			default -> null;
		};
	}


	/**
	 * Gibt zu der angegebenen Datenart den Pfadnamen zurück, der verwendet wird.
	 *
	 * @param adk Archivdatenart (siehe {@link Util#getDataKindFromIndex(int)})
	 * @return Suffix
	 */
	public static String getDataKindSuffix(final ArchiveDataKind adk) {
		return getDataKindSuffix(Util.getDataKindIndex(adk));
	}

	/**
	 * Gibt zu der angegebenen Datenart den Pfadnamen zurück, der verwendet wird.
	 *
	 * @param adkIdx Archivdatenart (siehe {@link Util#getDataKindFromIndex(int)})
	 * @return Suffix
	 */
	public static String getDataKindSuffix(final int adkIdx) {
		return switch (adkIdx) {
			case 0 -> "oa";
			case 1 -> "on";
			case 2 -> "na";
			case 3 -> "nn";
			default -> throw new IllegalArgumentException("adkIdx=" + adkIdx);
		};
	}

}
