/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.layout;

import de.bsvrz.ars.ars.persistence.PersistenceManager;
import de.bsvrz.ars.ars.persistence.StartupProperties;
import de.bsvrz.ars.ars.persistence.directories.mgmt.lock.LockFileManager;

import java.io.IOException;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Set;

/**
 * Iterator für ein bestimmtes Persistenzverzeichnis-Layout.
 */
public class DirectoryLayoutWalk {

	private DirectoryLayoutWalk() {
	}

	/**
	 * Führt einen rekursiven Verzeichnisdurchlauf durch. Der übergebene Listener bekommt alle gefundenen Verzeichnisse
	 * je nach Art des Verzeichnisses übermittelt.
	 *
	 * @param layoutInstance Layout des Persistenzverzeichnisses
	 * @param listener       Hier werden alle gefundenen Datenidentifikationsverzeichnisse und sonstigen Probleme
	 *                       durch entsprechende Methodenaufrufe ausgegeben
	 * @see DataIdentificationListener
	 */
	public static void walk(PersistenceDirectoryLayoutInstance layoutInstance, DataIdentificationListener listener) {
		final ArrayList<String> pathStack;
		Path basePath = layoutInstance.basePath();
		PersistenceDirectoryLayout layout = layoutInstance.layout();
		pathStack = new ArrayList<>();
		try {
			FileVisitor<Path> visitor = new FileVisitor<>() {

				@Override
				public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
					try {
						if (dir.equals(basePath)) {
							return FileVisitResult.CONTINUE;
						}
						pathStack.add(dir.getFileName().toString());
						PathType type = layoutInstance.getPathType(pathStack);
						switch (type) {
							case PartialIdPath -> {
								return FileVisitResult.CONTINUE;
							}
							case CompleteIdPath -> {
								listener.foundDataIdentification(layoutInstance.getPath(pathStack), dir);
								removeLast();
								return FileVisitResult.SKIP_SUBTREE;
							}
							case ForeignPath -> {
								if (!dir.equals(basePath.resolve(PersistenceManager.BACKUP_DIR_NAME))) {
									listener.foundForeignDirectory(dir);
								}
								removeLast();
								return FileVisitResult.SKIP_SUBTREE;
							}
						}
						throw new AssertionError();
					} catch (AbortWalkException e) {
						return FileVisitResult.TERMINATE;
					}
				}

				@Override
				public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
					try {
						if (file.equals(basePath.resolve(LockFileManager.ACCESS_FLAG_FILE_NAME))) {
							return FileVisitResult.CONTINUE;
						}
						if (file.equals(basePath.resolve(LockFileManager.NOT_PROTECTED_FLAG_FILE_NAME))) {
							return FileVisitResult.CONTINUE;
						}
						if (file.equals(basePath.resolve(StartupProperties.STARTUP_PROP_FILE_NAME))) {
							return FileVisitResult.CONTINUE;
						}
						listener.foundForeignFile(file);
						return FileVisitResult.CONTINUE;
					} catch (AbortWalkException e) {
						return FileVisitResult.TERMINATE;
					}
				}

				@Override
				public FileVisitResult visitFileFailed(Path file, IOException exc) {
					try {
						listener.ioException(file, exc);
						return FileVisitResult.CONTINUE;
					} catch (AbortWalkException e) {
						return FileVisitResult.TERMINATE;
					}
				}

				@Override
				public FileVisitResult postVisitDirectory(Path dir, IOException exc) {
					removeLast();
					return FileVisitResult.CONTINUE;
				}

				private void removeLast() {
					if (!pathStack.isEmpty()) {
						pathStack.remove(pathStack.size() - 1);
					}
				}
			};

			int maxDepth = layout.getMaxDepth() == Integer.MAX_VALUE ? Integer.MAX_VALUE : layout.getMaxDepth() + 1;
			Files.walkFileTree(basePath, Set.of(FileVisitOption.FOLLOW_LINKS), maxDepth, visitor);

		} catch (IOException e) {
			try {
				listener.ioException(basePath, e);
			} catch (AbortWalkException ignored) {
			}
		}
	}

}
