/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.layout;

import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.persistence.ContainerDirectory;
import de.bsvrz.ars.ars.persistence.IdContainerFileDir;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.ars.ars.persistence.PersistenceException;
import de.bsvrz.ars.ars.persistence.directories.mgmt.range.WeekDomain;
import de.bsvrz.ars.ars.persistence.index.IndexException;
import de.bsvrz.ars.ars.persistence.index.backend.management.BaseIndex;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import org.jetbrains.annotations.Contract;

import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

/**
 * Konkrete instanz eines {@link PersistenceDirectoryLayout}, welches sich an einer bestimmten Position im Dateisystem befindet.
 *
 * @param layout            Verzeichnis-Layout
 * @param basePath          Dateisystem-Pfad, unter dem die Archivdaten liegen. Dies ist z. B. das konkrete Wochenverzeichnis.
 * @param simulationVariant Simulationsvariante der Daten in diesem Verzeichnis
 */
public record PersistenceDirectoryLayoutInstance(PersistenceDirectoryLayout layout, Path basePath,
                                                 int simulationVariant) {

	/**
	 * Ermittelt von einer Index-Datei im Dateisystem die Datenidentifikation
	 * @param index Index
	 * @return Datenidentifikation und Datenart
	 * @throws IndexException Fehler bei Abfrage
	 */
	public IdContainerFileDir getContainerDirectory(BaseIndex<?> index) throws IndexException {
		try {
			return getContainerDirectory(index.getFile().getParent());
		} catch (PersistenceException e) {
			throw new IndexException("Kann Datenidentifikation von Index nicht bestimmen", index.getFile(), e);
		}
	}

	/**
	 * Ermittelt von einem Pfad im Dateisystem die Datenidentifikation und Datenart
	 * @param containerDirectory Containerdateien-Verzeichnis
	 * @return Datenidentifikation und Datenart
	 * @throws PersistenceException Fehler bei Abfrage
	 */
	public IdContainerFileDir getContainerDirectory(Path containerDirectory) throws PersistenceException {
		ArchiveDataKind dataKind = DataKinds.getArchiveDataKindOptional(containerDirectory);
		if (dataKind == null) {
			throw new PersistenceException("Ungültiges Verzeichnis: " + containerDirectory);
		}
		IdDataIdentification did = layout.getDataIdentificationFromPath(toStack(basePath.relativize(containerDirectory.getParent())), simulationVariant);
		return new IdContainerFileDir(did, dataKind);
	}

	/**
	 * Wandelt ein path-Objekt um, in eine Liste von Pfad-Bestandteilen
	 * @param path Pfad
	 * @return Liste
	 */
	private static List<String> toStack(Path path) {
		ArrayList<String> list = new ArrayList<>();
		for (Path element : path) {
			Path fileName = element.getFileName();
			if (fileName != null) {
				list.add(fileName.toString());
			}
		}
		return list;
	}

	/**
	 * Gibt zu einem ContainerDirectory das zugehörige Verzeichnis zurück.
	 *
	 * @param containerDirectory Datenidentifikation und Archivdatenart, die das Containerverzeichnis identifizieren
	 * @return Verzeichnis
	 */
	@NotNull
	@Contract(pure = true)
	public Path getPath(ContainerDirectory containerDirectory) {
		Path didPath = getPath(containerDirectory.dataIdentification());
		String suffix = DataKinds.getDataKindSuffix(containerDirectory.archiveDataKind());
		return didPath.resolve(suffix);
	}

	/**
	 * Gibt zu einer Datenidentifikation das Dateisystemverzeichnis zurück
	 * @param dataIdentification Datenidentifikation
	 * @return Verzeichnis
	 */
	public Path getPath(IdDataIdentification dataIdentification) {
		return layout.getPath(basePath, dataIdentification);
	}

	/**
	 * Identifiziert den Typ des angegebenen Verzeichnisses
	 * @param pathStack Stack wo man sich im Verzeichnisbaum befindet, siehe {@link #toStack(Path)}
	 * @return Typ des angegebenen Pfads   
	 */
	public PathType getPathType(ArrayList<String> pathStack) {
		return layout.identifyPath(pathStack);
	}

	/**
	 * Identifiziert die Datenidentifiaktion des angegebenen Verzeichnisses
	 *
	 * @param pathStack Stack wo man sich im Verzeichnisbaum befindet, siehe {@link #toStack(Path)}
	 * @return Typ des angegebenen Pfads
	 */
	public IdDataIdentification getPath(ArrayList<String> pathStack) {
		return layout.getDataIdentificationFromPath(pathStack, simulationVariant);
	}

	/**
	 * Versucht, von einem Verzeichnis zu ermitteln, welches persistenzverzeichnislayout in diesem
	 * Verzeichnis genutzt wird und gibt dies als {@link PersistenceDirectoryLayoutInstance} zurück.
	 * @param path Verzeichnis
	 * @return Instanz (nie null)
	 * @throws PersistenceException wenn die Instanz nicht ermittelt werden konnte
	 */
	@NotNull
	public static PersistenceDirectoryLayoutInstance detectFromPath(Path path) throws PersistenceException {
		path = path.toAbsolutePath();
		Path tmp = path.getRoot();
		for (Path element : path) {
			String pathName = element.getFileName().toString();
			if (ClassicPersistenceDirectoryLayout.OBJ_DIRNAME_PAT.matcher(pathName).matches()) {
				return ClassicPersistenceDirectoryLayout.instance(tmp, 0);
			}
			if (ShortPersistenceDirectoryLayout.decodeFast(pathName) != -1) {
				return ShortPersistenceDirectoryLayout.instance(tmp, 0);
			}
			if (tmp == null) {
				tmp = element;
			} else {
				tmp = tmp.resolve(element);
			}
		}
		try {
			WeekDomain domain = ArchiveManager.detectDomain(path, true);
			if (domain == null) {
				return ClassicPersistenceDirectoryLayout.instance(path, 0);
			}
			return ShortPersistenceDirectoryLayout.instance(path, 0);
		} catch (IOException e) {
			throw new PersistenceException("Kann Wurzelverzeichnis von '" + path + "' nicht bestimmen", e);
		}
	}
}
