/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.layout;

import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.ars.ars.persistence.PersistenceException;

import java.nio.file.Path;
import java.util.List;

/**
 * Modernes Layout für "kurze" Persistenzverzeichnisse. Die Dateien werden in Verzeichnissen wie
 * "2023/2023-01-08/0-4bpz/0-jh/2057-1b4/oa" relativ zum Persistenzverzeichnis gespeichert, wobei
 * das "oa" nicht zu der Implementierung dieses {@link PersistenceDirectoryLayout} gehört
 */
public class ShortPersistenceDirectoryLayout implements PersistenceDirectoryLayout {

	/**
	 * Singleton-Instanz
	 */
	public static final ShortPersistenceDirectoryLayout Instance = new ShortPersistenceDirectoryLayout();

	private ShortPersistenceDirectoryLayout() {
	}

	@Override
	public Path getPath(Path basePath, long objId, long atgId, long aspId) {
		return basePath.resolve(encode(atgId)).resolve(encode(aspId)).resolve(encode(objId));
	}

	@Override
	public IdDataIdentification getDataIdentificationFromPath(List<String> path, int simulationVariant) {
		String objComponent = path.get(2);
		String aspComponent = path.get(1);
		String atgComponent = path.get(0);
		try {
			return new IdDataIdentification(decode(objComponent), decode(atgComponent), decode(aspComponent), simulationVariant);
		} catch (PersistenceException e) {
			return null;
		}
	}

	/**
	 * Wandelt eine Objekt-ID in einen Verzeichnisnamen um. Der Verzeichnisname ergibt sich aus der KV-Kodierung der ID
	 * (die höchsten 16 Bits), dann einem Trennstrich ("-"), und dann die laufende Nummer der ID kodiert als Base 36.
	 *
	 * @param id Objekt-ID. Der Wert -1 ist nicht zulässig.
	 * @return Verzeichnisname.
	 */
	public static String encode(long id) {
		if (id == -1) {
			throw new UnsupportedOperationException("Ungültige Objekt-ID: -1");
		}
		int coding = (int) (id >>> 48);
		long runningNumber = id & 0xFFFFFFFFFFFFL;
		return coding + "-" + Long.toString(runningNumber, Character.MAX_RADIX);
	}

	/**
	 * Dekodiert einen Verzeichnisnamen in eine Objekt-ID
	 * @param string Verzeichnisname wie z. B. "2057-1b4"
	 * @return Objekt-ID
	 * @throws PersistenceException Der Pfad ist nicht als Objekt-ID interpretierbar
	 */
	public static long decode(String string) throws PersistenceException {
		long result = decodeFast(string);
		if (result == -1) {
			throw new PersistenceException("Ungültiger Pfad-Bestandteil: " + string);
		}
		return result;
	}

	/**
	 * Dekodiert einen Verzeichnisnamen in eine Objekt-ID
	 * @param string Verzeichnisname wie z. B. "2057-1b4"
	 * @return Objekt-ID oder -1, falls das Verzeichnis nicht verarbeitet werden konnte
	 */
	public static long decodeFast(String string) {
		try {
			int divider = string.indexOf('-');
			if (divider == -1) {
				return -1;
			}
			String kvPrefix = string.substring(0, divider);
			int kv = Integer.parseInt(kvPrefix);
			if (kv != (kv & 0xFFFF)) {
				// Bestandteile zu groß
				return -1;
			}

			String runningNumber = string.substring(divider + 1);
			long number = Long.parseLong(runningNumber, Character.MAX_RADIX);
			if (number != (number & 0xFFFFFFFFFFFFL)) {
				// Bestandteile zu groß
				return -1;
			}

			return (long) kv << 48 | number;
		} catch (NumberFormatException e) {
			return -1;
		}
	}

	@Override
	public int getMaxDepth() {
		return 3;
	}

	@Override
	public PathType identifyPath(List<String> pathStack) {
		if (pathStack.size() > 3) {
			return PathType.ForeignPath;
		}
		for (String s : pathStack) {
			if (decodeFast(s) == -1) {
				return PathType.ForeignPath;
			}
		}
		return pathStack.size() == 3 ? PathType.CompleteIdPath : PathType.PartialIdPath;
	}

	/**
	 * Erzeugt eine {@link PersistenceDirectoryLayoutInstance} basierend auf diesem Layout
	 *
	 * @param basePath          Basisverzeichnis, in dem das Layout Unterverzeichnisse erstellen soll.
	 * @param simulationVariant Simulationsvariante
	 * @return Eine PersistenceDirectoryLayoutInstance
	 */
	public static PersistenceDirectoryLayoutInstance instance(Path basePath, int simulationVariant) {
		return Instance.createInstance(basePath, simulationVariant);
	}
}
