/*
 * Copyright 2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.util;

import static java.nio.file.FileVisitResult.CONTINUE;

import java.io.IOException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;

/**
 * Hilfsklasse um einen Ordner auf einen Zielordner zu kopieren (zu synchronisieren) ohne alles erneut zu übertragen.
 *
 * @author Kappich Systemberatung
 */
public class FileSync {

	/**
	 * Synchronisiert zwei Verzeichnisse.
	 * @param from Quellverzeichnis
	 * @param to   Zielverzeichnis
	 * @throws IOException IO-Problem
	 * @return Statistik über kopierte Daten
	 */
	public SyncResult syncDirectories(Path from, Path to) throws IOException {
		SyncVisitor syncVisitor = new SyncVisitor(from, to);
		RemoveVisitor removeVisitor = new RemoveVisitor(from, to);
		Files.walkFileTree(from, syncVisitor);
		Files.walkFileTree(to, removeVisitor);
		return new SyncResult(syncVisitor.getFilesCopied(), removeVisitor.getFilesRemoved(), syncVisitor.getBytesCopied());
	}

	protected boolean shouldIgnore(final String fileName) {
		return false;
	}

	/**
	 * Prüft, ob targetFile die (wahrscheinlich) die gleiche Datei wie sourceFile ist, also nicht erneut übertragen werden muss.
	 * Hierzu wird die Dateigröße und der letzte Änderungszeitpunkt verglichen.
	 *
	 * @param sourceFile Quelldatei
	 * @param targetFile Zieldatei
	 * @return true: Dateien sind gleich, false: dateien sind ungleich oder Zieldatei fehlt
	 * @throws IOException Fehler beim Vergleich
	 */
	@SuppressWarnings("BooleanMethodIsAlwaysInverted")
	public static boolean isIdentical(final Path sourceFile, final Path targetFile) throws IOException {
		if(!Files.exists(targetFile)) {
			return false;
		}
		try {
			if(Files.isDirectory(targetFile)) return true;
			if(Files.size(sourceFile) != Files.size(targetFile)) return false;

			// Die Abweichung von einer Sekunde wird wegen eventueller Rundung toleriert.
			// Ein Dateisystem könnte 12:34:56.999 als 12:34:57 speichern,
			// aber wir würden dann 12:34:56 mit 12:34:57 vergleichen. 
			long sourceTime = getTruncatedModTime(sourceFile);
			long targetTime = getTruncatedModTime(targetFile);
			return Math.abs(sourceTime - targetTime) <= 1;
		}
		catch(NoSuchFileException e) {
			return false;
		}
	}

	/**
	 * Gibt den Zeitstempel der letzten Änderung zurück. Da Dateisysteme unterschiedliche Präzision haben, wird der
	 * Zeitstempel auf ganze Sekunden gerundet.
	 *
	 * @param file Datei
	 * @return Zeitstempel in Epoch-Sekunden
	 * @throws IOException Fehler beim Lesen
	 */
	private static long getTruncatedModTime(Path file) throws IOException {
		return Files.getLastModifiedTime(file).toMillis() / 1000;
	}

	private class SyncVisitor extends SimpleFileVisitor<Path> {

		private final Path _source;
		private final Path _target;
		private long _filesCopied;
		private long _bytesCopied;

		public SyncVisitor(final Path source, final Path target) {
			_source = source;
			_target = target;
		}

		@Override
		public FileVisitResult preVisitDirectory(final Path dir, final BasicFileAttributes attrs) throws IOException {
			Path targetDir = _target.resolve(_source.relativize(dir));
			try {
				if(!Files.isDirectory(targetDir)) {
					Path parent = targetDir.getParent();
					if(!Files.isDirectory(parent)) {
						Files.createDirectories(parent);
					}
					Files.copy(dir, targetDir, StandardCopyOption.COPY_ATTRIBUTES);
					_filesCopied++;
				}
			} catch (FileAlreadyExistsException e) {
				if (!Files.isDirectory(targetDir))
					throw e;
			}
			return CONTINUE;		
		}

		@Override
		public FileVisitResult visitFile(final Path file, final BasicFileAttributes attrs) throws IOException {
			Path targetFile = _target.resolve(_source.relativize(file));
			if(shouldIgnore(file.getFileName().toString())) {
				return CONTINUE;
			}
			if(!isIdentical(file, targetFile)) {
				Files.copy(file, targetFile, StandardCopyOption.COPY_ATTRIBUTES, StandardCopyOption.REPLACE_EXISTING);
				if(!isIdentical(file, targetFile)) {
					Files.setLastModifiedTime(targetFile, Files.getLastModifiedTime(file));
				}
				_filesCopied++;
				_bytesCopied += Files.size(targetFile);
			}
			return CONTINUE;
		}

		@Override
		public FileVisitResult visitFileFailed(final Path file, final IOException exc) {
			return CONTINUE;
		}

		public long getFilesCopied() {
			return _filesCopied;
		}

		public long getBytesCopied() {
			return _bytesCopied;
		}
	}
	
	private class RemoveVisitor extends SimpleFileVisitor<Path> {

		private final Path _source;
		private final Path _target;
		private long _filesRemoved;

		public RemoveVisitor(final Path source, final Path target) {
			_source = source;
			_target = target;
		}

		@Override
		public FileVisitResult postVisitDirectory(final Path dir, final IOException exc) throws IOException {
			super.postVisitDirectory(dir, exc);
			Path sourceDirectory = _source.resolve(_target.relativize(dir));
			if(!Files.exists(sourceDirectory)) {
				Files.delete(dir);
				_filesRemoved++;
			}
			return CONTINUE;
		}

		@Override
		public FileVisitResult visitFile(final Path file, final BasicFileAttributes attrs) throws IOException {
			Path sourceFile = _source.resolve(_target.relativize(file));
			if(!shouldIgnore(file.getFileName().toString()) && !Files.exists(sourceFile)) {
				Files.delete(file);
				_filesRemoved++;
			}
			return CONTINUE;
		}


		@Override
		public FileVisitResult visitFileFailed(final Path file, final IOException exc) {
			return CONTINUE;
		}

		public long getFilesRemoved() {
			return _filesRemoved;
		}
	}
}
