/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.walk;

import de.bsvrz.ars.ars.persistence.PersistenceException;
import de.bsvrz.ars.ars.persistence.PersistenceManager;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.walk.internal.StandardContainerWalk;
import de.bsvrz.ars.ars.persistence.walk.internal.StatusPrinter;
import de.bsvrz.dav.daf.main.archive.ArchiveDataSpecification;

import java.time.Duration;
import java.util.Collection;

/**
 * Interface für die Durchführung für Aktionen, die Containerdateien besuchen.
 * <p>
 * Die Arbeit kann dabei auf mehrere Threads verteilt werden.
 *
 * @author Kappich Systemberatung
 */
public interface ContainerWalk {

	/**
	 * Factory-Methode, die einen ContainerDirWalk erzeugt, der alle Containerverzeichnisse durchsucht.
	 * @return Ein Objekt, das die Aktion starten kann
	 * @see #execute(String, int, ContainerAction)
	 * @param persistenceDirectory Persistenzverzeichnis
	 * @param persistenceManager   Persistenzschicht
	 */
	static ContainerWalk allContainers(final PersistenceDirectory persistenceDirectory, final PersistenceManager persistenceManager) {
		return new StandardContainerWalk(persistenceManager, DataIdentificationDirWalk.allDirectories(persistenceDirectory), persistenceDirectory);
	}

	/**
	 * Factory-Methode, die einen ContainerDirWalk erzeugt, der die in der angegebenen Anfrage enthaltenen Archivdaten-Spezifikationen durchsucht.
	 *
	 * @param persistenceDirectory      Persistenzverzeichnis
	 * @param archiveDataSpecifications Anfrage
	 * @param persistenceManager        Persistenzschicht
	 * @return Ein Objekt, das die Aktion starten kann
	 * @see #execute(String, int, ContainerAction)
	 */
	static ContainerWalk fromArchiveDataSpecification(final PersistenceDirectory persistenceDirectory,
	                                                  Collection<? extends ArchiveDataSpecification> archiveDataSpecifications,
	                                                  final PersistenceManager persistenceManager) {
		return new StandardContainerWalk(persistenceManager, DataIdentificationDirWalk.create(persistenceDirectory, archiveDataSpecifications), persistenceDirectory);
	}

	/**
	 * Gibt das Intervall zurück, in dem Statusmeldungen ausgegeben werden (positiv)
	 * @return das Intervall
	 */
	Duration getStatusInterval();

	/**
	 * Setzt das Statusintervall
	 * @param statusInterval Intervall
	 */
	void setStatusInterval(Duration statusInterval);

	/**
	 * Gibt die Klasse zurück, die Statusmeldungen ausgibt.
	 * @return Klasse
	 */
	StatusPrinter getStatusPrinter();

	/**
	 * Setzt die Klasse, die Statusmeldungen schreibt.
	 * @param statusPrinter Beliebige Subklasse/Instanz von {@link StatusPrinter}
	 */
	void setStatusPrinter(StatusPrinter statusPrinter);

	/**
	 * Führt den Verzeichnisdurchlauf aus. Diese Methode kann je Objekt nur einmal ausgeführt werden.
	 *
	 * @param actionName Name (zur Benennung der Thread-Objekte, Ausgabe in Debug-Meldungen usw.)
	 * @param numThreads Anzahl Threads zur Bearbeitung
	 * @param action Durchzuführende Aktion
	 * @throws PersistenceException Falls ein Fehler Auftritt
	 */
	void execute(String actionName, int numThreads, ContainerAction action) throws PersistenceException;

	/**
	 * Sorgt für ein sofortiges, aber kontrolliertes Beenden. Der aktuelle Verzeichnisdurchlauf je Thread wird abgeschlossen.
	 */
	void terminate();
}
