/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.walk.internal;

import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SyncKey;
import de.bsvrz.ars.ars.persistence.*;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.index.IndexValues;
import de.bsvrz.ars.ars.persistence.index.result.IndexResult;
import de.bsvrz.ars.ars.persistence.sequence.SequenceSpecification;
import de.bsvrz.ars.ars.persistence.walk.ContainerAction;
import de.bsvrz.ars.ars.persistence.walk.ContainerWalk;
import de.bsvrz.ars.ars.persistence.walk.DataIdentificationDirWalk;
import de.bsvrz.dav.daf.main.archive.ArchiveDataSpecification;
import de.bsvrz.dav.daf.main.archive.ArchiveTimeSpecification;

import java.time.Duration;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

/**
 * Klasse, die über alle angegebenen Container iteriert. Die abzuarbeitenden Containerverzeichnisse werden von einem {@link DataIdentificationDirWalk}
 * geliefert, die Reihenfolge in denen die Containerverzeichnisse betrachtet werden ist daher im Allgemeinen undefiniert.
 * <p>
 * Die Container in einem Verzeichnis werden zusammen betrachtet, aber ebenfalls in nicht notwendigerweise sortierter Reihenfolge.
 *
 * @author Kappich Systemberatung
 */
public class StandardContainerWalk implements ContainerWalk {
	private final DataIdentificationDirWalk _dirWalk;
	private final PersistenceManager _persistenceManager;
	private final PersistenceDirectory _persistenceDirectory;

	/**
	 * Erstellt eine Instanz
	 *
	 * @param persistenceManager   Persistenzschicht
	 * @param dirWalk              Klasse, die über Containerverzeichnisse iteriert.
	 * @param persistenceDirectory Persistenzverzeichnis
	 */
	public StandardContainerWalk(PersistenceManager persistenceManager, final DataIdentificationDirWalk dirWalk, final PersistenceDirectory persistenceDirectory) {
		_persistenceManager = persistenceManager;
		_dirWalk = dirWalk;
		_persistenceDirectory = persistenceDirectory;
	}

	@Override
	public Duration getStatusInterval() {
		return _dirWalk.getStatusInterval();
	}

	@Override
	public void setStatusInterval(final Duration statusInterval) {
		_dirWalk.setStatusInterval(statusInterval);
	}

	@Override
	public StatusPrinter getStatusPrinter() {
		return _dirWalk.getStatusPrinter();
	}

	@Override
	public void setStatusPrinter(final StatusPrinter statusPrinter) {
		_dirWalk.setStatusPrinter(statusPrinter);
	}

	@Override
	public void execute(final String actionName, final int numThreads, final ContainerAction action) throws PersistenceException {
		_dirWalk.execute(actionName, numThreads, (dataIdentificationDir, standardContainerDirWalk) -> {
			Collection<ArchiveDataSpecification> ads = dataIdentificationDir.getArchiveDataSpecifications();
			action.preVisitDirectory(dataIdentificationDir);
			try (SyncKey<IdDataIdentification> lock = _persistenceManager.lockIndex(dataIdentificationDir.getDataIdentification())) {
				for (LockedContainerDirectory containerDirectory : dataIdentificationDir.getContainerDirectories(lock)) {
					Collection<ContainerHeaders> hdrList;
					if(ads == null) {
						IndexResult<ContainerManagementInformation> headers = _persistenceDirectory.getContainerHeaders(containerDirectory);
						_dirWalk.increaseContCount(headers.size());
						hdrList = IntStream.range(0, headers.size())
								.mapToObj(it -> new ContainerHeaders(headers, it, containerDirectory))
								.collect(Collectors.toList());
					}
					else {
						hdrList = new LinkedHashSet<>();
						for(ArchiveDataSpecification dataSpecification : ads) {
							ArchiveTimeSpecification timeSpec = dataSpecification.getTimeSpec();
							SequenceSpecification sequence = _persistenceManager.createSequenceFromArchiveTimeSpecification(timeSpec, containerDirectory);
							IndexResult<IndexValues> indexResult = _persistenceDirectory.getIndexResult(containerDirectory, sequence);
							_dirWalk.increaseContCount(indexResult.size());
							for(int j = 0; j < indexResult.size(); j++) {
								long containerId = indexResult.get(j, IndexValues.ContainerId);
								hdrList.add(_persistenceDirectory.getContainerHeaders(containerDirectory, containerId));
							}
						}
	
					}
					action.run(dataIdentificationDir, containerDirectory, hdrList);
				}
			}
		});
	}

	@Override
	public void terminate() {
		_dirWalk.terminate();
	}
}
