/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.walk.internal;

import de.bsvrz.ars.ars.mgmt.tasks.base.TaskProgressInterface;
import de.bsvrz.ars.ars.persistence.PersistenceException;
import de.bsvrz.ars.ars.persistence.layout.AbortWalkException;
import de.bsvrz.ars.ars.persistence.walk.DataIdentificationDirAction;
import de.bsvrz.ars.ars.persistence.walk.DataIdentificationDirWalk;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import java.time.Duration;
import java.time.Instant;
import java.util.Comparator;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

/**
 * Standardimplementierung eines DataIdentificationDirWalk
 *
 * @author Kappich Systemberatung
 */
public class StandardDataidentificationDirWalk implements DataIdentificationDirWalk {

	private static final Debug _debug = Debug.getLogger();

	/**
	 * Klasse, die Containerdatei_Verzeichnisse auflistet
	 */
	private final DataIdentificationDirGetter _contGetter;

	/**
	 * Exception, die während des Durchlaufsaufgetreten ist (oder null)
	 */
	private Exception _exception;

	/**
	 * Thread-Pool
	 */
	private ThreadPoolExecutor _executor;

	/**
	 * Anzahl besuchter Container (nur wenn innerhalb eines {@link de.bsvrz.ars.ars.persistence.walk.ContainerWalk} benutzt, sonst 0)
	 */
	private final AtomicLong _visitedContainerFiles = new AtomicLong();

	/**
	 * Anzahl besuchter Containerverz. (threadsave)
	 */
	private final AtomicLong _visitedContainerDirs = new AtomicLong();

	/**
	 * Anzahl gefundener Containerverz. (threadsave)
	 */
	private final AtomicLong _indexedContainerDirs = new AtomicLong();

	/**
	 * Intervall für periodische Debug-Ausgaben
	 */
	private Duration _statusInterval = Duration.ofMinutes(1);

	/**
	 * Letzter Ausgabezeitpunkt
	 */
	private Instant _lastStatus;

	/**
	 * Klasse, die Ausgaben formatiert
	 */
	private StatusPrinter _statusPrinter = new StatusPrinter();

	/**
	 * Startzeitpunkt
	 */
	private Instant _start;

	/**
	 * Soll die Ausführung sofort beendet werden (Aufträge in Warteschlange werden nicht mehr durchgeführt)
	 */
	private volatile boolean _terminateImmediately;

	private TaskProgressInterface _publisherTask;

	/**
	 * Erstellt einen neuen StandardContainerDirWalk
	 * @param contGetter Klasse, die Containerverzeichnisse auflistet.
	 */
	public StandardDataidentificationDirWalk(final DataIdentificationDirGetter contGetter) {
		_contGetter = contGetter;
	}

	@Override
	public Duration getStatusInterval() {
		return _statusInterval;
	}

	@Override
	public void setStatusInterval(final Duration statusInterval) {
		_statusInterval = statusInterval;
	}

	@Override
	public StatusPrinter getStatusPrinter() {
		return _statusPrinter;
	}

	@Override
	public void setStatusPrinter(final StatusPrinter statusPrinter) {
		_statusPrinter = statusPrinter;
	}

	@Override
	public TaskProgressInterface getPublisherTask() {
		return _publisherTask;
	}

	@Override
	public void setPublisherTask(final TaskProgressInterface publisherTask) {
		_publisherTask = publisherTask;
	}

	@Override
	public void execute(final String actionName, final int numThreads, final DataIdentificationDirAction action) throws PersistenceException {
		if(_executor != null) {
			throw new IllegalStateException("Bereits gestartet");
		}
		long id = Thread.currentThread().getId();

		// Queue für Threadpool. Wie verwenden eine PriorityBlockingQueue, die nach hashCode vergleicht, um
		// eine Queue ohne Kapazitätsgrenze zu erzeugen, die zufällig umsortiert, was eine bessere Vorhersehbarkeit der Ergebnisse
		// bei unterschiedlich großen Datenidentifikationsordnern ermöglicht (sonst werden ggf. alle großen Order
		// zusammen abgearbeitet).
		PriorityBlockingQueue<Runnable> queue = new PriorityBlockingQueue<>(512, Comparator.comparing(StandardDataidentificationDirWalk::getRandomHashCode));
		// threadpool für die eigentliche Ausführung
		_executor = new ThreadPoolExecutor(numThreads, numThreads, 0, TimeUnit.SECONDS, queue, new ThreadFactory() {
			private final AtomicInteger _counter = new AtomicInteger(0);

			@Override
			public Thread newThread(@NotNull final Runnable r) {
				return new Thread(r, "ContainerDirWalk-" + id + "-" + actionName + "-" + _counter.getAndIncrement());
			}
		});

		if (_statusPrinter != null) {
			_debug.info("Starte " + actionName);
		}
		_lastStatus = Instant.now();
		_start = _lastStatus;

		// Einzelner Thread, der die Containerverzeichnisse indiziert
		Thread indexThread = getIndexerThread(actionName, action, id);
		indexThread.start();
		try {
			indexThread.join();
			_executor.shutdown();
			_contGetter.setActualCount(_indexedContainerDirs.get());
			_debug.fine("ContainerDirWalk-" + id + "-" + actionName + ": Verzeichnisdurchlauf beendet.");
			while (!_executor.awaitTermination(1, TimeUnit.SECONDS)) {
				_debug.fine("ContainerDirWalk-" + id + "-" + actionName + ": Warte auf Beenden des Threadpools...");
				maybePrintStatus(actionName, true);
			}
			if (_statusPrinter != null) {
				if (_exception == null) {
					_debug.info(_statusPrinter.getSuccessMessage(actionName, Duration.between(_start, Instant.now()), _visitedContainerDirs.get(), _visitedContainerFiles.get()));
				} else {
					_debug.info(_statusPrinter.getErrorMessage(actionName, Duration.between(_start, Instant.now()), _visitedContainerDirs.get(), _visitedContainerFiles.get()));
				}
			}
		} catch (InterruptedException e) {
			throw new PersistenceException(e);
		}
		if (_exception != null) {
			throw new PersistenceException(_exception);
		}
	}

	/**
	 * Erzeugt den Thread, der die Datenidentifikationen indiziert und den {@link #_executor} mit {@link Runnable}-Objekten
	 * "füttert", die für jede Datenidentifikation die übergebene Aktion auslösen
	 *
	 * @param actionName Name der Aktion
	 * @param action     Eigentliche Aktion
	 * @param id         ID (Laufende Nummer) des Threads
	 * @return Thread-Objekt, das gestartet werden kann
	 */
	@NotNull
	private Thread getIndexerThread(String actionName, DataIdentificationDirAction action, long id) {
		final Runnable indexRunner = () -> _contGetter.run(dataIdentificationDir -> {
			if (_terminateImmediately) {
				throw new AbortWalkException();
			}

			maybePrintStatus(actionName, false);

			_indexedContainerDirs.incrementAndGet();

			// Dieser Code wird für jede Datenidentifikation ausgeführt
			Runnable didTask = () -> {
				if (_terminateImmediately) return;
				try {
					action.run(dataIdentificationDir, StandardDataidentificationDirWalk.this);
					_visitedContainerDirs.incrementAndGet();
				} catch (AbortWalkException e) {
					_terminateImmediately = true;
				} catch (Exception e) {
					_debug.warning("Fehler beim Ausführen der Aktion", e);
					setException(e);
					_executor.shutdown();
				}
			};
			try {
				_executor.execute(didTask);
			} catch (RejectedExecutionException e) {
				try {
					_executor.getQueue().put(didTask);
				} catch (InterruptedException ex) {
					_executor.shutdownNow();
					_exception = ex;
				}
			}
		});
		return new Thread(indexRunner, "ContainerDirWalk-" + id + "-" + actionName + "-index");
	}

	private static int getRandomHashCode(Runnable runnable) {
		int h = runnable.hashCode();
		return (h ^ h << 16 ^ h >> 8) & 0xFFFF;
	}

	private void maybePrintStatus(final String actionName, final boolean indexFinished) {
		Instant now = Instant.now();
		long upperBound = _indexedContainerDirs.get();
		StatusPrinter.ApproximationType approximationType;
		if (indexFinished) {
			approximationType = StatusPrinter.ApproximationType.Exact;
		} else {
			long estimate = _contGetter.estimateCount();
			if (estimate > upperBound) {
				approximationType = StatusPrinter.ApproximationType.Approximation;
				upperBound = estimate;
			} else {
				approximationType = StatusPrinter.ApproximationType.Minimum;
			}
		}

		if (_statusPrinter != null && _lastStatus.plus(_statusInterval).isBefore(now)) {
			_debug.info(_statusPrinter.getStatusMessage(actionName, Duration.between(_start, now), approximationType, upperBound, _visitedContainerDirs.get(), _visitedContainerFiles.get()));
		
			_lastStatus = now;
		}

		if (_publisherTask != null) {
			String estimateWord = switch (approximationType) {
				case Minimum -> "mindestens ";
				case Approximation -> "ungefähr ";
				case Exact -> "";
			};
			_publisherTask.setTaskProgress(actionName + ": %1$d von " + estimateWord + "%2$d Datenidentifikationen bearbeitet.", _visitedContainerDirs.get(), upperBound, -1);
		}
	}

	@Override
	public void terminate() {
		if(_executor != null) {
			_executor.shutdown();
		}
		_terminateImmediately = true;
	}

	/**
	 * Setzt die Exception, die beim Auftreten des Auftrags aufgetreten ist. Dadurch wird der Vorgang beendet.
	 * <p>
	 * Die Exception wird nachher im Haupt-Thread geworfen.
	 *
	 * @param exception Exception
	 */
	private void setException(final Exception exception) {
		_terminateImmediately = true;
		if(_exception == null) {
			_exception = exception;
		}
		else {
			_exception.addSuppressed(exception);
		}
	}

	@Override
	public void increaseContCount(final int increment) {
		_visitedContainerFiles.addAndGet(increment);
	}
}
