/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.writer;

import de.bsvrz.ars.ars.mgmt.tasks.ArchiveDataSerializer;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.losb.util.Util;

/**
 * Eine Implementierung von {@link SerializableDataset}, die asynchron von einem
 * {@link de.bsvrz.ars.ars.mgmt.tasks.SerializeTask} komplettiert wird
 */
public final class AsyncSerializableDataset implements SerializableDataset {

	private final long archiveTime;
	private final long dataTime;
	private final long dataIndex;
	private final byte dataStateCode;
	private final Data originalData;

	/**
	 * Serialisierte Daten. Werden asynchron gesetzt. Zugriff muss auf this synchronisiert werden. Die byte-Inhalte
	 * selbst sollten als unveränderlich gesehen werden.
	 */
	private byte[] serializedData;
	/**
	 * Unkomprimierte Datengröße. Der Wert ist nur gültig, wenn serializedData != null ist.
	 * Zugriff muss auf this synchronisiert werden.
	 */
	private int uncompressedSize;

	/**
	 * Erstellt einen neuen AsyncSerializableDataset.
	 *
	 * @param archiveTime   Archivzeit
	 * @param dataTime      Datenzeit
	 * @param dataIndex     Datenindex
	 * @param dataStateCode Code für den DataState (siehe {@link DataState#getCode()})
	 * @param originalData  Ursprünglicher Datenverteiler-Datensatz mit Nutzdaten
	 */
	public AsyncSerializableDataset(long archiveTime, long dataTime, long dataIndex, byte dataStateCode, Data originalData) {
		this.archiveTime = archiveTime;
		this.dataTime = dataTime;
		this.dataIndex = dataIndex;
		this.dataStateCode = dataStateCode;
		this.originalData = originalData;
	}

	/**
	 * Bestimmt den Archivzeitstempel des Datensatzes.
	 *
	 * @return Archivzeitstempel in Millisekunden seit 1970
	 */
	@Override
	public long archiveTime() {
		return archiveTime;
	}

	/**
	 * Bestimmt den Datenzeitstempel des Datensatzes.
	 *
	 * @return Datenzeitstempel in Millisekunden seit 1970
	 */
	@Override
	public long dataTime() {
		return dataTime;
	}

	/**
	 * Bestimmt den je Datenidentifikation eindeutigen vom Datenverteiler vergebenen Datensatzindex dieses Datensatzes.
	 *
	 * @return Datensatzindex
	 */
	@Override
	public long dataIndex() {
		return dataIndex;
	}

	/**
	 * Bestimmt den Datensatztyp des Datensatzes.
	 *
	 * @return Datensatztyp
	 */
	@NotNull
	@Override
	public DataState dataState() {
		return DataState.getInstance(dataStateCode);
	}

	/**
	 * Wird asynchron aufgerufen, um den serialisierten Datensatz zu übergeben
	 *
	 * @param data             Datensatz
	 * @param uncompressedSize Originale Datengröße falls die Daten komprimiert sind, sonst 0
	 */
	public void complete(byte[] data, int uncompressedSize) {
		synchronized (this) {
			this.uncompressedSize = uncompressedSize;
			this.serializedData = data;
			this.notifyAll();
		}
	}

	@NotNull
	@Override
	public byte[] serializedData() {
		ensureCompleted();
		return serializedData;
	}

	/**
	 * Wartet, bis der Datensatz asynchron serialisiert wurde, übernimmt notfalls selbst die Serialisierung.
	 */
	private void ensureCompleted() {
		synchronized (this) {
			if (serializedData != null) {
				return;
			}
			try {
				wait(1000);
			} catch (InterruptedException ignored) {
			}
			if (serializedData != null) {
				return;
			}
			new ArchiveDataSerializer().serializeData(this);
			assert serializedData != null;
		}
	}

	@Override
	public long estimateMemoryUsage() {
		return ArchiveJob.estimateDatasetUsage(this);
	}

	@Override
	public boolean hasData() {
		return dataStateCode == (byte) 1;
	}

	@Override
	public Data originalData() {
		return originalData;
	}

	@Override
	public int uncompressedSize() {
		ensureCompleted();
		return uncompressedSize;
	}

	@Override
	public String toString() {
		return "AsyncSerializableDataset[\n" +
				"  Archivzeit=" + Util.timestrMillisFormatted(archiveTime()) + "\n" +
				"  Datenzeit=" + Util.timestrMillisFormatted(dataTime()) + "\n" +
				"  Datenindex=" + Util.dIdx2StrExt(dataIndex()) + "\n" +
				"  Art=" + dataState() + "\n" +
				"  Komprimiert=" + Util.printHex(serializedData()) + "\n" +
				"  Speicherverbrauch=" + (uncompressedSize() == 0 ? serializedData().length : uncompressedSize()) + " Bytes\n" +
				']';
	}
}
