/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.writer;

import de.bsvrz.ars.ars.persistence.ContainerFile;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.losb.util.Util;

import static de.bsvrz.ars.ars.persistence.ContainerFile.*;

/**
 * Einfache {@link SerializableDataset}-Implementierung.
 *
 * @param archiveTime         Archivzeit
 * @param dataTime            Datenzeit
 * @param dataIndex           Datenindex
 * @param serializedData      Serialisierte Daten (nur Nutzdaten, ohne Zeitstempel etc.)
 * @param uncompressedSize    Unkomprimierte originale Größe oder
 *                            {@link ContainerFile#NOT_COMPRESSED}, falls nicht komprimiert.
 * @param isAlreadyCompressed Sind die Daten schon komprimiert?
 */
public record RegularSerializableDataset(
		long archiveTime,
		long dataTime,
		long dataIndex,
		@NotNull byte[] serializedData,
		int uncompressedSize,
		boolean isAlreadyCompressed
) implements SerializableDataset {

	/**
	 * Einfache {@link SerializableDataset}-Implementierung. Konstruktor ohne Komprimierung.
	 *
	 * @param archiveTime    Archivzeit
	 * @param dataTime       Datenzeit
	 * @param dataIndex      Datenindex
	 * @param serializedData Serialisierte Daten (nur Nutzdaten, ohne Zeitstempel etc.)
	 */
	public RegularSerializableDataset(long archiveTime, long dataTime, long dataIndex, byte[] serializedData) {
		this(archiveTime,
				dataTime,
				dataIndex,
				serializedData,
				ContainerFile.NOT_COMPRESSED,
				false);
	}

	@NotNull
	@Override
	public DataState dataState() {
		if (serializedData.length == 6) {
			if (Util.cmpBytes(serializedData, 0, NO_DATA)) {
				return DataState.NO_DATA;
			} else if (Util.cmpBytes(serializedData, 0, NO_SOURCE)) {
				return DataState.NO_SOURCE;
			} else if (Util.cmpBytes(serializedData, 0, NO_RIGHTS)) {
				return DataState.NO_RIGHTS;
			} else if (Util.cmpBytes(serializedData, 0, POT_GAP)) {
				return DataState.POSSIBLE_GAP;
			}
		}
		return DataState.DATA;
	}

	@Override
	public long estimateMemoryUsage() {
		return 45 + serializedData.length;
	}

	@Override
	public Data originalData() {
		return null;
	}


	@Override
	public String toString() {
		return "RegularSerializableDataset[\n" +
				"  Archivzeit=" + Util.timestrMillisFormatted(archiveTime()) + "\n" +
				"  Datenzeit=" + Util.timestrMillisFormatted(dataTime()) + "\n" +
				"  Datenindex=" + Util.dIdx2StrExt(dataIndex()) + "\n" +
				"  Art=" + dataState() + "\n" +
				"  Komprimiert=" + Util.printHex(serializedData()) + "\n" +
				"  Speicherverbrauch=" + (uncompressedSize() == 0 ? serializedData().length : uncompressedSize()) + " Bytes\n" +
				']';
	}

}
