/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.writer;

import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationFailedException;
import de.bsvrz.ars.ars.mgmt.tasks.ArchiveDataSerializer;
import de.bsvrz.ars.ars.persistence.*;
import de.bsvrz.ars.ars.persistence.directories.ActivePersistenceDirectory;
import de.bsvrz.ars.ars.persistence.index.IndexException;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.losb.datk.ContainerSettings;
import de.bsvrz.sys.funclib.losb.util.Util;

/**
 * Hilfsklasse, die die Archivierung von Datensätzen übernimmt und diese jeweils entweder an den offenen Container anhängt,
 * oder einen neuen Container anfängt. Dafür wird die Bedingung geprüft, ob ein Container abgeschlossen werden muss.
 * <p>
 * Der zu schreibende Datensatz wird erst mit dem {@link #getSerializer()} geschrieben, dort gepuffert, und
 * kann später von {@link ContainerFileHandle#appendSerializedData(ArchiveDataSerializer)} dort "abgeholt" werden
 * um die eigentlichen Bytes zum Schreiben abzurufen.
 * <p>
 * Dadurch ist diese Klasse nicht Threadsafe!
 */
public class SerializationHelper {
	private final ArchiveDataSerializer serializer = new ArchiveDataSerializer();
	private static final Debug _debug = Debug.getLogger();
	private final ContainerCreator containerCreator;
	private final boolean updateIndexes;
	private final CloseConditions closeConditions;

	/**
	 * Erstellt einen neuen SerializationHelper.
	 *
	 * @param closeConditions  Abschlussbedingungen
	 * @param containerCreator Hilfsklasse, die beim Anlegen von neuen Containern hilft
	 * @param updateIndexes    Sollen Indexe mit aktualisiert werden? Im Normalfall ist das der Fall, bei der Migration
	 *                         werden die Indexe erst im Nachhinein gebildet, dahier wird dann hier false übergeben.
	 */
	public SerializationHelper(CloseConditions closeConditions, ContainerCreator containerCreator, boolean updateIndexes) {
		this.closeConditions = closeConditions;
		this.containerCreator = containerCreator;
		this.updateIndexes = updateIndexes;
	}

	/**
	 * Schreibt einen Datensatz in den offenen Container oder legt einen neuen Container an.
	 *
	 * @param dataset              Datensatz
	 * @param persistenceDirectory persistenzverzeichnis
	 * @param containerDirectory   Containerverzeichnis
	 * @return Serialisierung erfolgreich?
	 * @throws PersistenceException           Fehler bei Serialisierung
	 * @throws IndexException                 Fehler bei Indexzugriff
	 * @throws SynchronizationFailedException Synchronisierung fehlgeschlagen
	 */
	public boolean writeData(SerializableDataset dataset, ActivePersistenceDirectory persistenceDirectory, ContainerDirectory containerDirectory) throws PersistenceException, IndexException, SynchronizationFailedException {
		int dataEntryLen = getSerializer().serializeData(dataset);
		return appendSerializedData(persistenceDirectory, containerDirectory, dataset, dataEntryLen);
	}

	/**
	 * Hängt die vorher serialisierten Daten an den offenen Container an.
	 *
	 * @param persistenceDirectory Persistenzverzeichnis
	 * @param containerFileDir     Containerverzeichnis
	 * @param dataset              Serialisierter Datensatz
	 * @param dataEntryLen         Länge der serialisierten Daten
	 * @return true, wenn erfolgreich, sonst false (insbesondere wenn der Datensatz aufgrund der Größe nicht archiviert werden konnte)
	 * @throws PersistenceException           Fehler beim Schreiben
	 * @throws IndexException                 Fehler beim Indexzugriff
	 * @throws SynchronizationFailedException Fehler bei Synchronisierung
	 */
	public boolean appendSerializedData(ActivePersistenceDirectory persistenceDirectory, ContainerDirectory containerFileDir, SerializableDataset dataset, int dataEntryLen) throws PersistenceException, IndexException, SynchronizationFailedException {

		OpenContainerData openContainerData;
		if (updateIndexes) {
			openContainerData = persistenceDirectory.getOpenContainerData(containerFileDir);
		} else {
			openContainerData = persistenceDirectory.getLoadedContainerData(containerFileDir);
		}

		long archTime = dataset.archiveTime();
		long dataIdx = dataset.dataIndex();
		long dataTime = dataset.dataTime();

		if (openContainerData instanceof StandardOpenContainerData containerData) {
			try (var _currentContainer = persistenceDirectory.accessOpenContainer(containerFileDir, containerData.getContainerId())) {
				if (!mustCloseContainer(containerData, dataIdx, dataTime, dataEntryLen, _currentContainer, containerFileDir.archiveDataKind())) {
					// Standard-Fall, an Container anhängen

					_currentContainer.appendSerializedData(serializer);
					persistenceDirectory.updateMaxValues(containerFileDir, dataIdx, archTime, dataTime);
					return true;
				}

				// Sonst: Container schließen und Daten in neuen Container schreiben (ganz unten in Methode)
				// Hinweis: Benötigt exklusives Locking auf Verzeichnis
				try (LockedContainerDirectory lockedDirectory = containerCreator.lockIndex(containerFileDir)) {
					ContainerFile cf = _currentContainer.getContainerFile();

					cf.closeContainer(containerData);
					if (updateIndexes) {
						persistenceDirectory.updateContainerIndex(_currentContainer, lockedDirectory);
						persistenceDirectory.updateStandardIndexes(lockedDirectory, containerData);
					}
					persistenceDirectory.removeOpenContainerData(lockedDirectory);
				}
			}
		}
		return createNewContainer(persistenceDirectory, containerFileDir, dataset, dataEntryLen);
	}

	/**
	 * Hängt die vorher serialisierten Daten an einen neuen Container an.
	 *
	 * @param persistenceDirectory Persistenzverzeichnis
	 * @param containerDirectory   Containerverzeichnis
	 * @param dataset              Serialisierter Datensatz
	 * @param dataEntryLen         Länge der serialisierten Daten
	 * @return true, wenn erfolgreich, sonst false (insbesondere wenn der Datensatz aufgrund der Größe nicht archiviert werden konnte)
	 * @throws PersistenceException           Fehler beim Schreiben
	 * @throws IndexException                 Fehler beim Indexzugriff
	 * @throws SynchronizationFailedException Fehler bei Synchronisierung
	 */
	public boolean createNewContainer(ActivePersistenceDirectory persistenceDirectory, ContainerDirectory containerDirectory, SerializableDataset dataset, int dataEntryLen) throws PersistenceException, IndexException, SynchronizationFailedException {

		long archTime = dataset.archiveTime();
		long dataIdx = dataset.dataIndex();
		long dataTime = dataset.dataTime();

		if (tooLargeForNewContainer(dataEntryLen, containerDirectory.getAtgId())) {
			return false;
		}
		long openContID = containerCreator.nextContainerID();
		try (LockedContainerDirectory lockedDirectory = containerCreator.lockIndex(containerDirectory)) {
			try (var currentContainer = persistenceDirectory.accessOpenContainer(lockedDirectory, openContID)) {
				currentContainer.createContainer();
				currentContainer.appendSerializedData(serializer);
				if (updateIndexes) {
					persistenceDirectory.updateContainerIndex(currentContainer, lockedDirectory);
					persistenceDirectory.addIndexEntries(lockedDirectory, dataIdx, archTime, dataTime, openContID);
				} else {
					final StandardOpenContainerData openContainerData = new StandardOpenContainerData(1, archTime, archTime, dataTime, dataTime, dataIdx, dataIdx, openContID);
					persistenceDirectory.setOpenContainerData(lockedDirectory, openContainerData);
				}
			}
		}
		return true;
	}


	/**
	 * Prüft, ob ein Container abgeschlossen werden muss. Die Prüfung basiert auf {@link #closeConditions} und Daten des offenen Containers.
	 * Der Aufrufer dieser Methode muss die Synchronisierung über den DataIdentNode durchführen.
	 *
	 * @param openContainerData Daten des offenen Containers
	 * @param dataIdx           Zu archivierender Datenindex
	 * @param dataTime          Zu archivierende Datenzeit
	 * @param dataEntryLen      Länge des zu archivierenden Datensatz in Bytes
	 * @param currentContainer  Aktueller Container
	 * @return Wahr, falls Container abgeschlossen werden muss, falsch sonst.
	 */
	private boolean mustCloseContainer(final StandardOpenContainerData openContainerData, long dataIdx, long dataTime, long dataEntryLen, ContainerFileHandle currentContainer, ArchiveDataKind adk) throws PersistenceException {
		long previousDataIndex = openContainerData.getMaxDataIdx();
		if (dataIdx < previousDataIndex) {
			_debug.fine("Datenindex-Ruecksprung in aktuellen Daten fuer " + currentContainer + ": " + Util.timestrMillisFormatted(dataTime) + " < " + Util.timestrMillisFormatted(openContainerData.getMaxDataTime()));
			return true;
		}
		/*
		 Bei nachgelieferten Daten muss die Datenzeit innerhalb eines Containers monoton sein,
		 damit das Einsortieren nach Datenzeit bei späteren Archivanfragen funktioniert.
		 Bei Online-Daten kann es Datenzeitrücksprünge geben (da dort eh nie einsortiert wird), 
		 aber die Startzeit des Containers darf nicht unterschritten werden.
		*/
		long previousDataTime = adk.isDelayed() ? openContainerData.getMaxDataTime() : openContainerData.getMinDataTime();

		if (dataTime < previousDataTime) {
			_debug.fine("Datenzeit-Ruecksprung in aktuellen Daten fuer " + currentContainer + ": " + Util.timestrMillisFormatted(dataTime) + " < " + Util.timestrMillisFormatted(previousDataTime));
			return true;
		}
		ContainerSettings.CloseCondition closeCond = closeConditions.getCloseCondition(currentContainer.getLocation().getAtgId());
		return checkCloseCondition(closeCond, openContainerData, dataTime, dataEntryLen, currentContainer);
	}

	/**
	 * Prüft extra auf ContainerGröße. Damit kann verhindert werden, dass ständig neue Container angelegt werden, wenn bereits mit einem Datum die Größe
	 * überschritten wuerde. Stattdessen wird mit der maximal erlaubten Größe eine Fehlermeldung ausgegeben. Der Aufrufer dieser Methode muss die
	 * Synchronisierung über den DataIdentNode durchfuehren.
	 *
	 * @return wahr, falls die maximale ContainerGröße überschritten ist, falsch sonst
	 */
	private boolean tooLargeForNewContainer(int dataEntryLen, long atgId) {
		long totalLen = ContainerFile.getHeaderLen() + dataEntryLen;
		ContainerSettings.CloseCondition closeCond = closeConditions.getCloseCondition(atgId);
		if (totalLen > closeCond.maxContSize) {
			_debug.warning("Datensatz (" + dataEntryLen + " Bytes) + Header (" + ContainerFile.getHeaderLen() + " Bytes) ist größer als die maximale Containerlänge (" + closeCond.maxContSize + " Bytes) und wurde nicht archiviert");
			return true;
		}
		return false;
	}

	/**
	 * Prüft, ob der aktuelle Container die angegebenen Kriterien zum Container-Abschliessen erfüllt.
	 *
	 * @param closeCond         Container-Abschluss-kriterien
	 * @param openContainerData Daten des offenen Containers
	 * @param dataTime          Zu archivierende Datenzeit
	 * @param dataEntryLen      Länge des zu archivierenden Datensatzes in Bytes
	 * @param currentContainer  Aktueller Container
	 * @return Wahr, wenn der Container abgeschlossen werden muss, falsch sonst.
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	private boolean checkCloseCondition(ContainerSettings.CloseCondition closeCond, final StandardOpenContainerData openContainerData, long dataTime, long dataEntryLen, ContainerFileHandle currentContainer) throws PersistenceException {
		return (openContainerData.getNumContainerEntries() >= closeCond.maxContAnzDS) || (dataTime - openContainerData.getMinDataTime() >= closeCond.maxContTime * 1000) || (currentContainer.getContainerSize() + dataEntryLen > closeCond.maxContSize); // erfordert die meiste Zeit
	}

	public ArchiveDataSerializer getSerializer() {
		return serializer;
	}
}
