/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.repair;

import com.google.common.base.Strings;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationFailedException;
import de.bsvrz.ars.ars.persistence.*;
import de.bsvrz.ars.ars.persistence.directories.mgmt.lock.LockFileManager;
import de.bsvrz.ars.ars.persistence.index.*;
import de.bsvrz.ars.ars.persistence.index.backend.management.AbstractIndex;
import de.bsvrz.ars.ars.persistence.index.backend.management.BaseIndex;
import de.bsvrz.ars.ars.persistence.index.backend.management.ColumnType;
import de.bsvrz.ars.ars.persistence.index.backend.management.IndexContentDescriptor;
import de.bsvrz.ars.ars.persistence.index.result.IndexResult;
import de.bsvrz.ars.ars.persistence.iter.DataIterator;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.time.Instant;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.zip.GZIPInputStream;

import static de.bsvrz.ars.ars.persistence.layout.DataKinds.getDataKindSuffix;

/**
 * Klasse, die die Inhalte beliebiger Archivdateien auf der Konsole ausgibt.
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings("UseOfSystemOutOrSystemErr")
public final class ArchiveFileViewer {

	/**
	 * Einsteigspunkt
	 * @param args Argumente
	 */
	public static void main(String[] args) {
		if(args.length == 0) {
			System.err.println("Keine Dateien angegeben");
			return;
		}
		for(String argument : args) {
			File file = new File(argument);
			if(!file.exists()) {
				System.err.println(file + ": Datei nicht gefunden");
			}
			else {
				analyzeFileOrDirectory(file);
			}
		}
	}

	/**
	 * Gibt eine Datei oder ein Verzeichnis rekursiv auf der Konsole aus.
	 * @param file Datei oder Verzeichnis
	 */
	public static void analyzeFileOrDirectory(final File file) {
		if(file.isFile()) {
			try {
				analyzeFile(file);
			}
			catch(IOException e) {
				System.err.print(file + ": Fehler beim Verarbeiten der Datei");
				e.printStackTrace(System.err);
			}
		}
		else if(file.isDirectory()) {
			File[] files = file.listFiles();
			if(files == null) {
				System.err.println(file + ": Verzeichnis ist leer");
				return;
			}
			Arrays.sort(files, Comparator.comparing(File::getName));
			for(File subFile : files) {
				analyzeFileOrDirectory(subFile);
			}
		}
		else {
			System.err.println(file + ": Wird ignoriert, keine Datei");
		}
	}

	/**
	 * Gibt den Inhalt einer einzelnen Datei aus
	 *
	 * @param file datei
	 * @throws IOException Fehler beim Lesen
	 */
	public static void analyzeFile(final File file) throws IOException {
		if(ContainerFile.CONT_FILENAME_FILTER.accept(file.getParentFile(), file.getName())) {
			try {
				StandaloneContainerFileHandle handle = new StandaloneContainerFileHandle(file.toPath());
				handle.readContainerHeader();
				printFileType("Containerdatei, Position: " + handle.getLocation(), file);
				printContainerHeadersTableStart();
				printContainerHeadersData(handle);
				System.out.println("Daten:");
				printContainerDataTable(handle.iterator());
			}
			catch(PersistenceException | SynchronizationFailedException e) {
				throw new IOException(e);
			}

		}
		else if(file.getName().equals(DeletedContainerFile.DELETED_CONTAINER_FILENAME)) {
			printFileType("Datei mit gelöschten Containern", file);
			dumpDeletedContainerFile(file);
		}
		else if(file.getName().equals(DeletedContainerFile.DELETED_CONTAINER_FILENAME_BACKUP)) {
			printFileType("Sicherungskopie von Datei mit gelöschten Containern", file);
			dumpDeletedContainerFile(file);
		}
		else if(file.getName().equals(DeletedContainerFile.DELETED_CONTAINER_FILENAME_TMP)) {
			printFileType("Temporäre Datei mit gelöschten Containern", file);
			dumpDeletedContainerFile(file);
		}
		else if(file.getName().equals(DataTimeIndexImpl.IDX_FILENAME)) {
			printFileType("Datenzeit-Index", file);
			dumpIndex(() -> new DataTimeIndexImpl(1024, file.toPath()));
		}
		else if(file.getName().equals(DataIndexIndexImpl.IDX_FILENAME)) {
			printFileType("Datenindex-Index", file);
			dumpIndex(() -> new DataIndexIndexImpl(1024, file.toPath()));
		}
		else if(file.getName().equals(ArchiveTimeIndexImpl.IDX_FILENAME)) {
			printFileType("Archivzeit-Index", file);
			dumpIndex(() -> new ArchiveTimeIndexImpl(1024, file.toPath()));
		}
		else if(file.getName().equals(ContainerManagementIndex.IDX_FILENAME)) {
			printFileType("Verwaltungsinformationen-Index", file);
			dumpIndex(() -> new ContainerManagementIndex(1024, file.toPath()));
		} else if (file.getName().endsWith(".datagaps") || file.getName().startsWith("_datagaps.txt")) {
			printFileType("Lückendatei zum Nachfordern", file);
			dumpRawFile(file);
		}
		else if(file.getName().equals(StartupProperties.STARTUP_PROP_FILE_NAME)){
			printFileType("Startup-Info", file);
			dumpRawFile(file);
		}
		else if(file.getName().equals(PersistenceManager.RESTART_TIME_FILE_NAME)){
			printFileType("Datenzeit-Neustart-Informationen", file);
			dumpRestartFileTimes(file);
		} else if (file.getName().equals(LockFileManager.ACCESS_FLAG_FILE_NAME)) {
			printFileType("Lock-File (Zugriff)", file);
		} else if (file.getName().equals(LockFileManager.NOT_PROTECTED_FLAG_FILE_NAME)) {
			printFileType("Lock-File (Beschreibbar)", file);
		}
		else {
			printFileType("Unbekannte Datei", file);
		}
	}

	private static void dumpRestartFileTimes(final File file) throws IOException {
		System.out.println("            Objekt-ID |    Attributgruppen-ID |             Aspekt-ID |  SV | DA |          Abmelde-Zeit");
		try(DataInputStream stream = new DataInputStream(new BufferedInputStream(new GZIPInputStream(Files.newInputStream(file.toPath()))))) {
			int numEntries = stream.readInt();
			for(int i = 0; i < numEntries; i++) {
				System.out.println("===================== | ===================== | ===================== | === | == | =====================");
				long time = stream.readLong();
				int numValues = stream.readInt();
				for(int f = 0; f < numValues; f++) {
					long objId = stream.readLong();
					long atgId = stream.readLong();
					long aspId = stream.readLong();
					int sv = stream.readShort();
					byte adkIndex = stream.readByte();
					System.out.print(Strings.padStart(String.valueOf(objId), 21, ' '));
					System.out.print(" | ");
					System.out.print(Strings.padStart(String.valueOf(atgId), 21, ' '));
					System.out.print(" | ");
					System.out.print(Strings.padStart(String.valueOf(aspId), 21, ' '));
					System.out.print(" | ");
					System.out.print(Strings.padStart(String.valueOf(sv), 3, ' '));
					System.out.print(" | ");
					System.out.print(getDataKindSuffix(adkIndex));
					System.out.print(" | ");
					System.out.println(Strings.padStart(String.valueOf(time), 21, ' '));
				}
			}
		}
	}

	private static void printFileType(final String fileType, final File file) {
		System.out.println();
		System.out.println(file);
		System.out.println("Dateityp: " + fileType + ", " + file.length() + " Bytes");
	}

	private static void dumpRawFile(final File file) throws IOException {
		Files.readAllLines(file.toPath(), StandardCharsets.US_ASCII).forEach(
				System.out::println
		);
	}

	private static <E extends Enum<E>> void dumpIndex(final IndexCreator<E> indexCreator) throws IOException {
		try(AbstractIndex<E> index = indexCreator.buildIndex()) {
			printTable(index.query());
		}
		catch(IndexException e) {
			throw new IOException(e);
		}
	}

	/**
	 * Gibt die Inhalte eines index auf System.out aus
	 * @param result Indexinhalte (Ergebnis von {@link BaseIndex#query()}
	 * @param <E> Typ der Werte in Index
	 */
	public static <E extends Enum<E>> void printTable(final IndexResult<E> result) {
		List<IndexContentDescriptor.IndexColumn> columns = result.getColumns();
		int[] columnWidths = new int[columns.size()];
		for(int i = 0; i < columns.size(); i++) {
			final IndexContentDescriptor.IndexColumn column = columns.get(i);
			columnWidths[i] = Math.max(column.getData().toString().length(), computeWidth(column));
		}
		for(int i = 0; i < columns.size(); i++) {
			final IndexContentDescriptor.IndexColumn column = columns.get(i);
			if(i != 0) {
				System.out.print(" | ");
			}
			System.out.print(Strings.padStart(column.getData().toString(), columnWidths[i], ' '));
		}
		System.out.println();
		for(int i = 0; i < columns.size(); i++) {
			final IndexContentDescriptor.IndexColumn column = columns.get(i);
			if(i != 0) {
				System.out.print(" | ");
			}
			System.out.print(Strings.padStart(column.getLengthBytes() + " " + formatType(column.getType()), columnWidths[i], ' '));
		}
		System.out.println();
		for(int i = 0; i < columns.size(); i++) {
			if(i != 0) {
				System.out.print("=|=");
			}
			System.out.print(Strings.repeat("=", columnWidths[i]));
		}
		System.out.println();
		for(int d = 0; d < result.size(); d++) {
			for(int i = 0; i < columns.size(); i++) {
				final IndexContentDescriptor.IndexColumn column = columns.get(i);
				if(i != 0) {
					System.out.print(" | ");
				}
				System.out.print(Strings.padStart(getString(result, d, column), columnWidths[i], ' '));
			}
			System.out.println();
		}
	}

	@SuppressWarnings("unchecked")
	private static <E extends Enum<E>> String getString(final IndexResult<E> result, final int d, final IndexContentDescriptor.IndexColumn column) {
		if(column.getType() == ColumnType.String) {
			return result.getString(d, (E) column.getData()).trim();
		}
		long number = result.get(d, (E) column.getData());
		String columnName = column.getData().toString();
		if(columnName.contains("DI") || columnName.contains("DataIndex")) {
			return Util.dIdx2Str(number);
		}
		if(columnName.equals("anzDS") && number == 4294967295L) {
			return "Offen";
		}
		if (columnName.contains("Time") || columnName.contains("Z")) {
			return Instant.ofEpochMilli(number).toString();
		}
		return String.valueOf(number);
	}


	private static String getString(final ContainerManagementData headers, final ContainerManagementInformation column) throws PersistenceException {
		if(!column.isNumeric()) {
			return headers.getContainerHeaderParamAsString(column).trim();
		}
		long number = headers.getContainerHeaderParamAsLong(column);
		String columnName = column.toString();
		if(columnName.contains("DI") || columnName.contains("DataIndex")) {
			return Util.dIdx2Str(number);
		}
		if(columnName.equals("anzDS") && (int)number == -1) {
			return "Offen";
		}
		if (columnName.contains("Z")) {
			return Instant.ofEpochMilli(number).toString();
		}
		return String.valueOf(number);
	}

	private static String formatType(final ColumnType type) {
		return switch (type) {
			case StrictlyIncreasing -> "SM";
			case Increasing -> "M";
			case Unordered -> "U";
			case Unique -> "Uq";
			case String -> "T";
		};
	}

	private static int computeWidth(final IndexContentDescriptor.IndexColumn column) {
		if(column.getType() == ColumnType.String) {
			return column.getLengthBytes();
		}
		else {
			return getStringWidthForBytes(column.getLengthBytes());
		}
	}

	public static int computeWidth(final ContainerManagementInformation column) {
		if(!column.isNumeric()) {
			return column.getValLen();
		} else if (column.toString().contains("Z")) {
			return 24;
		} else {
			return getStringWidthForBytes(column.getByteLength());
		}
	}

	/**
	 * Gibt für eine Byte-Anzahl die Mindest-Stringlänge zurück..
	 * @param lengthBytes Byte-Anzahl
	 * @return String-Länge
	 */
	private static int getStringWidthForBytes(final int lengthBytes) {
		return switch (lengthBytes) {
			case 8 -> 21; // Maximallänge für einen Long-Wert
			case 7 -> 17;
			case 6 -> 15;
			case 5 -> 13;
			case 4 -> 10;
			case 3 -> 8;
			case 2 -> 5;
			case 1 -> 3;
			default -> throw new IllegalArgumentException(String.valueOf(lengthBytes));
		};
	}

	private static void dumpDeletedContainerFile(final File file) throws IOException {
		try {
			DeletedContainerFile deletedContainerFile = new DeletedContainerFile(file.toPath().getParent(), null);
			deletedContainerFile.read();
			printContainerHeadersTableStart();
			for(Long contId : deletedContainerFile.containers()) {
				ContainerManagementData headers = deletedContainerFile.headers(contId);
				printContainerHeadersData(headers);
			}
			System.out.println("Gelöschte Bereiche:");
			System.out.println("       contID |                 DImin |                 DImax");
			System.out.println("==============|=======================|======================");
			for(Long contId : deletedContainerFile.containers()) {
				List<DeletedContainerFile.IndexRange> deletedBlocks = deletedContainerFile.deletedBlocks(contId);
				assert deletedBlocks != null;
				printDeletedBlocksTable(String.valueOf(contId), deletedBlocks);
			}
		}
		catch(PersistenceException e) {
			throw new IOException(e);
		}
	}

	private static void printContainerHeadersTableStart() {
		List<ContainerManagementInformation> columns = Arrays.asList(ContainerManagementInformation.values());
		int[] columnWidths = new int[columns.size()];
		for(int i = 0; i < columns.size(); i++) {
			final ContainerManagementInformation column = columns.get(i);
			columnWidths[i] = Math.max(column.toString().length(), computeWidth(column));
		}
		for(int i = 0; i < columns.size(); i++) {
			final ContainerManagementInformation column = columns.get(i);
			if(i != 0) {
				System.out.print(" | ");
			}
			System.out.print(Strings.padStart(column.toString(), columnWidths[i], ' '));
		}

		System.out.println();
		for(int i = 0; i < columns.size(); i++) {
			if(i != 0) {
				System.out.print("=|=");
			}
			System.out.print(Strings.repeat("=", columnWidths[i]));
		}
		System.out.println();
	}

	private static void printContainerHeadersData(final ContainerManagementData headers) throws PersistenceException {
		List<ContainerManagementInformation> columns = Arrays.asList(ContainerManagementInformation.values());
		int[] columnWidths = new int[columns.size()];
		for(int i = 0; i < columns.size(); i++) {
			final ContainerManagementInformation column = columns.get(i);
			columnWidths[i] = Math.max(column.toString().length(), computeWidth(column));
		}
		for(int i = 0; i < columns.size(); i++) {
			final ContainerManagementInformation column = columns.get(i);
			if(i != 0) {
				System.out.print(" | ");
			}
			System.out.print(Strings.padStart(getString(headers, column), columnWidths[i], ' '));
		}
		System.out.println();
	}

	private static void printContainerDataTable(final DataIterator dataIterator) throws PersistenceException, SynchronizationFailedException {
		final ContainerDataResult result = new ContainerDataResult();
		System.out.println("           Datenindex |         Datenzeit |        Archivzeit |                Zustand |      Bytes");
		System.out.println("======================|===================|===================|========================|===========");
		while(!dataIterator.isEmpty()) {
			dataIterator.poll(result);
			System.out.print(Strings.padStart(Util.dIdx2Str(result.getDataIndex()), 21, ' '));
			System.out.print(" | ");
			System.out.print(Strings.padStart(String.valueOf(result.getDataTime()), 17, ' '));
			System.out.print(" | ");
			System.out.print(Strings.padStart(String.valueOf(result.getArchiveTime()), 17, ' '));
			System.out.print(" | ");
			System.out.print(Strings.padStart(String.valueOf(result.getDataState()), 22, ' '));
			System.out.print(" | ");
			System.out.print(Strings.padStart(String.valueOf(result.getDataSize()), 10, ' '));
			System.out.println();
		}
	}

	private static void printDeletedBlocksTable(final String contId, final List<DeletedContainerFile.IndexRange> data) {
		for(DeletedContainerFile.IndexRange range : data) {
			System.out.print(Strings.padStart(contId, 13, ' '));
			System.out.print(" | ");
			System.out.print(Strings.padStart(Util.dIdx2Str(range.from()), 21, ' '));
			System.out.print(" | ");
			System.out.print(Strings.padStart(Util.dIdx2Str(range.to()), 21, ' '));
			System.out.println();
		}
	}

	@FunctionalInterface
	private interface IndexCreator<E extends Enum<E>> {
		AbstractIndex<E> buildIndex() throws IndexException;
	}
}
