/*
 *
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.repair;

import de.bsvrz.ars.ars.persistence.*;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.index.*;
import de.bsvrz.ars.ars.persistence.index.result.IndexResult;
import de.bsvrz.ars.ars.persistence.layout.DataIdentificationListener;
import de.bsvrz.ars.ars.persistence.layout.PersistenceDirectoryLayoutInstance;
import de.bsvrz.ars.ars.persistence.walk.internal.*;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.util.BufferedRandomAccessFile;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.io.*;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;
import java.util.regex.Pattern;

import static de.bsvrz.ars.ars.persistence.ContainerManagementInformation.*;
import static de.bsvrz.ars.ars.persistence.layout.DataKinds.getDataKindSuffix;

/**
 * Klasse, mit der es möglich ist, die Inhalte eines Persistenzverzeichnisses auf Konsistenz zu prüfen und
 * häufige Fehler (insbesondere aus alten Versionen) zu erkennen und nach Möglichkeit zu korrigieren.
 */
public class PersistenceCheckRunner {

	private static final FilenameFilter _containerFilenameFilter;

	private static final FilenameFilter _indexFilenameFilter;

	static {
		var _containerFilenamePattern = Pattern.compile("dc[0-9]*\\.dat");
		_containerFilenameFilter = (dir, name) -> _containerFilenamePattern.matcher(name).matches();
		var _indexFilenamePattern = Pattern.compile("_.+\\.idx");
		_indexFilenameFilter = (dir, name) -> _indexFilenamePattern.matcher(name).matches();
	}

	@NotNull
	public static RescueResult scan(PersistenceDirectory persistenceDirectory, ScanMode scanMode) throws PersistenceException {
		RescueResult rescueResult = new RescueResult();
		DataIdentificationListener listener = new DataIdentificationListener() {
			@Override
			public void foundDataIdentification(IdDataIdentification dataIdentification, Path dir) {
				rescueResult.incrDataIdentificationsVisited();
			}

			@Override
			public void foundForeignDirectory(Path dir) {
				new Defect(dir, rescueResult.getCantFix(), "Unerwartetes Verzeichnis in der Verzeichnisstruktur");
			}

			@Override
			public void foundForeignFile(Path file) {
				new Defect(file, rescueResult.getCantFix(), "Unerwartete Datei in der Verzeichnisstruktur");
			}

			@Override
			public void ioException(Path file, IOException exception) {
				new Defect(file, rescueResult.getCantFix(), "Fehler beim Zugriff", exception);
			}
		};
		StandardDataidentificationDirWalk containerDirWalk = new StandardDataidentificationDirWalk(DataIdentificationDirGetters.allWithDelegate(persistenceDirectory, listener));
		containerDirWalk.execute("Überprüfung", 11, (containerFileDir, standardContainerDirWalk) -> {
					Path dataIdentificationDir = containerFileDir.getDataIdentificationDir();
					for (ArchiveDataKind dataKind : containerFileDir.getArchiveDataKinds()) {
						String dataKindSuffix = getDataKindSuffix(dataKind);
						scanDirectory(dataIdentificationDir.resolve(dataKindSuffix),
								persistenceDirectory.getLayoutInstance(), scanMode, rescueResult);
					}
				}
		);
		return rescueResult;
	}

	public static void fixDefects(RescueResult rescueResult, PersistenceDirectoryLayoutInstance layoutInstance) {
		long t1 = System.currentTimeMillis();
		if (!rescueResult.getOneByteShortDefects().isEmpty()) {
			System.out.println("########## Korrektur der gelöschten Container, die ein Byte zu kurz sind. Anzahl: " + rescueResult.getOneByteShortDefects().size());
			final List<Defect> oneByteShortDefects = List.copyOf(rescueResult.getOneByteShortDefects());
			rescueResult.getOneByteShortDefects().clear();
			for (Defect oneByteShortDefect : oneByteShortDefects) {
				fixOneByteShortDefect(oneByteShortDefect);
			}
			System.out.println("  ########## Erneute Prüfung der korrigierten Container:");
			for (Defect oneByteShortDefect : oneByteShortDefects) {
				checkContainerFile(oneByteShortDefect.getFile(), rescueResult, true);
			}
			final long correctedDefectCount = oneByteShortDefects.size() - rescueResult.getOneByteShortDefects().size();
			System.out.println("  ########## Anzahl erfolgreich korrigierter Defekte: " + correctedDefectCount);
			rescueResult.incrCorrectedDefectCount(correctedDefectCount);
		}
		if (!rescueResult.getBadRangeEndDefects().isEmpty()) {
			System.out.println(
					"########## Korrektur der gelöschten Container, die eine falsche Endangabe eines Datensatzindexbereichs enthalten. Anzahl: "
							+ rescueResult.getBadRangeEndDefects().size()
			);
			final List<Defect> badRangeEndDefects = List.copyOf(rescueResult.getBadRangeEndDefects());
			rescueResult.getBadRangeEndDefects().clear();
			for (Defect badRangeEndDefect : badRangeEndDefects) {
				fixBadRangeEndDefect(badRangeEndDefect);
			}
			System.out.println("  ########## Erneute Prüfung der korrigierten Container:");
			for (Defect badRangeEndDefect : badRangeEndDefects) {
				checkContainerFile(badRangeEndDefect.getFile(), rescueResult, true);
			}
			final long correctedDefectCount = badRangeEndDefects.size() - rescueResult.getBadRangeEndDefects().size();
			System.out.println("  ########## Anzahl erfolgreich korrigierter Defekte: " + correctedDefectCount);
			rescueResult.incrCorrectedDefectCount(correctedDefectCount);
		}
		if (!rescueResult.getBadContainerFileEnd().isEmpty()) {
			System.out.println(
					"########## Korrektur der Container-Dateien mit ungültigen Bytes am Dateiende. Anzahl: "
							+ rescueResult.getBadContainerFileEnd().size()
			);
			final List<Defect> badContainerFileEnds = List.copyOf(rescueResult.getBadContainerFileEnd());
			rescueResult.getBadContainerFileEnd().clear();
			for (Defect badContainerFileEnd : badContainerFileEnds) {
				fixBadContainerFileEnd(badContainerFileEnd, layoutInstance);
			}
			System.out.println("  ########## Erneute Prüfung der korrigierten Container:");
			for (Defect badContainerFileEnd : badContainerFileEnds) {
				checkContainerFile(badContainerFileEnd.getFile(), rescueResult, true);
			}
			final long correctedDefectCount = badContainerFileEnds.size() - rescueResult.getBadContainerFileEnd().size();
			System.out.println("  ########## Anzahl erfolgreich korrigierter Defekte: " + correctedDefectCount);
			rescueResult.incrCorrectedDefectCount(correctedDefectCount);
		}
		if (!rescueResult.getShortContainerFiles().isEmpty()) {
			System.out.println(
					"########## Korrektur der leeren Container-Dateien. Anzahl: "
							+ rescueResult.getShortContainerFiles().size()
			);
			final List<Defect> shortContainerFiles = List.copyOf(rescueResult.getShortContainerFiles());
			rescueResult.getShortContainerFiles().clear();
			for (Defect shortContainerFile : shortContainerFiles) {
				fixShortContainerFile(shortContainerFile);
			}
			rescueResult.incrCorrectedDefectCount(shortContainerFiles.size());
		}
		if (!rescueResult.getBadIndexFiles().isEmpty()) {
			System.out.println(
					"########## Korrektur von Indexdateien, die fehlerhafte oder unvollständige Einträge enthalten. Anzahl: "
							+ rescueResult.getBadIndexFiles().size()
			);
			final List<Defect> badIndexFileDefects = List.copyOf(rescueResult.getBadIndexFiles());
			rescueResult.getBadIndexFiles().clear();
			for (Defect badIndexFileDefect : badIndexFileDefects) {
				fixBadIndexFileDefect(badIndexFileDefect);
			}
			System.out.println("  ########## Erneute Prüfung der korrigierten Indexdateien:");
			for (Defect badIndexFileDefect : badIndexFileDefects) {
				// Container-IDs neu holen, es könnten leere Container gelöscht worden sein
				Path directory = badIndexFileDefect.getFile().getParent();
				Path[] containerFiles = listFiles(directory, _containerFilenameFilter);
				HashSet<Long> containerIDs = getContainerIdsInDirectory(directory, layoutInstance, containerFiles);
				var actualContainerHeaders = loadContainerHeaders(directory, layoutInstance, containerFiles);
				checkIndexFile(badIndexFileDefect.getFile(), containerIDs, rescueResult, actualContainerHeaders);
			}
			final long correctedDefectCount = badIndexFileDefects.size() - rescueResult.getBadIndexFiles().size();
			System.out.println("  ########## Anzahl erfolgreich korrigierter Defekte: " + correctedDefectCount);
			rescueResult.incrCorrectedDefectCount(correctedDefectCount);
		}
		System.out.println("########## Gesamtzahl erfolgreich korrigierter Defekte: " + rescueResult.getCorrectedDefectCount());
		final long t2 = System.currentTimeMillis();
		long fixTime = t2 - t1;
		System.out.println("Dauer der Korrektur:             " + String.format("%.3f Sekunden", fixTime / 1000.0));
	}

	private static void fixShortContainerFile(final Defect defect) {
		Path containerFile = defect.getFile();
		try {
			Files.deleteIfExists(containerFile);
			Path indexFile = containerFile.resolveSibling(ContainerManagementIndex.IDX_FILENAME);
			Files.deleteIfExists(indexFile);
		} catch (Exception e) {
			System.out.println(containerFile + ": Unerwarteter Fehler bei der Reparatur: " + e);
		}
	}

	public static void scanDirectory(final Path directory, PersistenceDirectoryLayoutInstance layoutInstance, ScanMode scanMode, RescueResult rescueResult) {
		try {
			Path[] containerFiles = listFiles(directory, _containerFilenameFilter);
			if (scanMode.containerFiles() || scanMode.containerHeaderConsistency()) {
				for (Path containerFile : containerFiles) {
					checkContainerFile(containerFile, rescueResult, scanMode.containerHeaderConsistency());
				}
			}

			HashSet<Long> containerIDs = getContainerIdsInDirectory(directory, layoutInstance, containerFiles);
			if (scanMode.indexFiles() || scanMode.indexHeaderConsistency()) {

				Map<Long, ContainerHeaders> actualContainerHeaders = null;
				if (scanMode.indexHeaderConsistency()) {
					actualContainerHeaders = loadContainerHeaders(directory, layoutInstance, containerFiles);
				}

				Path[] indexFileArray = listFiles(directory, _indexFilenameFilter);
				for (Path indexFile : indexFileArray) {
					checkIndexFile(indexFile, containerIDs, rescueResult, actualContainerHeaders);
				}
			}
		} catch (Exception e) {
			System.out.println(directory.toString() + ": Unerwarteter Fehler bei der Analyse: " + e);
		}
	}

	private static Map<Long, ContainerHeaders> loadContainerHeaders(Path directory, PersistenceDirectoryLayoutInstance layoutInstance, Path[] containerFiles) {
		try {
			Map<Long, ContainerHeaders> actualContainerHeaders = new TreeMap<>();
			for (Path containerFile : containerFiles) {
				try (StandaloneContainerFileHandle containerFileHandle = new StandaloneContainerFileHandle(containerFile)) {
					containerFileHandle.readContainerHeader();
					actualContainerHeaders.put(containerFileHandle.getContainerId(), new ContainerHeaders(containerFileHandle));
				}
			}
			DeletedContainerFile deletedContainerFile = new DeletedContainerFile(directory, layoutInstance.getContainerDirectory(directory));
			if (deletedContainerFile.exists()) {
				deletedContainerFile.read();
				for (Long container : deletedContainerFile.containers()) {
					actualContainerHeaders.putIfAbsent(container, new ContainerHeaders(deletedContainerFile.headers(container)));
				}
			}
			return actualContainerHeaders;
		} catch (Exception e) {
			System.out.println(directory.toString() + ": Unerwarteter Fehler bei der Analyse: " + e);
			return null;
		}
	}

	private static HashSet<Long> getContainerIdsInDirectory(final Path directory, PersistenceDirectoryLayoutInstance layoutInstance, Path[] containerFileArray) {
		List<Path> containerFiles = Arrays.asList(containerFileArray);
		HashSet<Long> result = fileListToIdList(containerFiles);
		try {
			IdContainerFileDir containerDirectory = layoutInstance.getContainerDirectory(directory);
			DeletedContainerFile deletedContainerFile = new DeletedContainerFile(directory, containerDirectory);
			if (deletedContainerFile.exists()) {
				deletedContainerFile.read();
			}
			result.addAll(deletedContainerFile.containers());
		} catch (PersistenceException ignored) {
		}
		return result;
	}

	private static HashSet<Long> fileListToIdList(final List<Path> containerFiles) {
		final HashSet<Long> hashSet = new HashSet<>();
		for (Path containerFile : containerFiles) {
			hashSet.add(ContainerFile.getContID(containerFile.getFileName().toString()));
		}
		return hashSet;
	}

	private static void checkIndexFile(final Path indexFile, final HashSet<Long> containerIDs, RescueResult rescueResult, Map<Long, ContainerHeaders> actualContainerHeaders) {
		rescueResult.incrCheckedIndexFileCount();
		final Defect defect = inspectIndexFile(indexFile, containerIDs, rescueResult, actualContainerHeaders);
		if (defect == null) {
			rescueResult.incrValidIndexFileCount();
		} else {
			rescueResult.incrDefectIndexFileCount();
		}
	}

	private static Defect inspectIndexFile(final Path indexFile, final HashSet<Long> containerIDs, RescueResult rescueResult, Map<Long, ContainerHeaders> actualContainerHeaders) {
		try {
			if (!Files.exists(indexFile)) {
				// Eine fehlende Indexdatei wird automatisch neu erstellt und ist immer in Ordnung
				return null;
			}
			if (Files.isReadable(indexFile)) {
				try (RandomAccessFile raf = new RandomAccessFile(indexFile.toFile(), "r")) {
					String fileName = indexFile.getFileName().toString();
					if (fileName.startsWith("_didx")) {
						return checkDIDIndexFile(raf, indexFile, containerIDs, rescueResult, actualContainerHeaders);
					} else if (fileName.startsWith("_datatime")) {
						return checkDataTimeIndexFile(raf, indexFile, containerIDs, rescueResult, actualContainerHeaders);
					} else if (fileName.startsWith("_dataindex")) {
						return checkDataIndexFile(raf, indexFile, containerIDs, rescueResult, actualContainerHeaders);
					} else if (fileName.startsWith("_backupindex")) {
						checkBackupIndexFile(indexFile, rescueResult);
						return null;
					} else if (fileName.equals(ContainerManagementIndex.IDX_FILENAME)) {
						return checkManagementIndexFile(indexFile, containerIDs, rescueResult, actualContainerHeaders);
					}
				}
			} else {
				return new Defect(indexFile, rescueResult.getCantFix(), "Datei nicht lesbar");
			}
		} catch (EOFException e) {
			return new Defect(indexFile, rescueResult.getBadIndexFiles(), "Datei nicht vollständig");
		} catch (IOException e) {
			return new Defect(indexFile, rescueResult.getCantFix(), "Fehler beim Lesen: " + e);
		}
		return new Defect(indexFile, rescueResult.getCantFix(), "Unbekannter Dateityp");
	}

	private static Defect checkManagementIndexFile(final Path indexFile, final HashSet<Long> containerIDs, RescueResult rescueResult, Map<Long, ContainerHeaders> actualContainerHeaders) {
		try {
			final IndexResult<ContainerManagementInformation> result;
			final HashSet<Long> containerFileIDs;
			try (ContainerManagementIndex index = new ContainerManagementIndex(1024, indexFile)) {
				result = index.query();
			}
			containerFileIDs = new HashSet<>(containerIDs);

			for (int i = 0; i < result.size(); i++) {
				long contId = result.get(i, CHP_CONT_ID);

				if (actualContainerHeaders != null) {
					ContainerHeaders containerHeaders = actualContainerHeaders.get(contId);
					if (containerHeaders == null) {
						return new Defect(indexFile, rescueResult.getBadIndexFiles(), "Containerdatei fehlt: " + contId);
					}
					ContainerHeaders indexHeaders = new ContainerHeaders(result, i, containerHeaders.getLocation());

					int contAnz = containerHeaders.getContainerHeaderParamAsInt(CHP_ANZ_DS);
					int indexAnz = indexHeaders.getContainerHeaderParamAsInt(CHP_ANZ_DS);
					if (contAnz != indexAnz) {
						return new Defect(indexFile, rescueResult.getBadIndexFiles(), "Abweichung in Verwaltungsdatenindex von Container-Header: ANZ_DS (container: " + contAnz + ", index: " + indexAnz + ")");
					}
					if (contAnz != -1) {
						// Prüfung macht nur bei angeschlossenen Containern wirklich Sinn
						List<ContainerManagementInformation> relevantHeaders = List.of(
								CHP_DATA_IDX_MIN,
								CHP_DATA_IDX_MAX,
								CHP_ARC_TIME_MIN,
								CHP_ARC_TIME_MAX,
								CHP_DATA_TIME_MIN,
								CHP_DATA_TIME_MAX
						);
						for (ContainerManagementInformation info : relevantHeaders) {
							long contValue = containerHeaders.getContainerHeaderParamAsLong(info);
							long indexValue = indexHeaders.getContainerHeaderParamAsLong(info);
							if (contValue != indexValue) {
								return new Defect(indexFile, rescueResult.getBadIndexFiles(), "Abweichung in Verwaltungsdatenindex von Container-Header: " + info + " (container: " + contValue + ", index: " + indexValue + ")");
							}
						}
					}
				}

				containerFileIDs.remove(contId);
			}

			if (!containerFileIDs.isEmpty())
				return new Defect(indexFile, rescueResult.getBadIndexFiles(), "Es kommen nicht alle ContainerIDs in der Indexdatei vor.");
		} catch (IndexException e) {
			return new Defect(indexFile, rescueResult.getBadIndexFiles(), "ContainerHeaderIndex konnte nicht gelesen werden: " + e.getMessage());
		}
		return null;
	}

	private static void checkBackupIndexFile(final Path indexFile, RescueResult rescueResult) throws IOException {
		// BackupIndex wird nicht mehr benötigt und kann gelöscht werden
		if (Files.deleteIfExists(indexFile)) {
			rescueResult.incrDeletedFiles();
		}
	}

	/**
	 * Prüft Indexdateien der Klasse {@link ArchiveTimeIndexImpl}. In dieser Klasse ist gefordert, dass die Archivzeit monoton und
	 * die ContainerID streng monoton über alle Eintraege hinweg steigend sind.
	 *
	 * @param raf                    Geöffnete Datei
	 * @param fileName               Datei
	 * @param containerIDs           hashSet mit ContainerIDs, die in der IndexDatei vorkommen müssen.
	 * @param rescueResult           Ergebnisliste
	 * @param actualContainerHeaders Richtige Container-Header der Container
	 * @return Ein Defect, falls die Datei Defekt ist, sonst null
	 * @throws IOException Eingabe-Ausgabe-Fehler
	 */
	private static Defect checkDIDIndexFile(final RandomAccessFile raf, Path fileName, final HashSet<Long> containerIDs, RescueResult rescueResult, Map<Long, ContainerHeaders> actualContainerHeaders) throws IOException {
		final long length = raf.length();
		long atmax_old = Long.MIN_VALUE;
		long cid_old = Long.MIN_VALUE;

		final HashSet<Long> containerFileIDs = new HashSet<>(containerIDs);

		while (raf.getFilePointer() < length) {
			final long dimin = raf.readLong();
			final long dimax = raf.readLong();
			final long atmin = read6(raf);
			final long atmax = read6(raf);
			final long dtmin = read6(raf);
			final long dtmax = read6(raf);
			final long cid = read5(raf);

			if (!(atmax_old <= atmin))
				return new Defect(fileName, rescueResult.getBadIndexFiles(), "Archivzeit nicht monoton steigend");

			if (!(cid_old < cid))
				return new Defect(fileName, rescueResult.getBadIndexFiles(), "ContainerID nicht streng monoton steigend");

			if (actualContainerHeaders != null) {
				ContainerHeaders containerHeaders = actualContainerHeaders.get(cid);
				if (containerHeaders == null) {
					return new Defect(fileName, rescueResult.getBadIndexFiles(), "Containerdatei fehlt: " + cid);
				}
				if (containerHeaders.getContainerHeaderParamAsInt(CHP_ANZ_DS) != -1) {
					if (dimin != containerHeaders.getContainerHeaderParamAsLong(CHP_DATA_IDX_MIN))
						return new Defect(fileName, rescueResult.getBadIndexFiles(), "Abweichung in Indexdatei von Container-Header: DATA_IDX_MIN");
					if (dimax != containerHeaders.getContainerHeaderParamAsLong(CHP_DATA_IDX_MAX))
						return new Defect(fileName, rescueResult.getBadIndexFiles(), "Abweichung in Indexdatei von Container-Header: DATA_IDX_MAX");
					if (atmin != containerHeaders.getContainerHeaderParamAsLong(CHP_ARC_TIME_MIN))
						return new Defect(fileName, rescueResult.getBadIndexFiles(), "Abweichung in Indexdatei von Container-Header: ARC_TIME_MIN");
					if (atmax != containerHeaders.getContainerHeaderParamAsLong(CHP_ARC_TIME_MAX))
						return new Defect(fileName, rescueResult.getBadIndexFiles(), "Abweichung in Indexdatei von Container-Header: ARC_TIME_MAX");
					if (dtmin != containerHeaders.getContainerHeaderParamAsLong(CHP_DATA_TIME_MIN))
						return new Defect(fileName, rescueResult.getBadIndexFiles(), "Abweichung in Indexdatei von Container-Header: DATA_TIME_MIN");
					if (dtmax != containerHeaders.getContainerHeaderParamAsLong(CHP_DATA_TIME_MAX))
						return new Defect(fileName, rescueResult.getBadIndexFiles(), "Abweichung in Indexdatei von Container-Header: DATA_TIME_MAX");
				}
			}

			containerFileIDs.remove(cid);

			atmax_old = atmax;
			cid_old = cid;
		}

		if (!containerFileIDs.isEmpty())
			return new Defect(fileName, rescueResult.getBadIndexFiles(), "Es kommen nicht alle ContainerIDs in der Indexdatei vor.");

		return null;
	}

	/**
	 * Prüft Indexdateien der Klasse {@link DataTimeIndexImpl}. In dieser Klasse ist gefordert, dass die Einträge nach der minimalen
	 * Datenzeit sortiert sind
	 *
	 * @param raf                    Geöffnete Datei
	 * @param fileName               Datei
	 * @param containerIDs           hashSet mit ContainerIDs, die in der IndexDatei vorkommen müssen.
	 * @param rescueResult           Ergebnisliste
	 * @param actualContainerHeaders Richtige Container-Header der Container
	 * @return Ein Defect, falls die Datei Defekt ist, sonst null
	 * @throws IOException Eingabe-Ausgabe-Fehler
	 */
	private static Defect checkDataTimeIndexFile(final RandomAccessFile raf, Path fileName, final HashSet<Long> containerIDs, RescueResult rescueResult, Map<Long, ContainerHeaders> actualContainerHeaders) throws IOException {
		final long length = raf.length();
		long dtmin_old = Long.MIN_VALUE;

		final HashSet<Long> indexContainerIds = new HashSet<>();
		final HashSet<Long> containerFileIDs = new HashSet<>(containerIDs);

		while (raf.getFilePointer() < length) {
			final long dtmin = read6(raf);
			final long dtmax = read6(raf);
			final long dimin = raf.readLong();
			final long dimax = raf.readLong();
			final long cid = read5(raf);

			if (indexContainerIds.contains(cid))
				return new Defect(fileName, rescueResult.getBadIndexFiles(), "ContainerID mehrfach vorhanden");

			if (!(dtmin_old <= dtmin))
				return new Defect(fileName, rescueResult.getBadIndexFiles(), "Datenzeit nicht monoton steigend");

			if (actualContainerHeaders != null) {
				ContainerHeaders containerHeaders = actualContainerHeaders.get(cid);
				if (containerHeaders == null) {
					return new Defect(fileName, rescueResult.getBadIndexFiles(), "Containerdatei fehlt: " + cid);
				}
				if (containerHeaders.getContainerHeaderParamAsInt(CHP_ANZ_DS) != -1) {
					if (dimin != containerHeaders.getContainerHeaderParamAsLong(CHP_DATA_IDX_MIN))
						return new Defect(fileName, rescueResult.getBadIndexFiles(), "Abweichung in Indexdatei von Container-Header: DATA_IDX_MIN");
					if (dimax != containerHeaders.getContainerHeaderParamAsLong(CHP_DATA_IDX_MAX))
						return new Defect(fileName, rescueResult.getBadIndexFiles(), "Abweichung in Indexdatei von Container-Header: DATA_IDX_MAX");
					if (dtmin != containerHeaders.getContainerHeaderParamAsLong(CHP_DATA_TIME_MIN))
						return new Defect(fileName, rescueResult.getBadIndexFiles(), "Abweichung in Indexdatei von Container-Header: DATA_TIME_MIN");
					if (dtmax != containerHeaders.getContainerHeaderParamAsLong(CHP_DATA_TIME_MAX))
						return new Defect(fileName, rescueResult.getBadIndexFiles(), "Abweichung in Indexdatei von Container-Header: DATA_TIME_MAX");
				}
			}

			indexContainerIds.add(cid);
			containerFileIDs.remove(cid);
			dtmin_old = dtmin;
		}
		if (!containerFileIDs.isEmpty())
			return new Defect(fileName, rescueResult.getBadIndexFiles(), "Es kommen nicht alle ContainerIDs in der Indexdatei vor.");

		return null;
	}

	/**
	 * Prüft Indexdateien der Klasse {@link DataIndexIndexImpl}. In dieser Klasse ist gefordert, dass die Einträge nach der
	 * minimalen Datenzeit sortiert sind
	 *
	 * @param raf                    Geöffente Datei
	 * @param fileName               Datei
	 * @param containerIDs           hashSet mit ContainerIDs, die in der IndexDatei vorkommen müssen.
	 * @param rescueResult           Ergebnisliste
	 * @param actualContainerHeaders Richtige Container-Header der Container
	 * @return Ein Defect, falls die Datei Defekt ist, sonst null
	 * @throws IOException Eingabe-Ausgabe-Fehler
	 */
	private static Defect checkDataIndexFile(final RandomAccessFile raf, Path fileName, final HashSet<Long> containerIDs, RescueResult rescueResult, Map<Long, ContainerHeaders> actualContainerHeaders) throws IOException {
		final long length = raf.length();
		long dimin_old = Long.MIN_VALUE;

		final HashSet<Long> indexContainerIds = new HashSet<>();
		final HashSet<Long> containerFileIDs = new HashSet<>(containerIDs);

		while (raf.getFilePointer() < length) {
			final long dimin = raf.readLong();
			final long dimax = raf.readLong();
			final long cid = read5(raf);

			if (indexContainerIds.contains(cid))
				return new Defect(fileName, rescueResult.getBadIndexFiles(), "ContainerID mehrfach vorhanden");

			if (!(dimin_old <= dimin))
				return new Defect(fileName, rescueResult.getBadIndexFiles(), "Datenindex nicht monoton steigend");

			if (actualContainerHeaders != null) {
				ContainerHeaders containerHeaders = actualContainerHeaders.get(cid);
				if (containerHeaders == null) {
					return new Defect(fileName, rescueResult.getBadIndexFiles(), "Containerdatei fehlt: " + cid);
				}
				if (containerHeaders.getContainerHeaderParamAsInt(CHP_ANZ_DS) != -1) {
					if (dimin != containerHeaders.getContainerHeaderParamAsLong(CHP_DATA_IDX_MIN))
						return new Defect(fileName, rescueResult.getBadIndexFiles(), "Abweichung in Indexdatei von Container-Header: DATA_IDX_MIN");
					if (dimax != containerHeaders.getContainerHeaderParamAsLong(CHP_DATA_IDX_MAX))
						return new Defect(fileName, rescueResult.getBadIndexFiles(), "Abweichung in Indexdatei von Container-Header: DATA_IDX_MAX");
				}
			}

			indexContainerIds.add(cid);
			containerFileIDs.remove(cid);
			dimin_old = dimin;
		}
		if (!containerFileIDs.isEmpty())
			return new Defect(fileName, rescueResult.getBadIndexFiles(), "Es kommen nicht alle ContainerIDs in der Indexdatei vor.");

		return null;
	}

	private static long read5(final RandomAccessFile raf) throws IOException {
		return ((raf.readByte() & 0xffL) << 32) + (raf.readInt() & 0xFFFFFFFFL);
	}

	private static long read6(final RandomAccessFile raf) throws IOException {
		return ((raf.readShort() & 0xffffL) << 32) + (raf.readInt() & 0xFFFFFFFFL);
	}

	private static void fixBadIndexFileDefect(final Defect badIndexFileDefect) {
		try {
			Path indexFile = badIndexFileDefect.getFile();
			rebuildIndexFile(indexFile);
		} catch (Exception e) {
			System.out.println("Fehler beim Wiederherstellen der Indexdatei '" + badIndexFileDefect.getFile() + "'");
			e.printStackTrace(System.out);
		}
	}

	private static void rebuildIndexFile(final Path indexFile) throws IOException {
		// Indexe werden vom Archivsystem neu aufgebaut und können hier einfach gelöscht werden
		Files.delete(indexFile);
	}

	private static void checkContainerFile(final Path containerFile, RescueResult rescueResult, boolean checkConsistency) {
		//System.out.println("Prüfe " + containerFile);
		rescueResult.incrCheckedContainerFileCount();
		Defect defect = inspectContainerFile(containerFile, rescueResult);
		if (defect == null) {
			if (checkConsistency) {
				defect = checkConsistency(containerFile, rescueResult);
			}
		}
		if (defect == null) {
			rescueResult.incrValidContainerFileCount();
		} else {
			rescueResult.incrDefectContainerFileCount();
		}
	}

	private static Defect checkConsistency(Path containerFile, RescueResult rescueResult) {

		long lastDataIndex = Long.MIN_VALUE, lastArcTime = Long.MIN_VALUE;

		// Min/Max Werte der Nutzdaten/Paylod ermitteln
		long plDataIndexMin = Long.MAX_VALUE, plDataIndexMax = Long.MIN_VALUE;
		long plArcTimeMin = Long.MAX_VALUE, plArcTimeMax = Long.MIN_VALUE;
		long plDataTimeMin = Long.MAX_VALUE, plDataTimeMax = Long.MIN_VALUE;
		long plCount = 0;

		try {
			StandaloneContainerFileHandle containerFileHandle = new StandaloneContainerFileHandle(containerFile);
			containerFileHandle.readContainerHeader();
			if (!containerFileHandle.isContainerClosed()) {
				// Der offene Container muss nicht zwangsweise mit dem Index konsistent sein.
				return null;
			}
			long dIdxMin = containerFileHandle.getContainerHeaderParamAsLong(CHP_DATA_IDX_MIN);
			long dIdxMax = containerFileHandle.getContainerHeaderParamAsLong(CHP_DATA_IDX_MAX);
			long aTMin = containerFileHandle.getContainerHeaderParamAsLong(CHP_ARC_TIME_MIN);
			long aTMax = containerFileHandle.getContainerHeaderParamAsLong(CHP_ARC_TIME_MAX);
			long dTMin = containerFileHandle.getContainerHeaderParamAsLong(CHP_DATA_TIME_MIN);
			long dTMax = containerFileHandle.getContainerHeaderParamAsLong(CHP_DATA_TIME_MAX);
			ContainerDataIterator i = containerFileHandle.iterator();
			ContainerDataResult result = new ContainerDataResult();
			while (!i.isEmpty()) {
				plCount++;
				i.peek(result);
				// DataIndex
				if (result.getDataIndex() > plDataIndexMax) plDataIndexMax = result.getDataIndex();
				if (result.getDataIndex() < plDataIndexMin) plDataIndexMin = result.getDataIndex();
				if (lastDataIndex >= result.getDataIndex()) {
					return new Defect(containerFile, rescueResult.getCantFix(), "DatenIndex nicht streng monoton steigend\n" + "I_n-1=" + Util.dIdx2StrExt(lastDataIndex)
							+ " (" + lastDataIndex + "), I_n=" + Util.dIdx2StrExt(result.getDataIndex()) + " (" + result.getDataIndex() + ")");
				}
				if (result.getDataState() == DataState.NO_SOURCE && (Util.dIdxDaVBit(result.getDataIndex()) != 1 || Util.dIdxArSBit(result.getDataIndex()) != 0)) {
					return new Defect(containerFile, rescueResult.getCantFix(), "Datensatz mit Kennung 'keine Quelle' hat DAV-Bit nicht gesetzt oder Archiv-Bit gesetzt: Pos. " + result.getDataIndex());
				}
				if (result.getDataState() == DataState.NO_DATA && (Util.dIdxDaVBit(result.getDataIndex()) != 0 || Util.dIdxArSBit(result.getDataIndex()) != 0)) {
					return new Defect(containerFile, rescueResult.getCantFix(), "Datensatz mit Kennung 'keine Daten' hat DAV-Bit oder Archiv-Bit gesetzt: Pos. " + result.getDataIndex());
				}
				if (result.isData() && (Util.dIdxDaVBit(result.getDataIndex()) != 0 || Util.dIdxArSBit(result.getDataIndex()) != 0)) {
					return new Defect(containerFile, rescueResult.getCantFix(), "regulaerer Datensatz hat DAV-Bit oder Archiv-Bit gesetzt: Pos. " + result.getDataIndex());
				}
				if (result.getDataState() == DataState.POSSIBLE_GAP && (Util.dIdxDaVBit(result.getDataIndex()) != 0 || Util.dIdxArSBit(result.getDataIndex()) != 1)) {
					return new Defect(containerFile, rescueResult.getCantFix(), "Datensatz mit Kennung 'Datenluecke' hat DAV-Bit gesetzt oder Archiv-Bit nicht gesetzt: Pos. " + result.getDataIndex());
				}
				if (Util.dIdxSrcSubscrTime(result.getDataIndex()) == 0) {
					return new Defect(containerFile, rescueResult.getCantFix(), "Datenindex mit AnmeldezeitQuelle==0 gefunden: Pos. " + result.getDataIndex());
				}

				// ArcTime
				if (result.getArchiveTime() > plArcTimeMax) plArcTimeMax = result.getArchiveTime();
				if (result.getArchiveTime() < plArcTimeMin) plArcTimeMin = result.getArchiveTime();
				if (lastArcTime > result.getArchiveTime()) {
					return new Defect(containerFile, rescueResult.getCantFix(), "Archivzeit nicht monoton steigend\n" + "I_n-1=" + Util.timestrMillisFormatted(lastArcTime)
							+ " (" + lastArcTime + "), I_n=" + Util.timestrMillisFormatted(result.getArchiveTime()) + " (" + result.getArchiveTime() + ")"
					);
				}
				// DataTime
				if (result.getDataTime() > plDataTimeMax) plDataTimeMax = result.getDataTime();
				if (result.getDataTime() < plDataTimeMin) plDataTimeMin = result.getDataTime();

				lastDataIndex = result.getDataIndex();
				lastArcTime = result.getArchiveTime();

				i.remove();
			}
			i.close();

			// Anzahl Datensätze
			long headerCount = containerFileHandle.getContainerHeaderParamAsLong(ContainerHdr.CHP_ANZ_DS);
			if (plCount != headerCount) {
				return new Defect(containerFile, rescueResult.getCantFix(), "Anzahl Datensätze: Headerinformation (" + headerCount
						+ ") stimmt nicht mit den Nutzdaten (" + plCount + ") überein.");
			}

			// Indexe
			if (plDataIndexMin != dIdxMin) {
				return new Defect(containerFile, rescueResult.getCantFix(), "DataIndexMin: Headerinformation (" + dIdxMin
						+ ") stimmt nicht mit den Nutzdaten (" + plDataIndexMin + ") überein.");
			}
			if (plDataIndexMax != dIdxMax) {
				return new Defect(containerFile, rescueResult.getCantFix(), "DataIndexMax: Headerinformation (" + dIdxMax
						+ ") stimmt nicht mit den Nutzdaten (" + plDataIndexMax + ") überein.");
			}
			if (plArcTimeMin != aTMin) {
				return new Defect(containerFile, rescueResult.getCantFix(), "ArcTimeMin: Headerinformation (" + aTMin
						+ ") stimmt nicht mit den Nutzdaten (" + plArcTimeMin + ") überein.");
			}
			if (plArcTimeMax != aTMax) {
				return new Defect(containerFile, rescueResult.getCantFix(), "ArcTimeMax: Headerinformation (" + aTMax
						+ ") stimmt nicht mit den Nutzdaten (" + plArcTimeMax + ") überein.");
			}
			if (plDataTimeMin != dTMin) {
				return new Defect(containerFile, rescueResult.getCantFix(), "DataTimeMin: Headerinformation (" + dTMin
						+ ") stimmt nicht mit den Nutzdaten (" + plDataTimeMin + ") überein.");
			}
			if (plDataTimeMax != dTMax) {
				return new Defect(containerFile, rescueResult.getCantFix(), "DataTimeMax: Headerinformation (" + dTMax
						+ ") stimmt nicht mit den Nutzdaten (" + plDataTimeMax + ") überein.");
			}
		} catch (PersistenceException e) {
			return new Defect(containerFile, rescueResult.getCantFix(), "Fehler beim Lesen der Containerdatei", e);
		}
		return null;
	}


	private static Defect inspectContainerFile(final Path containerFile, RescueResult rescueResult) {

		if (!Files.exists(containerFile)) {
			// das kann passieren, nachdem eine Korrektur die Datei gelöscht hat und ist kein Fehler.
			return null;
		}

		final long fileLength;
		try {
			fileLength = Files.size(containerFile);
		} catch (IOException e) {
			return new Defect(containerFile, rescueResult.getCantFix(), "Die Dateigröße konnte nicht bestimmt werden.", e);
		}
		if (fileLength < 384) {
			return new Defect(containerFile, rescueResult.getCantFix(), "Unerwartete Dateigröße. Erwartet: mindestens 384 Bytes, gefunden: " + fileLength + " Bytes");
		} else if (fileLength == 384) {
			return new Defect(containerFile, rescueResult.getShortContainerFiles(), "Containerdatei enthält keine Daten");
		}
		try (BufferedInputStream in = new BufferedInputStream(Files.newInputStream(containerFile))) {
			byte[] header = new byte[384];
			final int headerLength = in.read(header);
			if (headerLength < 384) {
				return new Defect(
						containerFile, rescueResult.getCantFix(), "Unerwartete Anzahl gelesener Headerbytes. Erwartet: 384 Bytes, gefunden: " + headerLength + " Bytes"
				);
			}
			if (header[0] != 'v' || header[1] != '1' || header[2] != '0' || header[3] != '0') {
				return new Defect(
						containerFile,
						rescueResult.getCantFix(),
						"Unerwartete Containerversion. Erwartet: 'v100', gefunden: '" + (char) header[0] + (char) header[1] + (char) header[2] + (char) header[3]
								+ "'"
				);
			}
			if (header[4] != 0 || header[5] != 0 || header[6] != 1 || header[7] != 0x78) {
				return new Defect(
						containerFile, rescueResult.getCantFix(), "Unerwartete Headerlänge. Erwartet: 0x00000178, gefunden: " + String.format(
						"0x%02x%02x%02x%02x", header[4], header[5], header[6], header[7]
				)
				);
			}
			final char[] deletedParamNameChars = new char[10];
			for (int i = 0; i < deletedParamNameChars.length; i++) {
				deletedParamNameChars[i] = (char) header[0x14d + i];
			}
			final String deletedParamName = String.valueOf(deletedParamNameChars);
			if (!deletedParamName.equals("geloescht=")) {
				return new Defect(containerFile, rescueResult.getCantFix(), "Unerwartete Lösch-Kennung. Erwartet: 'geloescht=', gefunden: '" + deletedParamName + "'");
			}
			final boolean containerDeleted;
			if (header[0x14d + 10] == '0') {
				containerDeleted = false;
			} else if (header[0x14d + 10] == '1') {
				containerDeleted = true;
			} else {
				return new Defect(
						containerFile, rescueResult.getCantFix(), "Unerwarteter Wert der Lösch-Kennung. Erwartet: '0' oder '1', gefunden: " + (char) header[0x14d + 10]
				);
			}
			if (containerDeleted) {
				// Hinweis: Der folgende Fehler ist im Archivsystem korrigiert, es können Dateien von älteren Versionen repariert werden:
				//
				// Beim Löschen eines Containers mit der Methode ContainerFile.deleteContainerData() werden die in der Container-Datei enthaltenen Datensätze
				//  ersetzt durch eine optimierte Aufzählung der verwendeten Datensatzindizes. Dabei wird die Methode createDataIndexBuffer() verwendet, die
				//  diese optimierte Aufzählung erzeugt. Diese Methode gibt i.A. ein Byte zuviel zurück. Die deleteContainerData() Methode schreibt die von
				//  createDataIndexBuffer() zurückgelieferten Bytes und verkürzt anschließend die Datei um grundsätzlich ein Byte. In bestimmten Fällen, z.B.,
				//  wenn in der Container-Datei nur ein Datensatz enthalten ist oder nur Datensätze mit Sprüngen im Datensatzindex enthalten sind, liefert die
				//  Methode createDataIndexBuffer() allerdings die korrekte Anzahl Bytes zurück, sodass durch das anschließende Verkürzen der Datei um ein Byte
				//  ein Informationsverlust in der Datei entsteht. Es sollte möglich sein mit Hilfe des maximalen Datensatz-Index, der im Datei-Header
				//  gespeichert ist, das fehlende Byte zu berechnen und die defekte Datei zu reparieren.
				final long mod17remainder = (fileLength - 384) % 17;
				if (mod17remainder != 0) {
					if (mod17remainder == 16) {
						return new Defect(containerFile, rescueResult.getOneByteShortDefects(), "Falsche Dateilänge einer gelöschten Datei. Es fehlt ein Byte.");
					} else {
						return new Defect(
								containerFile,
								rescueResult.getCantFix(),
								"Unerwartete Dateilänge einer gelöschten Datei. Erwartet: 384 + 17 * x, gefunden : 384 + 17 * y + " + mod17remainder
						);
					}
				}

				// Beim Löschen eines Containers, in dem als erster Datensatz ein Potentielle-Lücke-Datensatz enthalten ist, wird die optimierte Aufzählung der
				//  enthaltenen Datensatzindizes mit falschem Endwert (-9223372036854775808) des ersten Bereichs erzeugt. Eine Reparatur der fehlerhaften
				//  Container-Dateien sollte möglich sein, indem der fehlerhafte Endwert -9223372036854775808, durch den entsprechenden Startwert ersetzt wird.
				byte[] range = new byte[17];
				if (in.read(range) == 17) {
					if (range[9] == -128 && range[10] == 0 && range[11] == 0 && range[12] == 0 && range[13] == 0 && range[14] == 0 && range[15] == 0
							&& range[16] == 0) {
						return new Defect(
								containerFile,
								rescueResult.getBadRangeEndDefects(),
								"Falsche Endangabe eines Datensatzindexbereichs in einem gelöschten Container. Erwartet: gültiger Datensatzindex, gelesen: 0x8000000000000000"
						);
					}
				}
			} else {
				// Über Datensätze iterieren
				DataInputStream dataInputStream = new DataInputStream(in);

				while (dataInputStream.available() > 0) {
					try {
						int length = dataInputStream.readInt();

						if (length <= 0) {
							return new Defect(containerFile, rescueResult.getBadContainerFileEnd(), "Der Datenbereich der Containerdatei ist nicht vollständig geschrieben worden, es befinden sich ungültige Bytes am Dateiende.");
						}

						byte[] tmp = new byte[length];
						dataInputStream.readFully(tmp);

						if (tmp[tmp.length - 3] != '#' || tmp[tmp.length - 2] != '#' || tmp[tmp.length - 1] != '\n') {
							// Jeder Datensatz muss auf "##\n" enden
							return new Defect(containerFile, rescueResult.getBadContainerFileEnd(), "Der Datenbereich der Containerdatei ist nicht vollständig geschrieben worden, es befinden sich ungültige Bytes am Dateiende.");
						}

					} catch (EOFException e) {
						return new Defect(containerFile, rescueResult.getBadContainerFileEnd(), "Der Datenbereich der Containerdatei ist nicht vollständig geschrieben worden, es befinden sich ungültige Bytes am Dateiende.", e);
					}

				}
			}
		} catch (IOException e) {
			return new Defect(containerFile, rescueResult.getCantFix(), "Lesefehler", e);
		}
		// ignored
		return null;
	}

	/**
	 * Beim Löschen eines Containers mit der Methode ContainerFile.deleteContainerData() werden die in der Container-Datei enthaltenen Datensätze ersetzt durch
	 * eine optimierte Aufzählung der verwendeten Datensatzindizes. Dabei wird die Methode createDataIndexBuffer() verwendet, die diese optimierte Aufzählung
	 * erzeugt. Diese Methode gibt i.A. ein Byte zuviel zurück. Die deleteContainerData() Methode schreibt die von createDataIndexBuffer() zurückgelieferten Bytes
	 * und verkürzt anschließend die Datei um grundsätzlich ein Byte. In bestimmten Fällen, z.B., wenn in der Container-Datei nur ein Datensatz enthalten ist oder
	 * nur Datensätze mit Sprüngen im Datensatzindex enthalten sind, liefert die Methode createDataIndexBuffer() allerdings die korrekte Anzahl Bytes zurück,
	 * sodass durch das anschließende Verkürzen der Datei um ein Byte ein Informationsverlust in der Datei entsteht. Eine Reparatur der fehlerhaften
	 * Container-Dateien wird mit dieser Methode durchgeführt, indem mit Hilfe des maximalen Datensatz-Index, der im Datei-Header gespeichert ist, das fehlende
	 * Byte berechnet und die defekte Datei damit korrigiert wird.
	 *
	 * @param defect Identifizierter Defekt einer Containerdatei
	 */
	private static void fixOneByteShortDefect(final Defect defect) {
		try (RandomAccessFile container = new RandomAccessFile(defect.getFile().toFile(), "rw")) {
			container.seek(0xA4);
			final byte[] propertyName = new byte[6];
			container.readFully(propertyName);
			final char[] properyValueChars = new char[19];
			for (int i = 0; i < properyValueChars.length; i++) {
				properyValueChars[i] = (char) container.readByte();
			}
			final byte propertySeparator = container.readByte();
			if (propertyName[0] != 'D' || propertyName[1] != 'I' || propertyName[2] != 'm' || propertyName[3] != 'a' || propertyName[4] != 'x'
					|| propertyName[5] != '=') {
				System.out.println(defect.getFile() + ": Reparatur nicht möglich, weil die Kennung DImax= nicht an der erwarteten Stelle ist.");
				return;
			}
			if (propertySeparator != '\n') {
				System.out.println(
						defect.getFile() + ": Reparatur nicht möglich, weil der Wert der Kennung DImax= nicht mit NewLine terminiert ist."
				);
				return;
			}
			final String propertyValueString = String.valueOf(properyValueChars);
			final long propertyValue = Long.parseLong(propertyValueString);
			final long fileLength = container.length();
			container.seek(fileLength - 8);
			final long lastLongValue = container.readLong() << 8;
			if (lastLongValue != Long.MIN_VALUE && lastLongValue != (propertyValue & 0xffffffffffffff00L)) {
				System.out.println(
						defect.getFile() + ": Reparatur nicht möglich, weil die letzten 7 Byte der Datei nicht den erwarteten Werten entsprechen."
				);
				return;
			}
			container.seek(fileLength - 7);
			container.writeLong(propertyValue);
			System.out.println(defect.getFile() + ": Container wurde repariert");
		} catch (IOException e) {
			System.out.println(defect.getFile() + ": Unerwarteter Fehler bei der Reparatur: " + e);
		}
	}

	/**
	 * Beim Löschen eines Containers, in dem als erster Datensatz ein Potentielle-Lücke-Datensatz enthalten ist, wird die optimierte Aufzählung der enthaltenen
	 * Datensatzindizes mit falschem Endwert (-9223372036854775808) des ersten Bereichs erzeugt. Eine Reparatur der fehlerhaften Container-Dateien wird durch diese
	 * Methode durchgeführt, indem der fehlerhafte Endwert -9223372036854775808, durch den entsprechenden Startwert ersetzt wird.
	 *
	 * @param defect Identifizierter Defekt einer Containerdatei
	 */
	private static void fixBadRangeEndDefect(final Defect defect) {
		try (RandomAccessFile container = new RandomAccessFile(defect.getFile().toFile(), "rw")) {
			container.seek(384);
			final long startRangeIndex = container.readLong();
			final byte separator = container.readByte();
			final long badEndRangeIndex = container.readLong();
			if (separator != '-') {
				System.out.println(defect.getFile() + ": Reparatur nicht möglich, weil Separatorzeichen nicht an erwarteter Stelle.");
				return;
			}
			if (badEndRangeIndex != Long.MIN_VALUE) {
				System.out.println(defect.getFile() + ": Reparatur nicht möglich, weil Bereichsende des ersten Bereichs nicht defekt ist.");
				return;
			}
			container.seek(384 + 9);
			container.writeLong(startRangeIndex);
			System.out.println(defect.getFile() + ": Container wurde repariert");
		} catch (IOException e) {
			System.out.println(defect.getFile() + ": Unerwarteter Fehler bei der Reparatur: " + e);
		}
	}

	private static void fixBadContainerFileEnd(final Defect defect, PersistenceDirectoryLayoutInstance layoutInstance) {
		Path containerFile = defect.getFile();
		long anzDs = -1;
		try (BasicContainerFileHandle handle = new BasicContainerFileHandle(containerFile, layoutInstance.getContainerDirectory(containerFile.getParent()))) {
			handle.ensureHeaderRead();
			anzDs = handle.getContainerHeaderParamAsLong(ContainerHdr.CHP_ANZ_DS);
		} catch (PersistenceException e) {
			System.out.println(containerFile + ": Unerwarteter Fehler bei der Reparatur: " + e);
		}

		try (BufferedRandomAccessFile container = new BufferedRandomAccessFile(containerFile.toFile(), "rw")) {
			if (container.length() < 384) {
				// Wir wollen direkt in den catch-Block unten springen, statt die Containerdatei zu verlängern
				throw new IOException("Container hat keinen Header, Länge ist nur " + container.length() + " Bytes.");
			}
			container.seek(384);

			long lastValidPosition = 384;
			long actualAnzDs = 0;
			final long oldLength = container.length();
			while (container.getFilePointer() < oldLength) {
				try {
					lastValidPosition = container.getFilePointer();
					int length = container.readInt();

					if (length <= 0) {
						break;
					}

					byte[] tmp = new byte[length];
					container.readFully(tmp);

					if (tmp[tmp.length - 3] != '#' || tmp[tmp.length - 2] != '#' || tmp[tmp.length - 1] != '\n') {
						// Jeder Datensatz muss auf "##\n" enden
						break;
					}

				} catch (EOFException ignored) {
					break;
				}
				actualAnzDs++;
			}
			if (lastValidPosition == container.length()) {
				System.out.println(containerFile + ": Unerwarteter Fehler bei der Reparatur: Es wurde kein Fehler zum reparieren gefunden");
			}
			if (anzDs == -1 || anzDs == actualAnzDs) {
				container.setLength(lastValidPosition);
			} else {
				System.out.println(containerFile + ": Fehler bei der Reparatur: Die Anzahl der gültigen Datensätze im Container stimmt nicht mit dem Container-Header überein.");
				return;
			}
		} catch (Exception e) {
			System.out.println(containerFile + ": Unerwarteter Fehler bei der Reparatur: " + e);
			return;
		}

		try {
			rebuildIndexFiles(containerFile.getParent());
		} catch (Exception e) {
			System.out.println(containerFile + ": Unerwarteter Fehler bei der Reparatur: " + e);
		}

	}

	private static void rebuildIndexFiles(final Path parentFile) throws IOException {
		Path[] indexFiles = listFiles(parentFile, _indexFilenameFilter);
		for (Path indexFile : indexFiles) {
			rebuildIndexFile(indexFile);
		}
	}

	@NotNull
	private static Path[] listFiles(Path parentFile, FilenameFilter filter) {
		File[] files = parentFile.toFile().listFiles(filter);
		if (files == null) {
			return new Path[0];
		}
		return Arrays.stream(files).map(File::toPath).toArray(Path[]::new);
	}


}
