/*
 * Copyright 2005 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2020-2023 by DTV-Verkehrsconsult, Aachen
 *
 * ALL RIGHTS RESERVED.
 *
 * THIS  SOFTWARE  IS  PROVIDED  "AS IS"  AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES,  INCLUDING,  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF  MERCHANTABILITY  AND  FITNESS  FOR  A  PARTICULAR  PURPOSE  ARE
 * DISCLAIMED. IN NO EVENT SHALL DTV-VERKEHRSCONSULT GmbH, OR ITS CON-
 * TRIBUTORS BE LIABLE FOR ANY  DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY,  OR CONSEQUENTIAL  DAMAGES  (INCL.,  BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;  LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCL.
 * NEGLIGENCE  OR OTHERWISE)  ARISING  IN  ANY  WAY OUT  OF THE USE OF
 * THIS  SOFTWARE,  EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.persistence;

import de.bsvrz.ars.ars.persistence.IdContainerFileDir;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.main.impl.archive.*;
import de.bsvrz.dav.daf.main.impl.archive.filesaver.ArchiveFileSaver;
import de.bsvrz.sys.funclib.dataSerializer.Deserializer;
import de.bsvrz.sys.funclib.dataSerializer.NoSuchVersionException;
import de.bsvrz.sys.funclib.dataSerializer.SerializingFactory;
import de.bsvrz.sys.funclib.debug.Debug;

import java.io.*;
import java.nio.file.Files;
import java.nio.file.Path;

/**
 * Diese Klasse implementiert einen Container, dieser speichert Datensätze und zusätzliche Containerinformationen. Die
 * Klasse ermöglicht es bestehende Container, die als Datei vorliegen, einzuladen und daraus Java-Objekte zu erzeugen,
 * gleichzeitig ist es möglich bestehende Container-Objekte in Dateien abzubilden.
 * <p>
 * Der Container bietet weiterhin die Möglichkeit die sogenannte Sicherung (zum Beispiel eine Implementierung des
 * Interfaces {@link ArchiveFileSaver}) zu beauftragen den Container persistent auf einem
 * Speichermdium Typ B zu sichern.
 * <p>
 * Aufbau der Datei, die einen Container speichert (Version 1): K2S Schriftzug, Containerversion, headerOffset (wo
 * beginnt der Header),
 * <p>
 * dataOffset (wo beginnen die Datensätze),
 * <p>
 * dataTableOffset (wo beginnt die Tabelle,
 * <p>
 * die die genaue Position der Datensätze speichert. Bei nicht abgeschlossenen Container ist die Tabelle noch nicht in
 * der Datei gespeichert),
 * <p>
 * ContainerId,
 * <p>
 * Anzahl Datensätze,
 * <p>
 * Vorhaltezeitraum,
 * <p>
 * max DataTime,
 * <p>
 * max ArchiveTime,
 * <p>
 * max DataIndex,
 * <p>
 * min DataTime,
 * <p>
 * min ArchiveTime,
 * <p>
 * min DataIndex,
 * <p>
 * Container abgeschlossen,
 * <p>
 * soll der Container gespeichert werden,
 * <p>
 * eindeutige Identifizierung Typ B (fehlerhaft, wird beim einladen des Containers aus der Sicherung aber erneut gesetzt
 * und ist dann aktuell),
 * <p>
 * Serialisierer-Version,
 * <p>
 * Größe byte-Array (serialisierte ContainerIdentifikation),
 * <p>
 * byte-Array(serialisierte ContainerIdentifikation)
 *
 * @author Kappich+Kniß Systemberatung Aachen (K2S)
 * @author Achim Wullenkord (AW)
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class Container implements PersistentDataContainer {

    /**
     * K2S Schriftzug
     */
    private static final String _k2sWriting = "K2S Container";

    /**
     * Eindeutiger Index dieses Containers
     */
    private final long _containerId;

    /**
     * Serilizer Version, die benutzt wurde um die Datensätze zu serialieseren. Der initiale Wert von -1 deutete an, dass
     * noch kein Serializer benutzt wurde.
     */
    private int _serializerVersion = -1;

    /**
     * Anzahl Datensätze in diesem Container.
     */
    private int _dataCount = 0;

    /**
     * aktueller Speicherberdarf des Containers
     */
    private long _containerSize = -1;

    /**
     * Pfad, in dem die Persistenz den Container auf ihrem Speichermedium ablegt (Pfad + Dateiname)
     */
    private final Path _containerFile;

    /**
     * Pfad eines externen Mediums, auf dem der Container gespeichert wird (Typ B Speicher)
     */
    private int _labelTypeB = -1;

    /**
     * Bestimmt die minimalen Werte von Datenzeitstempel, Archivzeitstempel und Datensatzindex aller in diesem Container
     * mit {@link #storeData} gespeicherten Daten.
     */
    private DataTiming _minimumDataTiming;

    /**
     * Bestimmt die maximalen Werte von Datenzeitstempel, Archivzeitstempel und Datensatzindex aller in diesem Container
     * mit {@link #storeData} gespeicherten Daten.
     */
    private DataTiming _maximumDataTiming;

    /**
     * Zeigt an, ob der Container abgeschlossen wurde
     */
    private boolean _containerFinalized = false;

    /**
     * Zeigt, ob der Container gespeichert werden soll
     */
    private boolean _storeContainer = false;

    /**
     * Wann soll dieser Container gelöscht werden (Zeit in ms ab 1970, abseluter Wert)
     */
    private long _deleteTime = -1;

    /**
     * Speichert SystemObject, DataDescription und ArchiveDataKind. Dieses Objekt wird beim erzeugen des Containers
     * persistent gespeichert.
     */
    private IdContainerFileDir _dataContainerIdentification;

    // Im folgenden wird die genaue Position jedes Eintrages, der sich im Header befindet, beschrieben.
    // Die Position setzt sich aus der Position des Vorgängers und der Länge der eigenen Daten zusammen.
    // Dadurch ist das nachträgliche einfügen neuer Daten oder das ändern bestehnder Datenformate
    // problemlos möglich (im vergleich zu abseluten Positionsangaben)

    /**
     * Diese Variable gibt an, ab welcher Position die Datensätze in der Datei beginnen, die einen Container speichert. Der
     * Wert kann nicht sofort berechnet werden, da die DataContainerIdentification variable ist.
     */
    private long _dataOffset;

    /**
     * Wieviele Verzeichnisse soll es pro Ebene geben
     */
    private static final int NUMBER_OF_DIRECTORIES = 100;
    /**
     * Wieviele Ebene soll es geben
     */
    private static final int NUMBER_OF_LAYERS = 8;
    /**
     * Wieviele Container sollen in den Verzeichnissen der letzten Ebene abgelegt werden. In den Ebene über der letzten
     * Ebene liegen keine Container. Es können insgesamt (NUMBER_OF_DIRECTORIES ^ NUMBER_OF_LAYERS) * NUMBER_OF_CONTAINERS
     * viele Container gespeichert werden.
     */
    private static final int NUMBER_OF_CONTAINERS = 1000;

    public static final String FILE_ENDING = ".arsaw";

    /**
     * DebugLogger für Debug-Ausgaben
     */
    private static final Debug _debug = Debug.getLogger();

    /**
     * Dieser Konstruktor erzeugt aus einer Datei ein Container-Objekt.
     *
     * @param containerId Eindeutige Identifikation des Containers
     * @param rootFolder  Wurzelverzeichnis aller Container
     * @throws IOException              Es ist ein Fehler beim Zugriff auf die Datei aufgetreten, die den Container
     *                                  physisch abbildet
     * @throws IllegalStateException    Der Container wurde mit einem K2S Modul erzeugt, konnte aber nicht zurückgelesen
     *                                  werden, da die Version des Containers unbekannt ist
     * @throws IllegalArgumentException Der angegebene Container kann nicht gelesen werden, da er nicht mit einem K2S Modul
     *                                  erzeugt wurde
     */
    public Container(long containerId, Path rootFolder) throws IOException, IllegalStateException, IllegalArgumentException {

        _containerId = containerId;

        // Datei und Verzeichnis, in dem sich der gespeicherte Container befinden muss
        _containerFile = getContainerDirectory(containerId, rootFolder).resolve(_containerId + FILE_ENDING);

        loadContainerTypeA();
    }

    /**
     * Diese Methode lädt einen Container aus einer Datei (Speichermedium Typ A). Das Verzeichnis und der Dateiname sind
     * bereits am Objekt gespeichert worden.
     *
     * @throws IOException           Fehler beim Zugriff auf die Datei, in dem der Container gespeichert wurde
     * @throws IllegalStateException Der Container ist zwar physisch vorhanden, seine Daten können aber nicht mehr
     *                               wiederhergestellt werden.
     */
    private void loadContainerTypeA() throws IllegalStateException, IOException {

        // Es wird nur lesend auf die Datei zugegriffen werden

        // Try/Finally zum schliessen der Datei
        try (RandomAccessFile randomAccessFile = new RandomAccessFile(_containerFile.toFile(), "r")) {
            // Ist der geöffnete Container von K2S ?
            if (_k2sWriting.equals(randomAccessFile.readUTF())) {
                // Der Container ist von K2S, Objekt erzeugen

                // Welcher Version hat der Container
                int containerVersion = randomAccessFile.readInt();

                if (containerVersion == 1) {
                    // In einem Container vom Typ 1, ist der erste Wert nach der Container Version die Position des Headers

                    // Diese Variable gibt an, ab welcher Position der Header gespeichert ist.
                    long headerOffset = randomAccessFile.readLong();
                    // Wo stehen die Datensätze
                    _dataOffset = randomAccessFile.readLong();

                    // Den filePointer auf die Header-Position setzen
                    randomAccessFile.seek(headerOffset);

                    // Die ContainerId muss nicht geladen werden, da diese bekannt ist
                    // TBD Das ist eine Debug abfrage, ob auch wirklich der richtige Container erwischt wurde
                    final long containerIdFile = randomAccessFile.readLong();
                    if (containerIdFile != _containerId) {
                        _debug.error("Es wurde der falsche Container aus einer Datei eingelesen. Datei: " + _containerFile + " erwartete ContainerId: " + _containerId + " ContainerId in der Datei: " + containerIdFile);
                    }
                    assert containerIdFile == _containerId : "Datei: " + _containerFile + " erwartete ContainerId: " + _containerId + " ContainerId in der Datei: " + containerIdFile;

                    _dataCount = randomAccessFile.readInt();
                    _deleteTime = randomAccessFile.readLong();
                    _containerSize = randomAccessFile.length();

                    final long dataTimeMax = randomAccessFile.readLong();
                    final long archiveTimeMax = randomAccessFile.readLong();
                    final long dataIndexMax = randomAccessFile.readLong();

                    _maximumDataTiming = new DataTiming(dataTimeMax, archiveTimeMax, dataIndexMax);

                    final long dataTimeMin = randomAccessFile.readLong();
                    final long archiveTimeMin = randomAccessFile.readLong();
                    final long dataIndexMin = randomAccessFile.readLong();

                    _minimumDataTiming = new DataTiming(dataTimeMin, archiveTimeMin, dataIndexMin);

                    _containerFinalized = randomAccessFile.readBoolean();

                    _storeContainer = randomAccessFile.readBoolean();

                    _labelTypeB = randomAccessFile.readInt();

                    _serializerVersion = randomAccessFile.readInt();

                    final int sizeOfArray = randomAccessFile.readInt();

                    byte[] byteArray = new byte[sizeOfArray];
                    // Den Wert einlesen und zurück geben
                    randomAccessFile.readFully(byteArray);

                    // Datenstrom für den Serialisierer bereitstellen
                    final InputStream in = new ByteArrayInputStream(byteArray);

                    Deserializer deserializer;

                    try {
                        deserializer = SerializingFactory.createDeserializer(2, in);
                    } catch (NoSuchVersionException e) {
                        _debug.error("Der Header des Containers: " + _containerId + " konnte nicht zurückgelesen werden, da die" +
                                " Version des Serialisieres (Version 2) unbekannt war.", e);
                        throw new IOException(e);
                    }

                    // Ein Objekt der Klasse DataContainerIdentification erzeugen

                    final long systemObject = deserializer.readLong();
                    // DataDescription erzeugen
                    final long attributeGroup = deserializer.readLong();
                    final long aspect = deserializer.readLong();
                    final short simulationVariant = deserializer.readShort();

                    final IdDataIdentification dataDescription = new IdDataIdentification(systemObject, attributeGroup, aspect, simulationVariant);

                    // ArchiveDataKind erzeugen
                    final ArchiveDataKind archiveDataKind = ArchiveDataKind.getInstance(deserializer.readInt());

                    _dataContainerIdentification = dataDescription.resolve(archiveDataKind);

                    // Wo beginnen die Datensätze
                    _dataOffset = randomAccessFile.getFilePointer();

                    // An dieser Stelle wurden alle Datensätze gezählt, sind 0 Datensätze im Container wird eine
                    // IOException geworfen.
                    // Dieser Fall tritt auf, wenn der Puffer des Containers nicht geschrieben werden konnte und
                    // das Archivsystem beendet wurde (Stromausfall, Absturz, usw)
                    if (_dataCount == 0) {
                        throw new IOException("Der Container mit der Id: " + _containerId + " enthält keine Datensätze, der Puffer konnte anscheinend vor Beendung des Archivsystems nicht gespeichert werden.");
                    }

                } else {
                    throw new IllegalStateException("Die Container-Version ist unbekannt, obwohl der Container mit einem K2S Modul erzeugt wurde. Containerpfad:" + _containerFile);
                }
            } else {
                throw new IllegalStateException("Der angegebene Container ist nicht mit einem Modul der Firma K2S erzeugt worden und kann somit nicht eingelesen werden. Containerpfad: " + _containerFile);
            }
        }
    }

    /**
     * Diese Methode gibt die ID des Containers zurück.
     *
     * @return ContainerID
     */
    public long getContainerId() {
        return _containerId;
    }

    /**
     * Diese Methode gibt ein Objekt mit den Eigenschaften des Containers zurück.
     *
     * @return Containereigenschaften
     */
    public IdContainerFileDir getContainerIdentification() {
        return _dataContainerIdentification;
    }

    @Override
    public DataContainerIdentification getDataContainerIdentification() {
        throw new UnsupportedOperationException();
    }

    /**
     * Diese Methode liefert die Version des Serialisieres mit dem alle Datensätze des Containers serialisiert wurden.
     *
     * @return Version des benutzten Serialisieres
     */
    public int getSerializerVersion() {
        return _serializerVersion;
    }

    @Override
    public void storeData(DataTiming dataTiming, DataState dataState, byte[] dataBytes, boolean flush) {

    }

    @Override
    public void finalizeAllData() {

    }

    /**
     * Diese Methode gibt die kleinsten Zeitstempel und den kleinsten Datenindex des Containers zurück.
     *
     * @return Zeitstempel und Index
     */
    public DataTiming getMinimumDataTiming() {
        synchronized (this) {
            return _minimumDataTiming;
        }
    }

    /**
     * Diese Methode gibt die größten Zeitstempel und den größten Datenindex des Containers zurück.
     *
     * @return Zeitstempel und Index
     */
    public DataTiming getMaximumDataTiming() {
        synchronized (this) {
            return _maximumDataTiming;
        }
    }

    /**
     * Diese Methode gibt die Anzahl der Datensätze, die im Container gepsichert sind, zurück.
     *
     * @return Anzahl Datensätze im Container
     */
    public int getDataCount() {
        synchronized (this) {
            return _dataCount;
        }
    }

    /**
     * Diese Methode gibt den Speicherbedarf des Containers an, wenn er jetzt mit finalizeAllData abgeschlossen werden
     * würde.
     *
     * @return Speicherberdarf des abgeschlossenen Containers
     */
    public long getStorageSize() {
        synchronized (this) {
            return _containerSize;
        }
    }

    /**
     * Siehe "return"
     *
     * @return true = Der Container soll der Sicherung übergeben werden, false = Der Container soll nicht der Sicherung
     * übergeben werden
     */
    public boolean getSaveStatus() {
        synchronized (this) {
            return _storeContainer;
        }
    }

    /**
     * Diese Methode gibt darüber Auskunft, ob der Container bereits auf einem Speichermedium TypB gespeichert wurde.
     *
     * @return true = Der Container wurde von der Sicherung auf einem Speichermedium Typ B gespeichert; false = Der
     * Container wurde noch nicht von der Sicherung auf einem Speichermedium Typ B gesichert
     */
    public boolean containerSavedTypeB() {
        return false;
    }

    /**
     * Diese Methode gibt den Vorhaltezeitraum des Containers zurück.
     *
     * @return Vorhaltezeitraum
     */
    public long getDeleteDate() {
        synchronized (this) {
            return _deleteTime;
        }
    }

    /**
     * Diese Methode gibt die eindeutige Identifikation des Speichermediums vom Typ B zurück, auf dem der Container
     * gesichert wurde.
     *
     * @return eindeutige Identifikation des Speichermediums vom Typ B
     * @throws IllegalStateException Der Container wurde noch gar nicht der Sicherung übergeben
     */
    public int getVolumeIdTypeB() throws IllegalStateException {
        if (_labelTypeB >= 0) {
            return _labelTypeB;
        } else {
            throw new IllegalStateException("Der Container " + _containerId + " wurde noch nicht der Sicherung übergeben.");
        }
    }

    /**
     * Diese Methode übergibt den Container der Sicherung, diese wird den Container persistent auf ein Speichermedium Typ B
     * ablegen.
     *
     * @return Eindeutige Identifizierung des Speichermedium Typ B, auf dem die Sicherung den Container gesichert wurde
     */
    public int saveAllData() {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean deleteAllData() {
        throw new UnsupportedOperationException();
    }

    @Override
    public void loadAllData(int mediaName) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void flush() {
        throw new UnsupportedOperationException();
    }

    /**
     * Diese Methode liefert den Zustand des Containers (abgeschlossen, ja/nein)
     *
     * @return true = Container ist abgeschlossen, false = Container nicht abgeschlossen
     */
    public boolean isContainerFinalized() {
        synchronized (this) {
            return _containerFinalized;
        }
    }

    /**
     * Diese Methode berechnet das Verzeichnis, in dem der Container abgelegt werden muss.
     *
     * @param containerID Eindeutige Identifizierung des Containers, aus ihm wird der Dateipfad berechnet
     * @param rootFolder  Grundverzeichnis, unter dem die Struktur aufgebaut werden soll
     * @return Dateipfad unter dem der Container gespeichert werden soll
     */
    private Path getContainerDirectory(long containerID, Path rootFolder) {
        long id = containerID;
        // In dieser Variablen wird der Pfad zusammen gebaut
        Path path = rootFolder;
        StringBuilder containerPath = new StringBuilder();

        for (int layer = 0; layer < NUMBER_OF_LAYERS; layer++) {

            // Wieviele Container sind pro Verzeichnis gespeichert?
            double containersPerDirectorylDouble = ((Math.pow(NUMBER_OF_DIRECTORIES, (NUMBER_OF_LAYERS - layer)) * NUMBER_OF_CONTAINERS) / NUMBER_OF_DIRECTORIES);
            long containersThisLevel = (long) containersPerDirectorylDouble;

            // Nun kann berechnet werden, in welchem Verzeichnis der Container liegen muss.
            long helpVariableDouble = id / containersThisLevel;
            long lowerIndex = helpVariableDouble * containersThisLevel;
            long upperIndex = ((helpVariableDouble + 1) * containersThisLevel) - 1;
            containerPath.append(lowerIndex);
            containerPath.append('-');
            containerPath.append(upperIndex);

            path = path.resolve(containerPath.toString());
            containerPath = new StringBuilder();

            id = id - (containersThisLevel * helpVariableDouble);
        }

        return path;
    }

    /**
     * Diese Methode stellt ein Objekt zur Verfügung mit dem einzelne Datensätze aus einem Container angefordert werden
     * können. Sobald keine Datensätze mehr zur Verfügung stehen, wird <code>null</code> zurück gegeben.
     *
     * @return Ein Objekt, das einzelne Datensätze aus einem Container zurückgeben kann
     */
    public ContainerDataSupplier getSupplier() throws IOException {
        return new ContainerDataSupplier();
    }

    /**
     * Diese innerClass ermöglicht den Zugriff auf Datensätze, die in der umgebenden Klasse (Container) gespeichert sind.
     * Beim erzeugen der Klasse wird angegeben, in welchem Bereich die Datensätze liegen müssen.
     */
    public final class ContainerDataSupplier implements AutoCloseable {

        private final DataInputStream file;

        /**
         * Legt ein Objekt an und ermöglicht den Zugriff mittels <code>fetchNextData</code> auf einzelne Datensätze.
         *
         * @throws IOException IO-Fehler
         */
        public ContainerDataSupplier() throws IOException {
            file = new DataInputStream(new BufferedInputStream(Files.newInputStream(_containerFile)));
            file.skipNBytes(_dataOffset);
        }

        /**
         * Liest einen Datensatz aus einem Container oder gibt ein null-Paket zurück, wenn kein Datensatz mehr vorhanden ist.
         *
         * @return Datensatz mit zusätzlichen Informationen
         * @throws IOException IO-Fehler
         */
        public PersistentData fetchNextData() throws IOException {

            PersistentData persistentData;

            try {
                // Den filePointer auf den Anfang des Datensatzes legen
                final long dataTime = file.readLong();
                final long archiveTime = file.readLong();
                final long dataIndex = file.readLong();

                if (dataTime == 0 && archiveTime == 0 && dataIndex == 0) {
                    return null;
                }

                final DataTiming dataTiming = new DataTiming(dataTime, archiveTime, dataIndex);

                final DataState dataState = DataState.getInstance(file.readInt());

                final ArchiveDataCompression archiveDataCompression = ArchiveDataCompression.getInstance(file.readByte());

                byte[] dataBytes = new byte[file.readInt()];

                if (dataState == DataState.DATA) {
                    // Nur in einem so gekennzeichneten Datensatz stehen wirklich Daten
                    if (dataBytes.length > 0) {
                        file.readFully(dataBytes);
                    } else {
                        dataBytes = null;
                    }
                } else {
                    dataBytes = null;
                }
                // Der Datensatz und seine Zusatzinformationen wurden eingelesen
                PersistentContainerData persistentContainerData = new PersistentContainerData(dataTiming, dataState, archiveDataCompression, dataBytes);
                persistentData = new PersistentData(Container.this, persistentContainerData);

            } catch (RuntimeException e) {
                _debug.error("Fehler beim Lesen des Containers \"" + _containerFile + "\". Der Rest des Containers wird übersprungen", e);
                return null; // Diesen Container ignorieren, kein lesbarer Datensatz mehr hier vorhanden
            }
            return persistentData;
        }

        @Override
        public void close() throws Exception {
            file.close();
        }
    }

}
