/*
 * Copyright 2005 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2020-2023 by DTV-Verkehrsconsult, Aachen
 *
 * ALL RIGHTS RESERVED.
 *
 * THIS  SOFTWARE  IS  PROVIDED  "AS IS"  AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES,  INCLUDING,  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF  MERCHANTABILITY  AND  FITNESS  FOR  A  PARTICULAR  PURPOSE  ARE
 * DISCLAIMED. IN NO EVENT SHALL DTV-VERKEHRSCONSULT GmbH, OR ITS CON-
 * TRIBUTORS BE LIABLE FOR ANY  DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY,  OR CONSEQUENTIAL  DAMAGES  (INCL.,  BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;  LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCL.
 * NEGLIGENCE  OR OTHERWISE)  ARISING  IN  ANY  WAY OUT  OF THE USE OF
 * THIS  SOFTWARE,  EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.persistence;

import de.bsvrz.dav.daf.main.impl.archive.PersistentContainerStreamSupplier;
import de.bsvrz.sys.funclib.debug.Debug;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.stream.Stream;

/**
 * Diese Klasse stellt aller Container, die sich im direkten Zugriff der Persistenz befinden, zur Verfügung. Die
 * Container werden dabei nicht als Liste zurückgegeben, sondern können einzeln mit der Methode {@link
 * PersistentContainerSupplier#fetchNextContainer()} angefordert werden. Die Container werden dabei streng monoton
 * steigend nach ihrer ContainerId zurückgegeben.
 *
 * @author Kappich+Kniß Systemberatung Aachen (K2S)
 * @author Achim Wullenkord (AW)
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class PersistentContainerSupplier implements PersistentContainerStreamSupplier {

    /**
     * In der Openliste werden alle Elemente gespeichert, die noch betrachtet werden müssen. Dieser Schritt ist nötig, da
     * eine rekursive Bearbeitung nicht möglich ist, also müssen die Zwischenergebnisse zwischengespeichert werden.
     */
    private final LinkedList<Path> _openList = new LinkedList<>();

    /**
     * Ober ein Objekt dieser Klasse kann ein Container mit seiner Id eingeladen werden.
     */
    private final FileSystemPersistenceModule _containerLoader;

    /**
     * Speichert die zuletzt benutze ContainerId
     */
    private long _lastContainerId = -1;

    /**
     * DebugLogger für Debug-Ausgaben
     */
    private static final Debug _debug = Debug.getLogger();

    /**
     * Wurde cancel aufegrufen, wird diese Variable true.
     * Ein erneuter fetch-Aufruf wird dann zu einer IllegalState führen.
     */
    private boolean _cancel = false;


    /**
     * @param rootFolder      Verzeichnis, in dem die Container gespeichert sind
     * @param containerLoader Objekt mit dem Container von der Platte eingeladen werden können
     */
    public PersistentContainerSupplier(Path rootFolder, FileSystemPersistenceModule containerLoader) {
        // Das erste Element in der Liste speichern
        _openList.add(rootFolder);
        _containerLoader = containerLoader;
    }

    /**
     * Diese Methode liefert jeden Container der Persistenz einzeln zurück.
     *
     * @return Container, der sich im direkten Zugriff der Persistenz befindet. Sobald alle Container zurückgegeben wurden
     * wird <code>null</code> zurückgegeben. Die Container werden streng monoton steigend nach ihrer ContainerId
     * zurückgegeben
     * @throws IllegalStateException Es wurde cancel aufgerufen und danach ein fetch
     */
    public Container fetchNextContainer() throws IOException {
        // Allgemeine Anmerkung für den Entwickler:
        // Diese Methode wird als Tiefensuche implementiert, da eine Breitensuche zuviel Speicher verbrauchen würde
        // (auf der letzten Ebene müßten ja alle Ebenen drüber gespeichert werden ((100^8) * 1000) Elemente, bei der
        // Tiefensuche sind es nur (8*99 + 1000) Elemente (ca. 2000 Elemente)). Beide Suchen kommen allerdings zum gleichen
        // Ergbnis bei gleicher Laufzeit.

        synchronized (_openList) {

            if (_cancel) {
                // Es wurde cancel Aufgerufen, also darf fetch nicht mehr aufgerufen werden
                throw new IllegalStateException("Es wurde bereits cancel aufgerufen ");
            }

            while (!_openList.isEmpty()) {

                // Das ist die Tiefensuche, der erste Knoten wird expandiert
                Path currentNode = _openList.removeFirst();

                if (currentNode.getFileName().toString().endsWith(Container.FILE_ENDING)) {
                    // Es wurde eine Datei gefunden, diese muss nicht weiter expandiert werden, sondern ist ein
                    // Ergebnis der Suche. Der Container muss nur noch geladen werden
                    return loadContainer(currentNode);
                } else {
                    Comparator<Path> fileComparator = (file1, file2) -> {
                        String a = file1.getFileName().toString();
                        String b = file2.getFileName().toString();
                        if (a.endsWith(Container.FILE_ENDING) && b.endsWith(Container.FILE_ENDING)) {
                            // Es sind Dateien, also muss die ContainerId ausgelesen werden
                            final Long ContainerId1Long1 = Long.parseLong(a.split("[.]")[0]);
                            final Long ContainerId1Long2 = Long.parseLong(b.split("[.]")[0]);
                            return ContainerId1Long1.compareTo(ContainerId1Long2);
                        } else if (a.contains("-") && b.contains("-")) {
                            // Es sind keine Dateien, sondern Verzeichnisse.
                            // Dies haben die Form XXXX-YYYY wobei XXXX < YYYY.
                            // Für die Sortierung reichen die Werte XXXX.
                            final Long ContainerId1Long1 = Long.parseLong(a.split("-")[0]);
                            final Long ContainerId1Long2 = Long.parseLong(b.split("-")[0]);
                            return ContainerId1Long1.compareTo(ContainerId1Long2);
                        } else {
                            return a.contains("-") ? -1 : 1;
                        }
                    };
                    try (Stream<Path> pathStream = Files.list(currentNode)) {
                        pathStream.filter(this::isArchiveFileOrDirectory).sorted(fileComparator.reversed()).forEachOrdered(
                                _openList::addFirst
                        );
                    }
                }
            }
            // Die Openliste ist abgearbeitet somit gibt es keine Container mehr
            return null;
        }
    }

    private boolean isArchiveFileOrDirectory(Path it) {
        if (Files.isRegularFile(it) && it.getFileName().toString().endsWith(Container.FILE_ENDING)) {
            // Containerdatei
            return true;
        }
        if (it.getFileName().toString().matches("\\d+-\\d+")) {
            // Container-Verzeichnis
            return true;
        }
        _debug.warning("Datei " + it + " ist keine bekannte Archivdatei und wird ignoriert.");
        return false;
    }

    /**
     * Löscht alle angelegten Datenstrukturen, die zur Bearbeitung benötigt wurden.
     */
    public void cancel() {
        synchronized (_openList) {
            _openList.clear();
            _cancel = true;
        }
    }

    /**
     * Diese Methode lädt einen Container aus einer Datei und stellt das entsprechende Objekt zur Verfügung.
     *
     * @param containerFile Pfad des Containers
     * @return Containerobjekt
     * @throws IOException Die Datei konnte nicht geöffnet werden, eine entsprechende Fehlermeldung wird erzeugt
     */
    private Container loadContainer(Path containerFile) throws IOException {
        final long containerId = Long.parseLong(containerFile.getFileName().toString().split("[.]")[0]);
        assert containerId > _lastContainerId : "Neue Container ID " + containerId + " letzte ContainerID " + _lastContainerId;

        _lastContainerId = containerId;
        try {
            return _containerLoader.getContainer(containerId);
        } catch (IOException e) {
            throw new IOException("Die Datei " + containerFile + " konnte nicht geöffnet werden.", e);
        }
    }
}
