/*
 *
 * Copyright 2018-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.migration.
 *
 * de.bsvrz.ars.migration is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.migration is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.migration.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.backup;

import de.bsvrz.ars.ars.persistence.ContainerFile;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;

import java.io.IOException;
import java.io.OutputStream;

/**
 * Klasse, die dem {@link BackupModule} bzw. genauer der {@link BackupImplementation} übergeben wird,
 * wenn ein Container gesichert wird. Diese Klasse bietet Möglichkeiten, den Container zu lesen sowie verschiedene
 * Werte abzufragen
 *
 * @author Kappich Systemberatung
 */
public interface Container {

	/**
	 * Sichert den Container
	 *
	 * @param stream   Der Inhalt des Containers wird in diesen {@link OutputStream} geschrieben, der vom Backup-Modul bereitgestellt werden muss.
	 *                 Beispielsweise ein {@link java.io.FileOutputStream}.
	 * @param mediumId Beim Übertragen des Containers in den {@link OutputStream} wird automatisch on-the-fly
	 *                 der Header des Containers angepasst, sodass dort die hier angegebene Medien-ID eingetragen wird.
	 *                 Es muss sich also um die Medien-ID handelt, auf der der Container gesichert wird.
	 * @throws IOException Eine möglicherweise auftretende IO-Exception beim Schreiben in den {@link OutputStream} oder aufgrund von anderen Problemen wird
	 *                     hiermit an den Aufrufer, also das Backup-Modul weitergegeben. Dieses braucht die Fehlermeldung in der Regel nicht zu fangen.
	 */
	void backup(OutputStream stream, int mediumId) throws IOException;

	/**
	 * Gibt die Archivdatenart des Containers zurück
	 *
	 * @return die Archivdatenart des Containers
	 */
	ArchiveDataKind getArchiveDataKind();

	/**
	 * Gibt die Datenidentifikation des Containers zurück
	 *
	 * @return die Datenidentifikation des Containers
	 */
	IdDataIdentification getDataIdentification();

	/**
	 * Gibt die ID zurück. Die ID bildet den Dateinamen des Containers.
	 *
	 * @return die ID
	 */
	long getContainerId();

	/**
	 * Gibt die Dateigröße des Containers zurück. In besonderen Ausnahmefällen kann sich die Byte-Größe hierbei geringfügig
	 * von der Datenlänge unterscheiden, die bei {@link #backup(OutputStream, int)} in den {@link OutputStream} geschrieben wird,
	 * da dabei der Header angepasst wird.
	 *
	 * @return die Dateigröße des Containers
	 */
	long getFileSize();

	/**
	 * Gibt den Dateinamen des Containers zurück
	 *
	 * @return den Dateinamen des Containers
	 */
	default String getFileName() {
		return ContainerFile.getContainerFileName(getContainerId());
	}

	/**
	 * Gibt das letzte Änderungsdatum zurück
	 *
	 * @return das letzte Änderungsdatum der Container-Datei, manche Backup-Module erlauben es, diese information zu sichern.
	 */
	long lastModified();
}
