/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.migration.
 *
 * de.bsvrz.ars.migration is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.migration is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.migration.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.migration;

import com.google.common.collect.Range;
import de.bsvrz.ars.ars.backup.BackupException;
import de.bsvrz.ars.ars.backup.BackupImplementation;
import de.bsvrz.ars.ars.backup.BackupInitializer;
import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SyncKey;
import de.bsvrz.ars.ars.persistence.DataIdentificationManager;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.ars.ars.persistence.PersistenceException;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.ReadonlyPersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.mgmt.lock.DirectoryIsLockedException;
import de.bsvrz.ars.ars.persistence.directories.mgmt.lock.LockFileManager;
import de.bsvrz.ars.ars.persistence.directories.mgmt.range.WeekDomain;
import de.bsvrz.ars.ars.persistence.layout.ClassicPersistenceDirectoryLayout;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKindCombination;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.debug.Debug;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Instant;
import java.util.function.Consumer;

/**
 * Hauptklasse für das Migrationstool. Diese Klasse liest Aufrufargumente und startet dann die eigentliche Migration, die im {@link MigrateWorker}
 * implementiert ist.
 */
public class Migrate {

	/**
	 * Main-methode, die beim Start der Migration ausgeführt wird.
	 *
	 * @param args Aufrufargumente
	 */
	public static void main(String[] args) {
		try {
			runWithArgs(args);
		} catch (IllegalArgumentException e) {
			printHelp();
			System.err.println();
			e.printStackTrace(System.err);
			System.exit(1);
		} catch (BackupException e) {
			System.err.println("Das angegebene Backup-Modul konnte nicht geladen werden.");
			e.printStackTrace(System.err);
			System.exit(1);
		} catch (PersistenceException e) {
			System.err.println("Bei der Migration trat ein Fehler in der Persistenzschicht auf:");
			e.printStackTrace(System.err);
			System.exit(1);
		} catch (Exception e) {
			System.err.println("Bei der Migration trat ein Fehler auf:");
			e.printStackTrace(System.err);
			System.exit(1);
		}
	}

	/**
	 * Diese methode entspricht der Main-methode, jedoch ohne Fehlerbehandlung.
	 *
	 * @param args Argumente
	 * @throws BackupException      Fehler beim Zugriff auf Backup-Modul
	 * @throws PersistenceException Fehler in Persistenzschicht
	 * @throws IOException          IO_Fehler (z. B. Probleme beim Erstellen/Löschen von Lockdateien)
	 */
	public static void runWithArgs(String[] args) throws BackupException, PersistenceException, IOException {
		ArgumentList argumentList = new ArgumentList(args);
		Path src = argumentList.fetchArgument("-src").asPath();
		Path dest = argumentList.fetchArgument("-dst").asPath();
		Instant from = argumentList.hasArgument("-from") ? argumentList.fetchArgument("-from").asInstant() : Instant.MIN;
		Instant to = argumentList.hasArgument("-to") ? argumentList.fetchArgument("-to").asInstant() : Instant.MAX;
		final String backupProps = argumentList.hasArgument("-backup") ? argumentList.fetchArgument("-backup").asString() : null;
		int threads = argumentList.fetchArgument("-threads=4").intValue();

		printArguments(src, dest, from, to, backupProps, threads);

		Debug.init("Migration", argumentList);
		argumentList.ensureAllArgumentsUsed();

		BackupImplementation backupModul = BackupInitializer.getBackupModul(backupProps);

		if (!Files.isDirectory(src)) {
			throw new PersistenceException("Quellverzeichnis existiert nicht.");
		}

		if (Files.isDirectory(dest) && Files.isSameFile(src, dest)) {
			throw new PersistenceException("Quelle und Ziel müssen unterschiedliche Verzeichnisse sein.");
		}

		WeekDomain domain = ArchiveManager.detectDomain(src, true);
		if (domain != null) {
			throw new PersistenceException("Verzeichnis " + src.toAbsolutePath() + " ist leer oder wurde bereits konvertiert.");
		}

		ClassicPersistenceDirectoryLayout layout = ClassicPersistenceDirectoryLayout.Instance;

		// Datenidentifikationsverwaltung
		var dataIdentificationManager = new DataIdentificationManager() {
			private Consumer<MigrationKey> closeIndexCallback = key -> {
			};

			@Override
			public SyncKey<IdDataIdentification> lockIndex(IdDataIdentification dataIdentification) {
				return new MigrationKey(dataIdentification, closeIndexCallback);
			}
		};

		PersistenceDirectory srcDir = new ReadonlyPersistenceDirectory(dataIdentificationManager, layout.createInstance(src, 0));

		dataIdentificationManager.closeIndexCallback = (key) -> srcDir.getIndexTree().closeIndexes();

		LockFileManager lockFileManager = new LockFileManager();
		try {
			lockFileManager.open(src);
		} catch (DirectoryIsLockedException e) {
			throw new PersistenceException("Verzeichnis " + src.toAbsolutePath() + " ist gesperrt.", e);
		}
		try {
			MigrateWorker worker = new MigrateWorker(srcDir, dest, threads, Range.closedOpen(from, to), backupModul);
			worker.start();
		} finally {
			lockFileManager.close(src);
		}
	}

	private static void printHelp() {
		System.err.printf("""
				Das Migrationswerkzeug besitzt folgende Argumente:
				     -src=       Erforderlich.
				                 Quellverzeichnis (altes Persistenzverzeichnis)
				                 
				     -dst=       Erforderlich.
				                 Leeres Zielverzeichnis (zukünftiges Persistenzverzeichnis im neuen Format)
				                 
				     -from=      Optional. Datum in Format dd.mm.yyyy (z. B. "31.12.1987").
				                 Daten konvertieren ab Zeitstempel (Archivzeitstempel)
				                 (Frühere Daten werden nicht kopiert)
				                 
				     -to=        Optional. Datum in Format dd.mm.yyyy (z. B. "31.12.1987").
				                 Daten konvertieren bis Zeitstempel (Archivzeitstempel)
				                 (Neuere Daten werden nicht kopiert)
				                 
				     -backup=    Optional.
				                 .properties-Datei für Backup-Konfiguration, falls ausgelagerte Container
				                 bei der Migration wieder von Sicherungsmedien integriert werden sollen.
				                 Achtung: Dies ist nachträglich nicht möglich!
				                 
				     -threads=   Optional.
				                 Anzahl Threads für die Konvertierung (beeinflusst ggf. Performance)
				%n""");
	}

	private static void printArguments(Path src,
	                                   Path dest,
	                                   Instant from,
	                                   Instant to,
	                                   final String backupProps,
	                                   int threads) {
		System.err.printf("""
				Das Migrationswerkzeug wurde mit folgenden Parametern gestartet:
				     -src=%s
				                 Quellverzeichnis (altes Persistenzverzeichnis)
				     -dst=%s
				                 Leeres Zielverzeichnis (zukünftiges Persistenzverzeichnis im neuen Format)
				     -from=%s
				                 Daten konvertieren ab Zeitstempel (Archivzeitstempel)
				                 (Frühere Daten werden nicht kopiert)
				     -to=%s
				                 Daten konvertieren bis Zeitstempel (Archivzeitstempel)
				                 (Neuere Daten werden nicht kopiert)
				     -backup=%s
				                 .properties-Datei für Backup-Konfiguration, falls ausgelagerte Container
				                 bei der Migration wieder von Sicherungsmedien integriert werden sollen.
				                 Achtung: Dies ist nachträglich nicht möglich!
				     -threads=%s
				                 Anzahl Threads für die Konvertierung (beeinflusst ggf. Performance)
				%n""", src, dest, from.equals(Instant.MIN) ? "(nicht angegeben)" : from, to.equals(Instant.MAX) ? "(nicht angegeben)" : to, backupProps == null ? "(nicht angegeben)" : backupProps, threads);
	}
}
