/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.migration.
 *
 * de.bsvrz.ars.migration is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.migration is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.migration.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.migration;

import com.google.common.collect.Range;
import de.bsvrz.ars.ars.backup.BackupException;
import de.bsvrz.ars.ars.backup.BackupImplementation;
import de.bsvrz.ars.ars.mgmt.datatree.DataIdentTree;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SyncKey;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationFailedException;
import de.bsvrz.ars.ars.persistence.*;
import de.bsvrz.ars.ars.persistence.directories.ActivePersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.ReadonlyPersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.mgmt.lock.LockFileManager;
import de.bsvrz.ars.ars.persistence.directories.mgmt.range.Week;
import de.bsvrz.ars.ars.persistence.directories.mgmt.range.WeekDomain;
import de.bsvrz.ars.ars.persistence.index.IndexException;
import de.bsvrz.ars.ars.persistence.index.result.IndexResult;
import de.bsvrz.ars.ars.persistence.iter.DataIterator;
import de.bsvrz.ars.ars.persistence.layout.ShortPersistenceDirectoryLayout;
import de.bsvrz.ars.ars.persistence.walk.DataIdentificationDirWalk;
import de.bsvrz.ars.ars.persistence.walk.internal.StatusPrinter;
import de.bsvrz.ars.ars.persistence.writer.SerializableDataset;
import de.bsvrz.ars.ars.persistence.writer.SerializationHelper;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKindCombination;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.losb.datk.ContainerSettings;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Stream;

import static de.bsvrz.ars.ars.persistence.StartupProperties.STUP_LAST_ATIME;
import static de.bsvrz.ars.ars.persistence.StartupProperties.STUP_MAX_CONT_ID;

/**
 * Diese Klasse führt die eigentliche Migration durch
 */
public class MigrateWorker implements ContainerCreator {

	private static final Debug _debug = Debug.getLogger();

	private static final String UNAVAILABLE_MEDIUM_ID_FILE = "_unavailable_mediumids.dat";

	private final Set<String> mediumIdIgnoreSet = new HashSet<>();

	private final PersistenceDirectory src;
	private final Path target;
	private final BackupImplementation backupImplementation;

	private final AtomicLong containerIdCounter = new AtomicLong();
	private final AtomicLong lastATime = new AtomicLong();
	private final WeekDomain domain;
	private final ConcurrentHashMap<Path, ActivePersistenceDirectory> pathCache = new ConcurrentHashMap<>();
	private final ContainerSettings.CloseCondition closeConditions;
	private final ThreadLocal<SerializationHelper> serializationHelper;
	private final int numThreads;
	private final Range<Instant> archiveTimeRange;
	private final LockFileManager lockFileManager;

	/**
	 * Erstellt einen neuen MigrateWorker.
	 *
	 * @param srcDir               Altes Persistenzverzeichnis
	 * @param target               Zielverzeichnis
	 * @param numThreads           Anzahl Threads für Migration
	 * @param archiveTimeRange     Archivzeitbereich, der Migriert werden soll
	 * @param backupImplementation Optionale Implementierung des Backup-Moduls, falls ausgelagerte Dateien
	 *                             während der Migration wieder eingelagert werden sollen.
	 */
	public MigrateWorker(PersistenceDirectory srcDir, Path target, int numThreads, Range<Instant> archiveTimeRange, BackupImplementation backupImplementation) {
		this.src = srcDir;
		this.target = target;
		this.numThreads = numThreads;
		this.archiveTimeRange = archiveTimeRange;
		this.backupImplementation = backupImplementation;
		this.domain = new WeekDomain();
		lockFileManager = new LockFileManager();

		CacheManager.getInstance().setCacheEnabled(false);

		if (Files.exists(target) && !isEmpty(target)) {
			throw new IllegalArgumentException("Zielverzeichnis existiert bereits: " + target);
		}

		closeConditions = new ContainerSettings.CloseCondition();
		closeConditions.maxContTime = 1_000_000;
		closeConditions.maxContSize = 50_000_000;
		closeConditions.maxContAnzDS = 100_000;

		serializationHelper = ThreadLocal.withInitial(() -> new SerializationHelper(atg -> closeConditions, MigrateWorker.this, false));

		initIgnoreList();
	}

	/**
	 * Prüft, ob ein Verzeichnis leer ist
	 *
	 * @param path Verzeichnis
	 * @return true: leer, false: kein Verzeichnis oder nicht leer oder anderer Fehler
	 */
	public boolean isEmpty(Path path) {
		if (Files.isDirectory(path)) {
			try (DirectoryStream<Path> stream = Files.newDirectoryStream(path)) {
				return !stream.iterator().hasNext();
			} catch (IOException e) {
				return false;
			}
		}
		return false;
	}

	/**
	 * Führt die Migration durch.
	 *
	 * @throws PersistenceException Fehler beim Lesen oder Schreiben
	 */
	public void start() throws PersistenceException {
		migrateData();
		closeDirectories();
		writeStartupProperties();
	}

	/**
	 * Schreibt die Startup-Properties.info
	 * @throws PersistenceException
	 */
	private void writeStartupProperties() throws PersistenceException {
		StartupProperties startupProperties = new StartupProperties(target);
		startupProperties.setVal(STUP_MAX_CONT_ID, containerIdCounter.get());
		startupProperties.setVal(STUP_LAST_ATIME, lastATime.get());
	}

	/**
	 * Schließt alle Wochenverzeichnisse ab (d.h. die letzte Containerdatei wird geschlossen und die Indexe werden generiert)
	 * @throws PersistenceException
	 */
	private void closeDirectories() throws PersistenceException {
		List<ActivePersistenceDirectory> directories = new ArrayList<>(pathCache.values());
		directories.sort(Comparator.comparing(it -> domain.ofPath(target.relativize(it.getBasePath()))));
		int count = directories.size();
		_debug.info("Schließe " + count + " Verzeichnisse ab.");
		Instant start = Instant.now();
		int finished = 0;
		StatusPrinter statusPrinter = new StatusPrinter();
		for (ActivePersistenceDirectory value : directories) {
			_debug.info(statusPrinter.getStatusMessage("Verzeichnisse abschließen", Duration.between(start, Instant.now()), StatusPrinter.ApproximationType.Exact, count, finished, 0) + "\nAktuelles Verzeichnis: " + value);
			try {
				RestorePersDirTsk.RestoreWorker worker = new RestorePersDirTsk.RestoreWorker("Indexe erzeugen", RebuildMode.Full, this, numThreads);
				worker.doRestore(value);
				value.closePermanently();
				Path basePath = value.getBasePath();
				lockFileManager.close(basePath);
				deleteDirectoryIfEmpty(basePath);
			} catch (InterruptedException e) {
				_debug.warning("Unterbrochen", e);
				return;
			} catch (IOException e) {
				_debug.warning("Lock-Datei konnte nicht gelöscht werden", e);
			} finally {
				value.getIndexTree().closeIndexes();
			}
			finished++;
		}
	}

	private void migrateData() throws PersistenceException {
		_debug.info("Starte Migration von '" + src + "' nach '" + target + "'.");
		DataIdentificationDirWalk walk = DataIdentificationDirWalk.allDirectories(src);
		walk.execute("Migration", numThreads, (dataIdentificationDir, walk1) -> new DataIdentificationMigration(dataIdentificationDir).run());
	}

	/**
	 * Löscht ein Verzeichnis, wenn es leer ist und gelöscht werden kann.
	 * Fehler beim Löschen werden ignoriert.
	 *
	 * @param path Verzeichnis
	 */
	private static void deleteDirectoryIfEmpty(Path path) {
		try {
			try (Stream<Path> stream = Files.list(path)) {
				if (stream.iterator().hasNext()) {
					return;
				}
			}
			Files.deleteIfExists(path);
		} catch (IOException ignored) {
		}
	}

	/**
	 * Liest die Datei MEDIUM_ID_IGNORE_FILE aus und schreibt die darin enthaltenen MedienIds in das Set mediumIdIgnoreSet. Fehler werden großzügig behandelt -
	 * notfalls bleibt das Set leer.
	 */
	private void initIgnoreList() {
		Path ignoreFile = src.getBasePath().resolve(UNAVAILABLE_MEDIUM_ID_FILE);
		if (!Files.exists(ignoreFile)) {
			_debug.info("Datei " + UNAVAILABLE_MEDIUM_ID_FILE + " existiert nicht.");
			return;
		}
		try (BufferedReader br = Files.newBufferedReader(ignoreFile, StandardCharsets.ISO_8859_1)) {
			while (true) {
				String s = br.readLine();
				if (s != null) {
					try {
						int mediumID = Integer.parseInt(s.trim());
						mediumIdIgnoreSet.add(String.valueOf(mediumID));
					} catch (NumberFormatException ignored) {
						// do nothing
					}
				} else {
					break;    //Dateiende erreicht
				}
			}
		} catch (IOException e) {
			_debug.warning(
					"Exception trat beim Lesen der Datei " + UNAVAILABLE_MEDIUM_ID_FILE + " auf. Wiederherstellung verwendet keine MedienID-Ignore-Liste.", e
			);
		}
	}

	/**
	 * Gibt zurück, ob eine Containerdatei mit dem angebenden Medium-Header-Wert wiederhergestellt werden soll.
	 *
	 * @param mediumID Medium-ID
	 * @return true: soll wiederhergestellt werden, false sonst.
	 */
	private boolean isMediumIdToBeRestored(String mediumID) {
		return !mediumID.equals(ContainerHdr.MEDIUM_DESTROYED) && !mediumID.equals(ContainerHdr.MEDIUM_UNSAVED) && !mediumIdIgnoreSet.contains(mediumID);
	}

	/**
	 * Erzeugt ein Dummy-Lock-Objekt für den synchronisierten Zugriff auf Datenidentifikationen.
	 * Da kein gleichzeitiger Zugriff in der Migration stattfindet, braucht hier auch nichts gelockt zu werden.
	 *
	 * @param dataIdentificationDir Did-Verzeichnis
	 * @return Lock-Objekt
	 */
	private SyncKey<IdDataIdentification> getLock(DataIdentificationDir dataIdentificationDir) {
		return getLock(dataIdentificationDir.getDataIdentification());
	}

	/**
	 * Erzeugt ein Dummy-Lock-Objekt für den synchronisierten Zugriff auf Datenidentifikationen.
	 * Da kein gleichzeitiger Zugriff in der Migration stattfindet, braucht hier auch nichts gelockt zu werden.
	 * <p>
	 * Das Schließen der Indexdateien des aktuelle nThreads nach Zugriff ist jedoch erforderlich.
	 *
	 * @param dataIdentification Datenidentifikation
	 * @return Lock-Objekt
	 */
	private SyncKey<IdDataIdentification> getLock(IdDataIdentification dataIdentification) {
		return new MigrationKey(dataIdentification, key -> {
			for (ActivePersistenceDirectory value : pathCache.values()) {
				value.getIndexTree().closeIndexes();
			}
		});
	}

	/**
	 * Gibt die nächste freie Container-ID zurück
	 *
	 * @return Container-ID
	 */
	@Override
	public long nextContainerID() {
		return containerIdCounter.incrementAndGet();
	}

	@Override
	public SyncKey<IdDataIdentification> lockIndex(IdDataIdentification dataIdentification) {
		return getLock(dataIdentification);
	}

	@Override
	public DataIdentTree getDataIdentTree() {
		return null;
	}

	/**
	 * Worker-Klasse für die Migration einer einzelnen Datenidentifikation
	 */
	private class DataIdentificationMigration {
		private final DataIdentificationDir dataIdentificationDir;
		private ActivePersistenceDirectory prevDirectory;
		private Week prevTimeRange;

		/**
		 * Konstruktor. Es muss {@link #run()} ausgeführt werden um die Migration zu starten
		 *
		 * @param dataIdentificationDir Datenidentifikation, die migriert werden soll.
		 */
		public DataIdentificationMigration(DataIdentificationDir dataIdentificationDir) {
			this.dataIdentificationDir = dataIdentificationDir;
		}

		/**
		 * Führt die Migration aus
		 */
		public void run() throws PersistenceException, IndexException, SynchronizationFailedException, ExecutionException, InterruptedException {
			if (dataIdentificationDir.getDataIdentification().getSimVariant() != 0) {
				// Simulationen werden nicht migriert
				return;
			}

			for (ArchiveDataKind adk : dataIdentificationDir.getArchiveDataKinds()) {

				SyncKey<IdDataIdentification> lock = getLock(dataIdentificationDir);

				LockedContainerDirectory containerDirectory = new LockedContainerDirectory(lock, adk);
				IndexResult<ContainerManagementInformation> indexResult = src.getContainerHeaders(containerDirectory);
				for (int i = 0; i < indexResult.size(); i++) {
					ContainerHeaders containerHeaders = new ContainerHeaders(indexResult, lock.getElement(), adk, i);

					// Container migrieren
					handleContainer(containerDirectory, containerHeaders);
				}
				src.getIndexTree().closeIndexes();
				_debug.fine("Verzeichnis abgeschlossen: " + containerDirectory);
				dataIdentificationDir.getPersistenceDirectory().getIndexTree().closeIndexes(containerDirectory);
			}
		}

		private void handleContainer(LockedContainerDirectory directory, ContainerHeaders containerHeaders) throws IndexException, SynchronizationFailedException, PersistenceException, ExecutionException, InterruptedException {
			ContainerDataResult tmp = new ContainerDataResult();

			long contMinArcTime = containerHeaders.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_ARC_TIME_MIN);
			long contMaxArcTime = containerHeaders.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_ARC_TIME_MAX);

			Range<Instant> containerRange = Range.closed(Instant.ofEpochMilli(contMinArcTime), Instant.ofEpochMilli(contMaxArcTime));

			long containerId = containerHeaders.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_CONT_ID);
			boolean isDeleted = containerHeaders.getContainerHeaderParamAsBoolean(ContainerManagementInformation.CHP_DELETED);

			if (!archiveTimeRange.isConnected(containerRange)) {
				_debug.fine("Daten von Container " + containerId + " sind nicht im Zeitbereich.");
			}

			if (isDeleted) {
				String mediumId = containerHeaders.getContainerHeaderParamAsString(ContainerManagementInformation.CHP_MEDIUM_ID);
				if (isMediumIdToBeRestored(mediumId)) {
					restoreFromMedium(directory, containerHeaders, mediumId, containerId, contMaxArcTime, tmp);
				} else {
					writeUnavailableBlock(containerHeaders, updateOutDir(contMaxArcTime, dataIdentificationDir));
				}
				return;
			}
			try (DataIterator iterator = src.iterator(directory.archiveDataKind(), containerId, directory.dataIdentification())) {
				while (!iterator.isEmpty()) {
					iterator.poll(tmp);
					if (!archiveTimeRange.contains(Instant.ofEpochMilli(tmp.getArchiveTime()))) {
						// Datensatz wird ignoriert, nicht im Zeitbereich
						continue;
					}
					writeToOutput(tmp, directory, updateOutDir(tmp.getArchiveTime(), dataIdentificationDir));
				}
			}
		}

		private void restoreFromMedium(LockedContainerDirectory directory, ContainerHeaders containerHeaders, String mediumId, long containerId, long contMaxArcTime, ContainerDataResult tmp) throws PersistenceException, IndexException, SynchronizationFailedException, ExecutionException, InterruptedException {
			Path tempDirectory = null;
			try {
				int backupMediumID = Integer.parseInt(mediumId.trim());
				tempDirectory = Files.createTempDirectory("restore-" + backupMediumID);
				try (ContainerFileHandle containerFileHandle = restoreFromBackup(backupMediumID, directory, containerId, tempDirectory)) {
					if (containerFileHandle == null) {
						writeUnavailableBlock(containerHeaders, updateOutDir(contMaxArcTime, dataIdentificationDir));
						return;
					}
					try (DataIterator iterator = containerFileHandle.iterator()) {
						while (!iterator.isEmpty()) {
							iterator.poll(tmp);
							if (!archiveTimeRange.contains(Instant.ofEpochMilli(tmp.getArchiveTime()))) {
								// Datensatz wird ignoriert, nicht im Zeitbereich
								return;
							}
							writeToOutput(tmp, directory, updateOutDir(tmp.getArchiveTime(), dataIdentificationDir));
						}
					}
				}
			} catch (PersistenceException e) {
				// Fehler z. B. bei Aktualisieren des Ausgabeverzeichnisses an Aufrufer weitergeben
				throw e;
			} catch (BackupException e) {
				_debug.warning("Container " + containerId + " konnte vom Medium " + mediumId + " nicht wiederhergestellt werden und wird ignoriert.", e);
				writeUnavailableBlock(containerHeaders, updateOutDir(contMaxArcTime, dataIdentificationDir));
			} catch (Exception e) {
				throw new PersistenceException("Schwerer Fehler beim Wiederherstellen von " + containerId + " vom Medium " + mediumId + ".", e);
			} finally {
				if (tempDirectory != null) {
					try {
						PersistenceManager.deletePath(tempDirectory);
					} catch (IOException e) {
						_debug.warning("Temporäres Verzeichnis " + tempDirectory + " konnte nicht gelöscht werden.", e);
					}
				}
			}
		}

		private void writeUnavailableBlock(ContainerHeaders containerHeaders, ActivePersistenceDirectory persistenceDirectory) throws IndexException, SynchronizationFailedException, PersistenceException, ExecutionException, InterruptedException {
			ContainerDataResult result = new ContainerDataResult();
			result.setArchiveTime(containerHeaders.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_ARC_TIME_MAX));
			result.setDataTime(containerHeaders.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_DATA_TIME_MAX));
			result.setDataIndex(containerHeaders.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_DATA_IDX_MAX));
			result.setContainerID(containerHeaders.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_CONT_ID));
			result.setDataUncompressedSize(0);
			result.setCompressed(false);
			result.setData(null);
			result.setDataKind(containerHeaders.getLocation().archiveDataKind());
			result.setDataState(DataState.POSSIBLE_GAP);
			writeToOutput(result, containerHeaders.getLocation(), persistenceDirectory);
		}

		private void writeToOutput(ContainerDataResult result, ContainerDirectory location, ActivePersistenceDirectory persistenceDirectory) throws IndexException, SynchronizationFailedException, PersistenceException {

			lastATime.accumulateAndGet(result.getArchiveTime(), Math::max);
			
			SerializableDataset dataset = SerializableDataset.create(result);
			serializationHelper.get().writeData(dataset, persistenceDirectory, location);
		}

		private ActivePersistenceDirectory updateOutDir(long archiveTime, DataIdentificationDir dataIdentificationDir) throws PersistenceException {
			Week timeRange = domain.ofEpochMillis(archiveTime);
			if (timeRange.equals(prevTimeRange)) {
				return prevDirectory;
			}
			if (prevDirectory != null) {
				for (ArchiveDataKind archiveDataKind : ArchiveDataKindCombination.all()) {
					LockedContainerDirectory containerDirectory = new LockedContainerDirectory(getLock(dataIdentificationDir), archiveDataKind);
					closeOpenContainer(containerDirectory);
				}
			}
			prevTimeRange = timeRange;
			Path targetDir = target.resolve(domain.getPath(timeRange));
			prevDirectory = pathCache.computeIfAbsent(targetDir, path -> createPersistenceDirectory(targetDir));
			return prevDirectory;
		}

		private void closeOpenContainer(LockedContainerDirectory containerDirectory) throws PersistenceException {
			// Hier nicht ActivePersistenceDirectory#closeOpenContainer benutzen, da die Indexdateien noch nicht erstellt werden sollen
			OpenContainerData openContainerData = prevDirectory.getLoadedContainerData(containerDirectory);

			if (openContainerData instanceof StandardOpenContainerData containerData) {
				try (var containerFileHandle = prevDirectory.accessOpenContainer(containerDirectory, containerData.getContainerId())) {
					ContainerFile cf = containerFileHandle.getContainerFile();
					cf.closeContainer(containerData);
					prevDirectory.removeOpenContainerData(containerDirectory);
				}
			}
		}
	}


	@NotNull
	private ActivePersistenceDirectory createPersistenceDirectory(Path targetDir) throws UncheckedIOException {
		try {
			lockFileManager.createWritable(targetDir);
			return new ActivePersistenceDirectory(MigrateWorker.this, ShortPersistenceDirectoryLayout.Instance.createInstance(targetDir, 0));
		} catch (IOException e) {
			throw new UncheckedIOException(e);
		}
	}


	/**
	 * Holt einen Container aus dem Backup zurück.
	 *
	 * @return Handle auf temporär wiederhergestellte Containerdatei
	 */
	private ContainerFileHandle restoreFromBackup(int backupMediumID, LockedContainerDirectory directory, long containerId, Path tempDirectory)
			throws PersistenceException, BackupException, IOException {

		if (backupImplementation == null) {
			return null;
		}
		
		// Zeit messen:
		long startTime = Util.startTimer();

		String containerFileName = ContainerFile.getContainerFileName(containerId);

		PersistenceDirectory tmpDir = new ReadonlyPersistenceDirectory(MigrateWorker.this, ShortPersistenceDirectoryLayout.Instance.createInstance(tempDirectory, 0));

		try (InputStream is = backupImplementation.restoreContainer(backupMediumID, containerFileName)) {
			if (is == null) {
				_debug.warning("Container " + containerFileName + " nicht gefunden.");
				return null;
			}

			Path targetDir = tmpDir.getPath(directory);
			Files.createDirectories(targetDir);
			Path file = targetDir.resolve(containerFileName);
			try (OutputStream fos = Files.newOutputStream(file)) {
				Util.copyStreams(is, fos);
			}
		} catch (BackupException | IOException e) {
			_debug.error("Wiederherstellung fehlgeschlagen.", e);
			throw e;
		}

		ContainerFileHandle handle = tmpDir.accessContainer(directory, ContainerFile.getContID(containerFileName));
		// Containerheader anpassen auf "Wiederhergestellt"
		handle.readContainerHeader();
		handle.setContainerHeaderParam(ContainerHdr.CHP_LOESCHUTZ, 0);
		handle.setContainerHeaderParam(ContainerHdr.CHP_RESTORED, "1");
		handle.setContainerHeaderParam(ContainerHdr.CHP_MEDIUM_ID, Util.leadZero(backupMediumID, ContainerHdr.CHP_MEDIUM_ID.getValLen()));

		handle.writeContainerHeaderWithoutUpdatingIndex();

		// Zeit messen & Log-Ausgabe:
		_debug.info("Container " + containerFileName + " erfolgreich wiederhergestellt (" + Util.stopTimer(startTime) + " sec.)");

		return handle;
	}
}
