/*
 * Copyright (c) 2016-2024 by inovat
 * innovative systeme - verkehr - tunnel - technik,
 * Dipl.-Ing. H. C. Kniss
 *
 * This file is part of de.inovat.dua.pufferlzzsnachfordern.Auftrag
 *
 * de.inovat.dua.pufferlzzsnachfordern.Auftrag is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.inovat.dua.pufferlzzsnachfordern.Auftrag is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.inovat.dua.pufferlzzsnachfordern.Auftrag.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Diese Datei ist Teil von de.inovat.dua.pufferlzzsnachfordern.Auftrag.
 *
 * de.inovat.dua.pufferlzzsnachfordern.Auftrag ist Freie Software: Sie können es unter den Bedingungen
 * der GNU General Public License, wie von der Free Software Foundation,
 * Version 3 der Lizenz oder (nach Ihrer Wahl) jeder späteren
 * veröffentlichten Version, weiterverbreiten und/oder modifizieren.
 *
 * de.inovat.dua.pufferlzzsnachfordern.Auftrag wird in der Hoffnung, dass es nützlich sein wird, aber
 * OHNE JEDE GEWÄHRLEISTUNG, bereitgestellt; sogar ohne die implizite
 * Gewährleistung der MARKTFÄHIGKEIT oder EIGNUNG FÜR EINEN BESTIMMTEN ZWECK.
 * Siehe die GNU General Public License für weitere Details.
 *
 * Sie sollten eine Kopie der GNU General Public License zusammen mit diesem
 * Programm erhalten haben. Wenn nicht, siehe <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * inovat, Dipl.-Ing. H. C. Kniss
 * An der Krautwiese 37
 * D-53783 Eitorf
 * +49 (0)2243 8464 193
 * info@inovat.de
 * www.inovat.de
 */


package de.inovat.dua.pufferlzzsnachfordern;

//~ JDK IMPORTE ===============================================================

import java.text.SimpleDateFormat;

import java.util.Date;

//~ KLASSEN ===================================================================

/**
 * Enthält alle Informationen für einen Pufferabrufauftrag.
 *
 * @author inovat, innovative systeme - verkehr - tunnel - technik
 * @author Dipl.-Ing. Hans Christian Kniß (HCK)
 */
public class Auftrag {

    /**
     * Zeitpunkt der geplanten Ausführung für diesen Auftrag in Millisekunden seit 1.1.1970.
     */
    private final long _ausfuehrungsZeitPunkt;

    /**
     * Datumsformatierung für Ausgabe toString().
     */
    SimpleDateFormat _datumsFormatierer = new SimpleDateFormat("dd.MM.yyyy HH:mm:ss,SSS");

    /**
     * IntervallBeginn
     */
    private final long _intervallBeginn;

    /**
     * IntervallEnde: Intervallbeginn + Intervalllänge (ausschließlich!), also der nächste Intervallbeginn.
     */
    private final long _intervallEnde;

    /**
     * Anzahl der Abrufversuche für diesen Auftrag.
     */
    private final int _aktAnzahlAbrufVersuche;

    /**
     * VewDeLve255 - Verwaltungsobjekt, welches den Auftrag ausführen soll.
     */
    private final VewDeLve255 _vewDeLve255;

    //~ KONSTRUKTOREN  (und vom Konstruktor verwendete Methoden) ==============

    /**
     * Konstruktor, um den initialen Erstauftrag für ein Verwaltungsobjekt und das erste zu berechnende Intervall
     * anzulegen.
     *
     * @param ausfuehrungsZeitpunkt  Zeitpunkt, zu dem Auftrag frühestens ausgeführt werden soll.
     * @param intervallBeginn        Intervallbeginn (EINSCHLIESSLICH!) in Millisekunden seit 1.1.1970
     * @param intervallEnde          Intervallende (AUSSCHLIESSLICH!) in Millisekunden seit 1.1.1970
     * @param aktAnzahlAbrufVersuche Anzahl der Abrufversuche für diesen Auftrag
     * @param vewDeLve255            Verwaltungsobjekt, welches den Auftrag letztendlich ausführen soll
     */
    public Auftrag(long ausfuehrungsZeitpunkt, long intervallBeginn, long intervallEnde, int aktAnzahlAbrufVersuche, VewDeLve255 vewDeLve255) {
        _ausfuehrungsZeitPunkt = ausfuehrungsZeitpunkt;
        _intervallBeginn = intervallBeginn;
        _intervallEnde = intervallEnde;
        _aktAnzahlAbrufVersuche = aktAnzahlAbrufVersuche;
        _vewDeLve255 = vewDeLve255;
    }

    //~ METHODEN ==============================================================

    /**
     * Führt den Auftrag aus, indem am zugeordneten Verwaltungsobjekt die Methode {@link
     * VewDeLve255#ausfuehren(Auftrag)} aufgerufen wir.
     */
    public void ausfuehren() {
        _vewDeLve255.ausfuehren(this);
    }

    //~ GET METHODEN ==========================================================

    /**
     * Liefert die Anzahl der Abrufversuche für diesen Auftrag.
     *
     * @return Anzahl der Abrufversuche für diesen Auftrag
     */
    public int getAktAnzahlAbrufVersuche() {
        return _aktAnzahlAbrufVersuche;
    }

    /**
     * Liefert den Zeitpunkt der geplanten Ausführung für diesen Auftrag in Millisekunden seit 1.1.1970.
     *
     * @return Zeitpunkt der geplanten Ausführung für diesen Auftrag in Millisekunden seit 1.1.1970
     */
    public long getAusfuehrungsZeitPunkt() {
        return _ausfuehrungsZeitPunkt;
    }

    /**
     * Liefert den Zeitpunkt des Intervallbeginns (EINSCHLIESSLICH!) in Millisekunden seit 1.1.1970.
     *
     * @return Zeitpunkt des Intervallbeginns (EINSCHLIESSLICH!) in Millisekunden seit 1.1.1970
     */
    public long getIntervallBeginn() {
        return _intervallBeginn;
    }

    /**
     * Liefert den Zeitpunkt des Intervallendes (AUSSCHLIESSLICH!) in Millisekunden seit 1.1.1970.
     *
     * @return Zeitpunkt des Intervallendes (AUSSCHLIESSLICH!) in Millisekunden seit 1.1.1970
     */
    public long getIntervallEnde() {
        return _intervallEnde;
    }

    /**
     * Liefert das Verwaltungsobjekt, welches den Auftrag ausführen soll.
     *
     * @return Verwaltungsobjekt, welches den Auftrag ausführen soll.
     */
    public VewDeLve255 getVewDeLve255() {
        return _vewDeLve255;
    }

    //~ METHODEN ==============================================================

    @Override

    public String toString() {
        return String.format("Auftrag (%3d Versuch) Ausführung am/um %s: Intervall %s - %s für %s", _aktAnzahlAbrufVersuche, _datumsFormatierer.format(new Date(_ausfuehrungsZeitPunkt)), _datumsFormatierer.format(new Date(_intervallBeginn)), _datumsFormatierer.format(new Date(_intervallEnde)), _vewDeLve255.getPid());
    }
}

