/*
 * Copyright (c) 2016-2024 by inovat
 * innovative systeme - verkehr - tunnel - technik,
 * Dipl.-Ing. H. C. Kniss
 *
 * This file is part of de.inovat.dua.pufferlzzsnachfordern.AuftragsListe
 *
 * de.inovat.dua.pufferlzzsnachfordern.AuftragsListe is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.inovat.dua.pufferlzzsnachfordern.AuftragsListe is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.inovat.dua.pufferlzzsnachfordern.AuftragsListe.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Diese Datei ist Teil von de.inovat.dua.pufferlzzsnachfordern.AuftragsListe.
 *
 * de.inovat.dua.pufferlzzsnachfordern.AuftragsListe ist Freie Software: Sie können es unter den Bedingungen
 * der GNU General Public License, wie von der Free Software Foundation,
 * Version 3 der Lizenz oder (nach Ihrer Wahl) jeder späteren
 * veröffentlichten Version, weiterverbreiten und/oder modifizieren.
 *
 * de.inovat.dua.pufferlzzsnachfordern.AuftragsListe wird in der Hoffnung, dass es nützlich sein wird, aber
 * OHNE JEDE GEWÄHRLEISTUNG, bereitgestellt; sogar ohne die implizite
 * Gewährleistung der MARKTFÄHIGKEIT oder EIGNUNG FÜR EINEN BESTIMMTEN ZWECK.
 * Siehe die GNU General Public License für weitere Details.
 *
 * Sie sollten eine Kopie der GNU General Public License zusammen mit diesem
 * Programm erhalten haben. Wenn nicht, siehe <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * inovat, Dipl.-Ing. H. C. Kniss
 * An der Krautwiese 37
 * D-53783 Eitorf
 * +49 (0)2243 8464 193
 * info@inovat.de
 * www.inovat.de
 */

package de.inovat.dua.pufferlzzsnachfordern;

//~ JDK IMPORTE ===============================================================

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.NavigableSet;
import java.util.concurrent.ConcurrentSkipListSet;

//~ KLASSEN ===================================================================

/**
 * Singletonklasse, welche die Auftragsliste für Pufferabfrageaufträge
 * verwaltet.
 * <p>
 * Die Auftragsliste ermöglicht das Hinzufügen von Aufträgen ({@link Auftrag}
 * sowie die Entnahme des aktuell auszuführenden Auftrags. Die Zugriffe auf die
 * Auftragsliste ist intern synchronisiert.
 *
 * @author inovat, innovative systeme - verkehr - tunnel - technik
 * @author Dipl.-Ing. Hans Christian Kniß (HCK)
 */
public class AuftragsListe {
	private static final AuftragsListe dieEinzigeInstanz = new AuftragsListe();
	private static final NavigableSet<Auftrag> _listeAuftraege = new ConcurrentSkipListSet<>(
			Comparator.comparing(Auftrag::getAusfuehrungsZeitPunkt).thenComparing(Comparator
					.comparing(a -> a.getVewDeLve255().toString())));

	// ~ KONSTRUKTOREN (und vom Konstruktor verwendete Methoden) ==============

	/**
	 * Privater Konstruktor.
	 */
	private AuftragsListe() {
	}

	// ~ GET METHODEN ==========================================================

	/**
	 * Liefert die Instanz des Singletons.
	 *
	 * @return Instanz des Singletons
	 */
	public static AuftragsListe getInstanz() {
		return dieEinzigeInstanz;
	}

	// ~ METHODEN ==============================================================

	/**
	 * Fügt einen neuen Auftrag zur Liste der Aufträge hinzu. Die Liste wird
	 * intern so sortiert, dass die Methode {@link #holeNaechstenAuftrag()} immer
	 * den zum aktuellen Zeitpunkt nächsten auszuführenden Auftrag zurückliefert.
	 *
	 * @param auftrag Neuer Auftrag, der in die Liste der Aufträge eingetragen
	 *                werden soll.
	 */
	public void addAuftrag(Auftrag auftrag) {
		if (!istAuftragBereitsVorhanden(auftrag)) {
			_listeAuftraege.add(auftrag);
		}
	}

	/**
	 * Lieferte die formatierte Ausgabe aller Aufträge in der Auftragsliste.
	 *
	 * @return Formatierte Ausgabe aller Aufträge in der Auftragsliste
	 */
	public String ausgabeGesamtListe() {
		StringBuilder sb = new StringBuilder();
		int m = 0;

		for (Auftrag auftrag : AuftragsListe.getInstanz().holeKopieAuftragsListe()) {
			sb.append(String.format("%06d: %s%n", ++m, auftrag.toString()));
		}

		return sb.toString();
	}

	/**
	 * Lieferte die formatierte Ausgabe aller Aufträge für ein bestimmtes
	 * Verwaltungsobjekt in der Auftragsliste.
	 *
	 * @param vewDeLve255 Verwaltungsobjekt, für das die Auftragsliste ausgegeben
	 *                    werden soll
	 * @return Formatierte Ausgabe aller Aufträge in der Auftragsliste
	 */
	public String ausgabeTeilListe(VewDeLve255 vewDeLve255) {
		StringBuilder sb = new StringBuilder();
		int m = 0;

		for (Auftrag auftrag : AuftragsListe.getInstanz().holeKopieAuftragsListe()) {
			if (auftrag.getVewDeLve255()._deLve255.equals(vewDeLve255._deLve255)) {
				sb.append(String.format("%05d: %s%n", ++m, auftrag));
			}
		}

		return sb.toString();
	}

	/**
	 * Liefert den Ausführungszeitpunkt des aktuell letzten Auftrags für ein
	 * spezifisches Verwaltungsobjekt in der Auftragsliste zurück.
	 *
	 * @param vewDeLve255 Verwaltungsobjekt, für den die Aufträge untersucht
	 *                    werden.
	 * @return Ausführungszeitpunkt des aktuell letzten Auftrags in der
	 *         Auftragsliste in ms oder aktueller Zeitpunkt, wenn keine passenden
	 *         Aufträge in der Liste vorhanden sind.
	 */
	public long ermittleSpaetestenAusfuehrungsZeitPunkt(VewDeLve255 vewDeLve255) {
		long letzterAusfuehrungsZeitPunkt = System.currentTimeMillis();

		for (Auftrag auftrag : _listeAuftraege) {
			if (auftrag.getVewDeLve255()._deLve255.equals(vewDeLve255._deLve255)) {
				letzterAusfuehrungsZeitPunkt = auftrag.getAusfuehrungsZeitPunkt();
			}
		}

		return letzterAusfuehrungsZeitPunkt;
	}

	/**
	 * Gibt zurück, ob noch Aufträge (aktuell oder zukünftig) anstehen.
	 *
	 * @return <code>true</code>, wenn noch Aufträge (aktuell oder zukünftig)
	 *         anstehen, andernfalls <code>false</code>.
	 */
	public boolean hatNochAuftraege() {
		return !_listeAuftraege.isEmpty();
	}

	/**
	 * Liefert eine Kopie der Auftragsliste.
	 *
	 * @return Kopie der Auftragsliste.
	 */
	public List<Auftrag> holeKopieAuftragsListe() {
		List<Auftrag> listeKopieAuftraege = new ArrayList<>();
		listeKopieAuftraege.addAll(_listeAuftraege);
		return listeKopieAuftraege;
	}

	/**
	 * Liefert den, bezogen auf den aktuellen Zeitpunkt, nächsten auszuführenden
	 * Auftrag zurück.
	 *
	 * @return Nächster auszuführender Auftrag oder <code>null</code>, wenn kein
	 *         Auftrag aktuell auszuführen ist. Der zurückgegebene Auftrag wird
	 *         entfernt und kann nicht noch einmal abgerufen werden.
	 */
	public Auftrag holeNaechstenAuftrag() {
		Auftrag naechsterAuftrag = null;

		if ((!_listeAuftraege.isEmpty())
				&& (_listeAuftraege.first().getAusfuehrungsZeitPunkt() < System.currentTimeMillis())) {
			naechsterAuftrag = _listeAuftraege.pollFirst();
		}
		return naechsterAuftrag;
	}

	/**
	 * Prüft, ob ein identischer Auftrag bereits in der Liste vorhanden ist. Zwei
	 * Auftraege sind identisch, wenn Intervallbeginn, Intervallende und
	 * Verwaltungsobjekt identisch sind.
	 *
	 * @param auftrag Auftrag, für den getestet werden soll, ob ein identischer
	 *                Auftrag bereits in der Liste vorliegt
	 * @return <code>true</code>, wenn identischer Auftrag bereits in der
	 *         Auftragsliste vorhanden ist, sonst <code>false</code>
	 */
	public boolean istAuftragBereitsVorhanden(Auftrag auftrag) {
		for (Auftrag auftragInListe : _listeAuftraege) {
			if (sindAuftraegeIdentisch(auftragInListe, auftrag)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Prüft, ob zwei Auftraege identisch sind. Zwei Auftraege sind identisch, wenn
	 * Intervallbeginn, Intervallende und Verwaltungsobjekt identisch sind (der
	 * Ausführungszeitpunkt wird NICHT berücksichtigt).
	 *
	 * @param a1 Zu vergleichende Auftraege.
	 * @param a2 Zu vergleichende Auftraege.
	 * @return <code>true</code>, wenn Auftraege gleich sind, sonst
	 *         <code>false</code>
	 */
	public boolean sindAuftraegeIdentisch(Auftrag a1, Auftrag a2) {
		if ((a1.getIntervallBeginn() == a2.getIntervallBeginn()) && (a1.getIntervallEnde() == a2.getIntervallEnde())
				&& a1.getVewDeLve255()._deLve255.equals(a2.getVewDeLve255()._deLve255)) {
			return true;
		}

		return false;
	}
}
