/*
 * Copyright (c) 2016-2024 by inovat
 * innovative systeme - verkehr - tunnel - technik,
 * Dipl.-Ing. H. C. Kniss
 *
 * This file is part of de.inovat.dua.pufferlzzsnachfordern.PufferAbrufLzzsNachfordern
 *
 * de.inovat.dua.pufferlzzsnachfordern.PufferAbrufLzzsNachfordern is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.inovat.dua.pufferlzzsnachfordern.PufferAbrufLzzsNachfordern is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.inovat.dua.pufferlzzsnachfordern.PufferAbrufLzzsNachfordern.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Diese Datei ist Teil von de.inovat.dua.pufferlzzsnachfordern.PufferAbrufLzzsNachfordern.
 *
 * de.inovat.dua.pufferlzzsnachfordern.PufferAbrufLzzsNachfordern ist Freie Software: Sie können es unter den Bedingungen
 * der GNU General Public License, wie von der Free Software Foundation,
 * Version 3 der Lizenz oder (nach Ihrer Wahl) jeder späteren
 * veröffentlichten Version, weiterverbreiten und/oder modifizieren.
 *
 * de.inovat.dua.pufferlzzsnachfordern.PufferAbrufLzzsNachfordern wird in der Hoffnung, dass es nützlich sein wird, aber
 * OHNE JEDE GEWÄHRLEISTUNG, bereitgestellt; sogar ohne die implizite
 * Gewährleistung der MARKTFÄHIGKEIT oder EIGNUNG FÜR EINEN BESTIMMTEN ZWECK.
 * Siehe die GNU General Public License für weitere Details.
 *
 * Sie sollten eine Kopie der GNU General Public License zusammen mit diesem
 * Programm erhalten haben. Wenn nicht, siehe <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * inovat, Dipl.-Ing. H. C. Kniss
 * An der Krautwiese 37
 * D-53783 Eitorf
 * +49 (0)2243 8464 193
 * info@inovat.de
 * www.inovat.de
 */


package de.inovat.dua.pufferlzzsnachfordern;

//~ JDK IMPORTE ===============================================================

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;

//~ NICHT JDK IMPORTE =========================================================

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.ConfigurationArea;
import de.bsvrz.dav.daf.main.config.ConfigurationObject;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.NonMutableSet;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.application.StandardApplication;
import de.bsvrz.sys.funclib.application.StandardApplicationRunner;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.debug.Debug;

//~ KLASSEN ===================================================================

/**
 * Klasse, die die den Pufferabruf für fehlende Daten der TLS-De-Daten für die LZZS durchführt.
 * <p>
 * Die Intervalldauer der nachzufordernden Langzeitdaten wird wegen der Abhängigkeit zu den BASt-Band-Daten hier fest
 * mit 1 Stunde angenommen.
 *
 * @author inovat, innovative systeme - verkehr - tunnel - technik
 * @author Dipl.-Ing. Hans Christian Kniß (HCK)
 */
public class PufferAbrufLzzsNachfordern implements StandardApplication {

    /**
     * WarteZeit in ms, bis die Auftragsliste erneut geprüft wird, nachdem keinerlei Aufträge mehr vorlagen
     */
    private static final long WARTEZEIT_AUF_NAECHSTEN_AUFTRAG = 60 * 1000;  // 1 Minute
    private static final long EINE_MINUTE_IN_MILLISEKUNDEN = 60 * 1000;
    private static final long EINE_STUNDE_IN_MILLISEKUNDEN = EINE_MINUTE_IN_MILLISEKUNDEN * 60;

    /**
     * DebugLogger für Debug-Ausgaben.
     */
    private static Debug debug;

    //~ FELDER ================================================================

    /**
     * Liste der PIDs der Langzeitzählstellen, für die der Pufferabruf arbeitet
     */
    private List<String> _listePidsLzzs = new ArrayList<>();

    /**
     * Liste der Systemobjekte der Langzeitzählstellen, für die der Pufferabruf arbeitet
     */
    private final List<SystemObject> _listeLzzsSystemObjekte = new ArrayList<>();

    /**
     * Liste der PIDs der Langzeitzählstellen im System, die nicht verwendet werden. (Wird nur für die
     * Konfigurationszusammenfassung verwendet)
     */
    private final List<String> _listePidsLzzsNichtVorhanden = new ArrayList<>();

    /**
     * Map der Verwaltungsobjekte der DE255 für die Pufferabfrage TLS, Schlüssel ist die Pid des DE255
     */
    private  final Map<String, VewDeLve255> _mapVewDeLve255 = new HashMap<>();

    /**
     * Datenverteilerverbindung
     */
    private ClientDavInterface _dav;

    /**
     * Die Konfiguration des DaV.
     */
    private DataModel _konfiguration;

    /**
     *
     */
    private String _listePidsLzzsAufrufParameter;

    /**
     * Map der DeLve255 für Pufferabruf mit der Menge der zugeordneten FS
     */
    Map<SystemObject, Set<SystemObject>> _mapDe255ZuFs;

    /**
     * Map aller DE-LVE und der zugeordneten DE255 der LVE
     */
    Map<SystemObject, SystemObject> _mapDeLveZuDe255;

    /**
     * Map aller DE-LVE und der zugeordneten FS von LZZS
     */
    Map<SystemObject, SystemObject> _mapDeLveZuFs;

    /**
     * Anzahl der Tage, für die Langzeitdaten ggf. nachgefordert werden sollen
     */
    private int _datenNachforderungInTagen;

    /**
     * Aufrufparameter für _datenNachforderungInTagen
     */
    private String _datenNachforderungInTagenAufrufParameter;

    /**
     * Wartezeit, die mindestens zwischen zwei Abrufversuchen für einen fehlenden Datensatz liegen muss
     */
    private int _minWarteZeitZwischenAbrufVersuchenInStunden;

    /**
     * Aufrufparameter für _minWarteZeitZwischenAbrufVersuchenInStunden
     */
    private String _minWarteZeitZwischenAbrufVersuchenInStundenAufrufParameter;

    /**
     * Maximale Anzahl der Versuche für die Abfrage eines fehlenden Datensatzes
     */
    private int _maxAnzahlAbrufVersuche;

    /**
     * Aufrufparameter für _maxAnzahlAbrufVersuche
     */
    private String _maxAnzahlAbrufVersucheAufrufParameter;

    //~ METHODEN ==============================================================

    /**
     * Main-Methode der SWE PufferAbrufLzzsNachfordern.
     *
     * @param args Aufrufargumente der Applikation.
     */
    public static void main(String[] args) {
        PufferAbrufLzzsNachfordern pufferAbrufLzzsNachfordern = new PufferAbrufLzzsNachfordern();

        try {
            StandardApplicationRunner.run(pufferAbrufLzzsNachfordern, args);

            // Debug erst hier initialisieren, da erst ab hier verfügbar!
            debug = Debug.getLogger();
            pufferAbrufLzzsNachfordern.auswertungAufrufParameterListePids();
            pufferAbrufLzzsNachfordern.initialisierung();
            debug.config(pufferAbrufLzzsNachfordern.ausgabeKonfigurationsZusammenfassung());
        } catch (RuntimeException e) {
            e.printStackTrace();
            debug.error(String.format("%n=================================================================%n" +
                    "Programm 'PufferAbruf Langzeitdaten nachfordern' wegen unerwartetem Fehler beendet...%n" +
                    "=================================================================%n"));
            System.exit(-1);
        }

        pufferAbrufLzzsNachfordern.run();
    }

    /**
     * Gibt Informationen zur Konfiguration aus. Ist nur zu Analysezwecken bzw. zur zusammenfassenden Ausgabe der
     * Konfigurationsphase dieser Applikation interessant.
     *
     * @return Informationen zur Konfiguration (Aufbau der Ausgabe kann sich ändern).
     */
    private String ausgabeKonfigurationsZusammenfassung() {
        StringBuilder sb = new StringBuilder();

        // Listen der SystemObjekte sortieren...
        _listeLzzsSystemObjekte.sort(Comparator.comparing(SystemObject::getPid));
        _listePidsLzzsNichtVorhanden.sort(String::compareTo);

        sb.append(System.lineSeparator()+"####################################################################################"+System.lineSeparator());
        sb.append("START: Konfigurationszusammenfassung für PufferAbrufLzzsNachfordern"+System.lineSeparator());
        sb.append("####################################################################################"+System.lineSeparator());
        sb.append("===================================================================================="+System.lineSeparator());
        sb.append("Im Aufrufparameter angegebene PIDs, zu denen keine SystemObjekte existieren"+System.lineSeparator());
        sb.append("===================================================================================="+System.lineSeparator());

        int i = 0;

        for (String pid : _listePidsLzzsNichtVorhanden) {
            sb.append(String.format("%04d: %s%n", ++i, pid));
        }
        sb.append("===================================================================================="+System.lineSeparator());
        sb.append("Liste der PIDs der verwendeten Systemobjekte"+System.lineSeparator());
        sb.append("===================================================================================="+System.lineSeparator());

        int j = 0;

        for (SystemObject so : _listeLzzsSystemObjekte) {
            sb.append(String.format("%04d: %s%n", ++j, so.getPidOrNameOrId()));
        }

        sb.append("===================================================================================="+System.lineSeparator());
        sb.append("Liste der DeLve255 zum Pufferabruf und den zugeordneten Fahrstreifenobjekten"+System.lineSeparator());
        sb.append("===================================================================================="+System.lineSeparator());

        int k = 0;

        for (SystemObject soDeLve255 : _mapDe255ZuFs.keySet()) {
            sb.append(String.format("%04d: %s%n", ++k, soDeLve255.getPidOrNameOrId()));

            int l = 0;

            for (SystemObject soFs : _mapDe255ZuFs.get(soDeLve255)) {
                sb.append(String.format("      %04d: %s%n", ++l, soFs.getPidOrNameOrId()));
            }
        }

        sb.append("===================================================================================="+System.lineSeparator());
        sb.append("Aufrufparameter -datenNachforderungInTagen=..."+System.lineSeparator());
        sb.append("===================================================================================="+System.lineSeparator());
        sb.append(String.format("Parameter beim Aufruf (Vorgabe)       : [%s]%n", _datenNachforderungInTagenAufrufParameter));
        sb.append(String.format("Parameter nach Auswertung (Verwendung): [%s]%n", _datenNachforderungInTagen));
        sb.append("===================================================================================="+System.lineSeparator());
        sb.append("Aufrufparameter -minWarteZeitZwischenAbrufVersuchenInStunden=..."+System.lineSeparator());
        sb.append("===================================================================================="+System.lineSeparator());
        sb.append(String.format("Parameter beim Aufruf (Vorgabe)       : [%s]%n", _minWarteZeitZwischenAbrufVersuchenInStundenAufrufParameter));
        sb.append(String.format("Parameter nach Auswertung (Verwendung): [%s]%n", _minWarteZeitZwischenAbrufVersuchenInStunden));
        sb.append("===================================================================================="+System.lineSeparator());
        sb.append("Aufrufparameter -maxAnzahlAbrufVersuche=..."+System.lineSeparator());
        sb.append("===================================================================================="+System.lineSeparator());
        sb.append(String.format("Parameter beim Aufruf (Vorgabe)       : [%s]%n", _maxAnzahlAbrufVersucheAufrufParameter));
        sb.append(String.format("Parameter nach Auswertung (Verwendung): [%s]%n", _maxAnzahlAbrufVersuche));
        sb.append("===================================================================================="+System.lineSeparator());
        sb.append("===================================================================================="+System.lineSeparator());
        sb.append("Auftragsliste initial..."+System.lineSeparator());
        sb.append("===================================================================================="+System.lineSeparator());
        sb.append(AuftragsListe.getInstanz().ausgabeGesamtListe());
        sb.append(System.lineSeparator());
        sb.append("####################################################################################"+System.lineSeparator());
        sb.append("ENDE: Konfigurationszusammenfassung für PufferAbrufLzzsNachfordern"+System.lineSeparator());
        sb.append("####################################################################################"+System.lineSeparator());

        return sb.toString();
    }


    /**
     * Wertet die Aufrufparameter <code>-listePids</code> aus.
     * <p>
     * Es werden alle Systemobjekte vom Typ <code>typ.langZeitZählstelle</code> ermittelt und dabei diejenigen für die
     * weiteren Berechnungen übernommen, deren Pid in der Liste der Pids aufgeführt ist (bzw. alle, falls Aufruf mit
     * <code>-listePids=alle</code> erfolgt). Wird kein passendes Objekt gefunden, wird das Programm mit einer Fehlermeldung
     * abgebrochen.
     */
    private void auswertungAufrufParameterListePids() {
        try {

            // Pids der zu berücksichtigenden Objekte ermitteln
            _listePidsLzzs = Arrays.asList(_listePidsLzzsAufrufParameter.split("(:|,|;)"));
            
        	// Alle möglichen Langzeitzählstellen auch aus KB's ermitteln
			_listePidsLzzs.stream().map(s -> _konfiguration.getObject(s)).distinct().flatMap(o -> {
				if (o instanceof ConfigurationArea) {
					return ((ConfigurationArea) o).getCurrentObjects().stream()
							.filter(so -> so.isOfType("typ.langZeitZählstelle"));
				} else if (o instanceof SystemObject && o.isOfType("typ.langZeitZählstelle")) {
					return Stream.of(o);
				}
				return Stream.empty();
			}).distinct().forEach(_listeLzzsSystemObjekte::add);


            // Alle möglichen SystemObjekte ermitteln...
            List<SystemObject> listeAlleLzzsSystemObjekte = _konfiguration.getType("typ.langZeitZählstelle").getElements();

            // ... und jetzt Liste mit zu verwendenden aufbauen (falls nicht "alle" verwendet werden sollen) ...
            boolean isAlle = ((_listePidsLzzs.size() == 1)
                    && _listePidsLzzs.get(0).equals("alle"));

            for (SystemObject so : listeAlleLzzsSystemObjekte) {
                if (isAlle
                        || _listePidsLzzs.contains(so.getPid())) {
                    _listeLzzsSystemObjekte.add(so);
                }
            }

            // ... und für Konfigurationszusammenfassung Liste der nicht verwendeten PID's aus dem Aufrufparameter ermitteln
            List<String> listePidsVerwendet = new ArrayList<>();

            for (SystemObject so : _listeLzzsSystemObjekte) {
                listePidsVerwendet.add(so.getPid());
            }

            for (String pid : _listePidsLzzs) {
                if (!listePidsVerwendet.contains(pid)) {
                    _listePidsLzzsNichtVorhanden.add(pid);
                }
            }
        } catch (Exception e) {
            debug.error("Es konnten keine Objekte vom typ.langZeitZählstelle ermittelt werden! Programm wird beendet.", e);

            throw new RuntimeException();
        }

        if (_listeLzzsSystemObjekte.isEmpty()) {
            debug.error("Es konnten keine Objekte vom typ.langZeitZählstelle ermittelt werden! Programm wird beendet.");

            throw new RuntimeException();
        }
    }

    /**
     * Ermittelt aus allen FS und den zugeordneten DE die DELve255 für jeden FS, damit für diesen die Pufferabfrage
     * durchgeführt werden kann.
     *
     * @param mapDeLveZuDe255 Map mit der Zuordnung der DeLve zur DeLve255 für die Pufferabfrage
     * @param mapDeLveZuFs    Map mit der Zuordnung der DeLve zum FS einer LZZS
     * @return Map mit den DeLve255 und der Menge der FS, deren Daten über diese De aus dem Puffer abgerufen werden.
     */
    private Map<SystemObject, Set<SystemObject>> ermittleZuordnungDe255ZuFs(Map<SystemObject, SystemObject> mapDeLveZuDe255, Map<SystemObject, SystemObject> mapDeLveZuFs) {
        Map<SystemObject, Set<SystemObject>> mapDe255ZuFs = new HashMap<>();

        for (SystemObject soDe : mapDeLveZuFs.keySet()) {

            // Aus der anderen Map die zur De passende De255 holen
            SystemObject soDeLve255 = mapDeLveZuDe255.get(soDe);

            // Testen, ob für die DE255 bereits ein Eintrag in der Ergebnisliste vorliegt, sonst neu anlegen
            Set<SystemObject> mengeFs = mapDe255ZuFs.get(soDeLve255);

            if (mengeFs == null) {

                // Eintrag und Menge für FS existieren noch nicht
                mengeFs = new HashSet<>();
            }

            // Menge existiert jetzt in jedem Fall, dann einfach den zugeordneten FS holen und eintragen
            mengeFs.add(mapDeLveZuFs.get(soDe));
            mapDe255ZuFs.put(soDeLve255, mengeFs);
        }

        return mapDe255ZuFs;
    }

    /**
     * Ermittelt die Liste der Zuordnungen der DeLve zur DeLv255 am selben SM, also welche DeLve über welchen
     * Sammelkanal 255 abgerufen werden können.
     *
     * @return Map mit der Zuordnung der DeLve zur DeLve255 für die Pufferabfrage
     */
    private Map<SystemObject, SystemObject> ermittleZuordnungDeLveZuDe255() {
        Map<SystemObject, SystemObject> mapDeLveZuDe255 = new HashMap<>();
        AttributeGroup atg = _konfiguration.getAttributeGroup("atg.de");
        Aspect asp = _konfiguration.getAspect("asp.eigenschaften");

        // Hole alle SM
        List<SystemObject> listeSm = _konfiguration.getType("typ.steuerModul").getElements();

        for (SystemObject soSm : listeSm) {

            // Hole alle EAK am SM
            List<SystemObject> listeEak = ((ConfigurationObject) soSm).getNonMutableSet("Eak").getElements();

            // Hole alle DE an den EAK
            List<SystemObject> listeAllerDeAmSm = new ArrayList<>();

            for (SystemObject soEak : listeEak) {
                List<SystemObject> listeDe = ((ConfigurationObject) soEak).getNonMutableSet("De").getElements();

                listeAllerDeAmSm.addAll(listeDe);
            }

            // Ermittele das DE255 in der FG1, wenn es keins gibt, dann keine weiteren Eintragungen für diesen SM
            _konfiguration.getConfigurationData(listeAllerDeAmSm.toArray(new SystemObject[listeAllerDeAmSm.size()]), atg, asp);

            SystemObject soDeLve255 = null;

            for (SystemObject soDe : listeAllerDeAmSm) {
                Data[] daten = _konfiguration.getConfigurationData(new SystemObject[]{soDe}, atg, asp);

                if (soDe.isOfType("typ.deLve")
                        && (daten[0].getUnscaledValue("DEKanal").intValue() == 255)) {
                    soDeLve255 = soDe;

                    break;
                }
            }

            if (soDeLve255 != null) {

                // Ermittle alle DE der FG1 und speichere Paar DE-LVE:DE-LVE-255 in der Map
                for (SystemObject soDe : listeAllerDeAmSm) {
                    Data[] daten = _konfiguration.getConfigurationData(new SystemObject[]{soDe}, atg, asp);

                    if (soDe.isOfType("typ.deLve")
                            && (daten[0].getUnscaledValue("DEKanal").intValue() != 255)) {
                        mapDeLveZuDe255.put(soDe, soDeLve255);
                    }
                }
            }
        }

        return mapDeLveZuDe255;
    }

    /**
     * Ermittelt zu allen LZZS die DE an den FS der MQ der AWQ der LZZS.
     *
     * @param listeLzzsSystemObjekte Liste mit allen LZZS, für die die Zuordnung DE-FS ermittelt werden soll.
     * @return Map mit der Zuordnung DE-FS für die LZZS.
     */
    private Map<SystemObject, SystemObject> ermittleZuordnungDeZuFsAusLzzs(List<SystemObject> listeLzzsSystemObjekte) {
        Map<SystemObject, SystemObject> mapDeLveZuFs = new HashMap<>();

        for (SystemObject lzzs : listeLzzsSystemObjekte) {

            // ------------------------------------------------------------------------------------------------
            // MQ inRichtung und gegenRichtung des LZZS ermitteln
            // ------------------------------------------------------------------------------------------------
            SystemObject soAwqInRichtung, soAwqGegenRichtung, soMqInRichtung, soMqGegenRichtung, soDe;

            // ATG an der LZZS mit den Referenzen auf die AWQ ermitteln
            AttributeGroup atg = _konfiguration.getAttributeGroup("atg.langZeitZählstelle");
            Aspect asp = _konfiguration.getAspect("asp.eigenschaften");
            Data[] daten = _konfiguration.getConfigurationData(new SystemObject[]{lzzs}, atg, asp);

            // AWQ InRichtung ermitteln
            soAwqInRichtung = daten[0].getItem("InRichtung").getReferenceValue("Auswertequerschnitt").getSystemObject();

            // AWQ GegenRichtung ermitteln
            soAwqGegenRichtung = daten[0].getItem("GegenRichtung").getReferenceValue("Auswertequerschnitt").getSystemObject();

            // MQ InRichtung ermitteln
            atg = _konfiguration.getAttributeGroup("atg.auswerteQuerschnitt");
            asp = _konfiguration.getAspect("asp.eigenschaften");
            daten = _konfiguration.getConfigurationData(new SystemObject[]{soAwqInRichtung}, atg, asp);
            soMqInRichtung = daten[0].getReferenceValue("Messquerschnitt").getSystemObject();

            // MQ GegenRichtung ermitteln
            daten = _konfiguration.getConfigurationData(new SystemObject[]{soAwqGegenRichtung}, atg, asp);
            soMqGegenRichtung = daten[0].getReferenceValue("Messquerschnitt").getSystemObject();

            // ATG mir FS-Informationen und DE-Referenz initialisieren
            atg = _konfiguration.getAttributeGroup("atg.fahrStreifen");
            asp = _konfiguration.getAspect("asp.eigenschaften");
            
			// Menge mit den FS am MQ InRichtung ermitteln
			NonMutableSet nonMutableSetInRichtung = ((ConfigurationObject) soMqInRichtung)
					.getNonMutableSet("FahrStreifen");
			if (nonMutableSetInRichtung != null) {
				  List<SystemObject> listeSoFS = nonMutableSetInRichtung.getElements();

				for (SystemObject soFs : listeSoFS) {

					// Referenzierte DE ermitteln und Paar DE-FS in der Map speichern
					daten = _konfiguration.getConfigurationData(new SystemObject[] { soFs }, atg, asp);
					soDe = daten[0].getReferenceValue("FahrStreifenQuelle").getSystemObject();

					if (soDe != null) {
						mapDeLveZuFs.put(soDe, soFs);
					}
				}
			}

			// Menge mit den FS am MQ GegenRichtung ermitteln
			NonMutableSet nonMutableSetGegenRichtung = ((ConfigurationObject) soMqGegenRichtung).getNonMutableSet("FahrStreifen");
			if (nonMutableSetGegenRichtung != null) {
				List<SystemObject> listeSoFS = nonMutableSetGegenRichtung.getElements();

				for (SystemObject soFs : listeSoFS) {

					// Referenzierte DE ermitteln und Paar DE-FS in der Map speichern
					daten = _konfiguration.getConfigurationData(new SystemObject[] { soFs }, atg, asp);
					soDe = daten[0].getReferenceValue("FahrStreifenQuelle").getSystemObject();

					if (soDe != null) {
						mapDeLveZuFs.put(soDe, soFs);
					}
				}
			}
        }

        return mapDeLveZuFs;
    }

    /**
     * Erzeugt für alle Verwaltungsobjekte die Aufträge, die zwischen der Initialisierung des Programms und dem
     * aktuellen Zeitpunkt entstehen.
     */
    private void erzeugeAuftraegeFuerNeueDatenLuecken() {
        AuftragsListe auftragsListe = AuftragsListe.getInstanz();

        for (VewDeLve255 vewDeLve255 : _mapVewDeLve255.values()) {
            while ((vewDeLve255.getLetzterIntervallBeginn() + EINE_STUNDE_IN_MILLISEKUNDEN) < System.currentTimeMillis() - (_minWarteZeitZwischenAbrufVersuchenInStunden * EINE_STUNDE_IN_MILLISEKUNDEN)) {
                long ausfuehrungsZeitPunkt = auftragsListe.ermittleSpaetestenAusfuehrungsZeitPunkt(vewDeLve255) + EINE_MINUTE_IN_MILLISEKUNDEN;
                long intervallBeginn = vewDeLve255.getLetzterIntervallBeginn() + EINE_STUNDE_IN_MILLISEKUNDEN;
                long intervallEnde = intervallBeginn + EINE_STUNDE_IN_MILLISEKUNDEN;
                Auftrag auftrag = new Auftrag(ausfuehrungsZeitPunkt, intervallBeginn, intervallEnde, 1, vewDeLve255);

                auftragsListe.addAuftrag(auftrag);
                vewDeLve255.setLetzterIntervallBeginn(intervallBeginn);
                debug.info(String.format("Neuer erstmalig eingestellter Auftrag: %s", auftrag));
            }
        }
    }

    /**
     * Initialisierung von PufferAbrufLzzsNachfordern
     */
    private void initialisierung() {

        // Map aller DE-LVE und der zugeordneten DE255 der LVE ermitteln
        _mapDeLveZuDe255 = ermittleZuordnungDeLveZuDe255();

        // Aus den LZZS alle FS und die zugeordneten DE-LVE ermitteln
        _mapDeLveZuFs = ermittleZuordnungDeZuFsAusLzzs(_listeLzzsSystemObjekte);

        // Über die DE-LVE die Liste mit den DE255 der LVE und den zugeordneten FS ermitteln,
        // aus dieser Liste dann die eigentlichen Verwaltungsobjekte VewDeLve255 konstruieren.
        _mapDe255ZuFs = ermittleZuordnungDe255ZuFs(_mapDeLveZuDe255, _mapDeLveZuFs);

    	while (!_dav.getArchive().isArchiveAvailable()) {
			try {
				debug.warning("Archiv ist nicht verfügbar. Warte 1 Minute auf Verfügbarkeit...");
				Thread.sleep(TimeUnit.MINUTES.toMillis(1));
			} catch (InterruptedException e) {
				debug.error("Wartezeit auf Archivverfügbarkeit unterbrochen", e);
				Thread.currentThread().interrupt();
			}
		}
        
        // VewDeLve255-Objekte konstruieren
        for (SystemObject deLve255 : _mapDe255ZuFs.keySet()) {
            VewDeLve255 vewDeLve255 = new VewDeLve255(_dav, deLve255, _mapDe255ZuFs.get(deLve255), _datenNachforderungInTagen, _minWarteZeitZwischenAbrufVersuchenInStunden, _maxAnzahlAbrufVersuche);

            _mapVewDeLve255.put(deLve255.getPid(), vewDeLve255);
        }
    }

    /**
     * Initialisierung. Wird durch den {@link de.bsvrz.sys.funclib.application.StandardApplicationRunner} aufgerufen.
     *
     * @param dav Die Datenverteilerverbindung.
     */
    @Override
    public void initialize(ClientDavInterface dav) {
        _dav = dav;
        _konfiguration = _dav.getDataModel();
    }

    /**
     * Wertet die Aufrufparameter aus.
     *
     * @param argumentList Aufrufparameter der Applikation
     * @throws Exception wenn die Aufrufparameter nicht vollständig ausgewertet werden können.
     */
    @Override
    public void parseArguments(ArgumentList argumentList) throws Exception {
        debug = Debug.getLogger();

        // ------------------------------------------------------------------------------------------------------
        // Aufrufparameter: -listePids
        // ------------------------------------------------------------------------------------------------------
        _listePidsLzzsAufrufParameter = argumentList.fetchArgument("-listePids=alle").asNonEmptyString();

        // ------------------------------------------------------------------------------------------------------
        // Aufrufparameter: -datenNachforderungInTagen
        //
        // <tt>-datenNachforderungInTagen</tt>: Wird auf Gültigkeit getestet (ganze Zahl > 0) und bei
        // ungültigem Wert auf 1 Tag gesetzt.
        // ------------------------------------------------------------------------------------------------------
        try {
            _datenNachforderungInTagenAufrufParameter = argumentList.fetchArgument("-datenNachforderungInTagen=40").asNonEmptyString().trim();
            _datenNachforderungInTagen = Integer.parseInt(_datenNachforderungInTagenAufrufParameter);

            if (_datenNachforderungInTagen < 1) {
                throw new NumberFormatException();
            }
        } catch (NumberFormatException e) {
            _datenNachforderungInTagen = 40;
            debug.warning(String.format("Aufrufparameter [-datenNachforderungInTagen=%s] ist ungültig. Wert wurde für weitere Berechnung auf [-datenNachforderungInTagen=%d] gesetzt.", _datenNachforderungInTagenAufrufParameter, _datenNachforderungInTagen));
        }

        // ------------------------------------------------------------------------------------------------------
        // Aufrufparameter: -minWarteZeitZwischenAbrufVersuchenInStunden
        //
        // <tt>-minWarteZeitZwischenAbrufVersuchenInStunden</tt>: Wird auf Gültigkeit getestet (ganze Zahl > 0) und bei
        // ungültigem Wert auf 24 Stunden gesetzt.
        // ------------------------------------------------------------------------------------------------------
        try {
            _minWarteZeitZwischenAbrufVersuchenInStundenAufrufParameter = argumentList.fetchArgument("-minWarteZeitZwischenAbrufVersuchenInStunden=24").asNonEmptyString().trim();
            _minWarteZeitZwischenAbrufVersuchenInStunden = Integer.parseInt(_minWarteZeitZwischenAbrufVersuchenInStundenAufrufParameter);

            if (_minWarteZeitZwischenAbrufVersuchenInStunden < 1) {
                throw new NumberFormatException();
            }
        } catch (NumberFormatException e) {
            _minWarteZeitZwischenAbrufVersuchenInStunden = 12;
            debug.warning(String.format("Aufrufparameter [-minWarteZeitZwischenAbrufVersuchenInStunden=%s] ist ungültig. Wert wurde für weitere Berechnung auf [-minWarteZeitZwischenAbrufVersuchenInStunden=%d] gesetzt.", _minWarteZeitZwischenAbrufVersuchenInStundenAufrufParameter, _minWarteZeitZwischenAbrufVersuchenInStunden));
        }

        // ------------------------------------------------------------------------------------------------------
        // Aufrufparameter: -maxAnzahlAbrufVersuche
        //
        // <tt>-maxAnzahlAbrufVersuche</tt>: Wird auf Gültigkeit getestet (ganze Zahl > 0) und bei
        // ungültigem Wert auf 10 gesetzt.
        // ------------------------------------------------------------------------------------------------------
        try {
            _maxAnzahlAbrufVersucheAufrufParameter = argumentList.fetchArgument("-maxAnzahlAbrufVersuche=10").asNonEmptyString().trim();
            _maxAnzahlAbrufVersuche = Integer.parseInt(_maxAnzahlAbrufVersucheAufrufParameter);

            if (_maxAnzahlAbrufVersuche < 1) {
                throw new NumberFormatException();
            }
        } catch (NumberFormatException e) {
            _maxAnzahlAbrufVersuche = 10;
            debug.warning(String.format("Aufrufparameter [-maxAnzahlAbrufVersuche=%s] ist ungültig. Wert wurde für weitere Berechnung auf [-maxAnzahlAbrufVersuche=%d] gesetzt.", _maxAnzahlAbrufVersucheAufrufParameter, _maxAnzahlAbrufVersuche));
        }
    }

    /**
     * Startet die eigentliche Applikation.
     */
    private void run() {
        debug.info(String.format("%n=================================================================%n" + "Programm 'PufferAbruf Langzeitdaten nachfordern' gestartet...%n" + "=================================================================%n"));

        // Auftragsliste holen
        AuftragsListe auftragsListe = AuftragsListe.getInstanz();
        long zeitPunktNeueDatenLueckenFuellen = System.currentTimeMillis() + EINE_STUNDE_IN_MILLISEKUNDEN;

        // Endlosschleife mit Prüfung der Auftragsliste.
        while (true) {
            Auftrag auftrag;

            if ((auftrag = auftragsListe.holeNaechstenAuftrag()) != null) {
                auftrag.ausfuehren();
            } else {

                // Zurzeit ist nichts zu tun, also warten und es dann noch mal versuchen...
                debug.info(String.format("Zur Zeit keine sofort auszuführenden Aufträge in der Auftragsliste. Warte %d Sekunden.", WARTEZEIT_AUF_NAECHSTEN_AUFTRAG / 1000));

                try {
                    Thread.sleep(WARTEZEIT_AUF_NAECHSTEN_AUFTRAG);

                    if (System.currentTimeMillis() > zeitPunktNeueDatenLueckenFuellen) {
                        erzeugeAuftraegeFuerNeueDatenLuecken();
                    }
                } catch (InterruptedException e) {
                    debug.info("Sleep der Auftragsbearbeitung wurde vorzeitig unterbrochen: ", e);
                }
            }
        }
    }
}

