/*
 * Copyright 2018 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.dav.
 *
 * de.bsvrz.dav.dav is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.dav is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.dav.dav.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Martin-Luther-Straße 14
 * 52062 Aachen, Germany
 * phone: +49 241 4090 436
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.dav.main;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Speichert die Informationen um eine aktive Verbindung aufzubauen
 *
 * @author Kappich Systemberatung
 */
public class ApplicationInfo {

	private static final Pattern PATTERN = Pattern.compile("(\\S+):([0-9]+)");
	
	/** Host */
	private final String _address;
	
	/** Port */
	private final int _subAddress;

	/** 
	 * Erstellt ein neues ApplicationInfo-Objekt
	 * @param address    Host
	 * @param subAddress Port
	 */
	public ApplicationInfo(final String address, final int subAddress) {
		_address = address;
		_subAddress = subAddress;
	}

	public static ApplicationInfo parse(final String value) {
		Matcher matcher = PATTERN.matcher(value);
		if(matcher.matches()) {
			return new ApplicationInfo(matcher.group(1), Integer.parseInt(matcher.group(2)));
		}
		else {
			throw new IllegalArgumentException("Unbekanntes Format: \"" + value + "\", gefordert ist host:port, z. B. \"example.com:1234\"");
		}
	}

	/** 
	 * Gibt den Host zurück
	 * @return den Host
	 */
	public String getAddress() {
		return _address;
	}

	/** 
	 * Gibt den Port zurück
	 * @return den Port
	 */
	public int getSubAddress() {
		return _subAddress;
	}

	@Override
	public boolean equals(final Object o) {
		if(this == o) return true;
		if(o == null || getClass() != o.getClass()) return false;

		final ApplicationInfo that = (ApplicationInfo) o;

		if(_subAddress != that._subAddress) return false;
		return _address.equals(that._address);
	}

	@Override
	public int hashCode() {
		int result = _address.hashCode();
		result = 31 * result + _subAddress;
		return result;
	}

	@Override
	public String toString() {
		return _address + ":" + _subAddress;
	}
}
