/*
 * Copyright 2018 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.dav.
 *
 * de.bsvrz.dav.dav is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.dav is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.dav.dav.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Martin-Luther-Straße 14
 * 52062 Aachen, Germany
 * phone: +49 241 4090 436
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.dav.main;

import com.google.common.collect.ImmutableList;
import de.bsvrz.dav.dav.communication.appProtocol.T_A_HighLevelCommunication;
import de.bsvrz.dav.dav.communication.davProtocol.T_T_HighLevelCommunication;

import java.util.*;

/**
 * Eine Map von ID zu Verbindung
 *
 * @author Kappich Systemberatung
 */
public class ConnectionMap<T extends ServerHighLevelCommunication> {
	
	private final Map<Long, T> _communicationsById = new HashMap<>();

	private final Set<T> _unsortedCommunications = new HashSet<>();

	/**
	 * Gibt die Verbindung zum Datenverteiler mit der angegebenen ID zurück
	 * @param id Datenverteiler-ID
	 * @return die Verbindung zum Datenverteiler mit der angegebenen ID, oder {@code null} falls nicht vorhanden
	 */
	public T getConnection(final long id) {
		return _communicationsById.get(id);
	}

	/**
	 * Fügt eine Verbindung hinzu, deren ID unbekannt ist
	 * @param communication Verbindung
	 */
	public void addUnsortedConnection(T communication) {
		_unsortedCommunications.add(communication);
	}

	/**
	 * Merkt sich die angegebene Verbindung unter der angegebenen ID. Die ID wird separat übergeben, da {@link T#getId()} die verbundene ID erst zurückliefert, wenn die Authentifizierung ausreichend abgeschlossen wurde.
	 * Für ausgehende Verbindungen wird der Datenverteiler mit der erwarteten ID sofort eingetragen, damit die Verbindung immer der konfigurierten Verbindung zugeordnet werden kann. (Sollte die ID später nicht der erwarteten ID entsprechen, muss das {@linkplain OutgoingTransmitterConnections#updateId(T_T_HighLevelCommunication)} korrigiert werden}.
	 * Besteht berets eine Verbindung mit dieser ID, wird diese überschrieben.
	 * @param id ID
	 * @param communication Verbindung
	 */
	public void putConnection(final long id, final T communication) {
		if(id == -1) {
			throw new IllegalArgumentException();
		}
		_unsortedCommunications.remove(communication);
		_communicationsById.put(id, communication);
	}

	/**
	 * Entfernt eine Verbindung (zum Beispiel weil diese geschlossen wurde)
	 * @param id ID der Verbindung
	 * @param communication zu entfernende Verbindung unter dieser ID (wenn die gespeicherte Verbindung unter dieser ID nicht dem parameter entspricht, tut diese Methode nichts. Damit wird verhindert, dass die falsche Verbindung entfernt wird)
	 * @return true falls erfolgreich entfernt, sonst false
	 */
	public boolean removeConnection(final long id, final T communication) {
		if(getConnection(id) == communication) {
			return _communicationsById.remove(id) != null;
		}
		return false;
	}

	/**
	 * Entfernt eine Verbindung (zum Beispiel weil diese geschlossen wurde)
	 * @param communication zu entfernende Verbindung unter dieser ID (wenn die gespeicherte Verbindung unter dieser ID nicht dem parameter entspricht, tut diese Methode nichts. Damit wird verhindert, dass die falsche Verbindung entfernt wird)
	 * @return true falls erfolgreich entfernt, sonst false
	 */
	public boolean removeConnection(final T communication) {
		return _unsortedCommunications.remove(communication) || removeConnection(communication.getId(), communication);
	}

	/**
	 * Gibt alle bekannten (typischerweise aktiven) Verbindungen zurück. Diese Methode erzeugt einen nicht änderbaren, Schnappschuss
	 * über den iteriert werden kann, ohne das {@link ConcurrentModificationException}s auftreten können.
	 * 
	 * @return alle aktiven Verbindungen
	 */
	public Collection<T> getAllConnections() {
		return ImmutableList.<T>builder().addAll(_unsortedCommunications).addAll(_communicationsById.values()).build();
	}

	@Override
	public String toString() {
		return _communicationsById.toString();
	}

	public void clear() {
		_communicationsById.clear();
		_unsortedCommunications.clear();
	}
}
