/*
 * Rahmenwerk-Plug-in Darstellungsobjekt "RDS-Symbole"
 * Copyright (C) 2009 BitCtrl Systems GmbH
 * Copyright (C) 2014 Regierungspräsidium Tübingen, Landesstelle für Straßentechnik, Thomas Bräuner
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 * 
 * Landesstelle für Straßentechnik
 * Heilbronner Straße 300 - 302
 * 70469 Stuttgart
 * Phone: +49 0711 8910-200
 * mailto: abteilung9@rpt.bwl.de
 */
package de.bsvrz.buv.plugin.netz.rdssymbole;

import java.util.List;

import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.draw2d.geometry.PrecisionPoint;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.buv.plugin.netz.rdssymbole.util.RdsCode;
import de.bsvrz.buv.plugin.netz.rdssymbole.util.TabelleEinleser;
import de.bsvrz.sys.funclib.bitctrl.geolib.WGS84Polygon;
import de.bsvrz.sys.funclib.bitctrl.geolib.WGS84Punkt;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateEvent;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.parameter.PdRdsMeldung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.util.geolib.WGS84Util;

/**
 * Darstellung einer <em>RDS-Meldung</em> als Symbol.
 * 
 * @author BitCtrl Systems GmbH, thierfelder
 * @author BitCtrl Systems GmbH, Falko Schumann
 * @author Anpassungen: RPT, LST, 95-31, Thomas Bräuner
 */
class RdsMeldungSymbolEmpfaenger extends RdsMeldungEmpfaenger {

	// Ergänzung RDS-Symbole
	// Tabelle mit der Klassifizierung von RDS-Typen.
	private static final TabelleEinleser TABELLE = new TabelleEinleser();
	// Die Figure, um das Symbol ändern zu können
	private final RdsMeldungSymbolFigure figure;
	// Ende Ergänzung
	
	RdsMeldungSymbolEmpfaenger(final RdsMeldungEditPart<?> editPart) {
		super(editPart);
		// Ergänzung RDS-Symbole
		figure = null;
		// Ende Ergänzung
	}

	// Ergänzung RDS-Symbole
	RdsMeldungSymbolEmpfaenger(final RdsMeldungEditPart<?> editPart, RdsMeldungSymbolFigure figure) {
		super(editPart);
		this.figure = figure;
	}
	
	public TabelleEinleser getTabelle() {
		return TABELLE;
	}
	// Ende Ergänzung
	
	@Override
	protected PointList getStreckenabschnitt(final PdRdsMeldung.Daten datum) {
		final RdsMeldungVerortung verortung = RdsConverter.getInstance()
				.verorte(getRdsMeldung());
		if (verortung == null || verortung.getSegmentListe().isEmpty()) {
			return new PointList();
		}

		final List<StrassenSegment> strassenSegmente = verortung
				.getSegmentListe();

		final WGS84Polygon polygon = WGS84Util.ausschneiden(
				strassenSegmente.get(0), verortung.getStartOffset(), -1);
		if (polygon.getKoordinaten().size() <= 1) {
			return new PointList();
		}

		final WGS84Punkt wgs84Punkt = polygon.getKoordinaten().get(0);
		Point point = new PrecisionPoint(wgs84Punkt.getLaenge(),
				wgs84Punkt.getBreite());
		point = getProjektion().umrechnenWeltNachModel(point);

		final WGS84Punkt nextWgs84Punkt = polygon.getKoordinaten().get(1);
		Point nextPoint = new PrecisionPoint(nextWgs84Punkt.getLaenge(),
				nextWgs84Punkt.getBreite());
		nextPoint = getProjektion().umrechnenWeltNachModel(
				new PrecisionPoint(nextWgs84Punkt.getLaenge(), nextWgs84Punkt
						.getBreite()));

		final PointList pointList = new PointList();
		pointList.addPoint(point);
		pointList.addPoint(nextPoint);
		return pointList;
	}
	
	// Ergänzung RDS-Symbole
	@Override
	public void datensatzAktualisiert(final DatensatzUpdateEvent event) {
		if (!editPart.isActive()) {
			return;
		}

		updateZeitstempelLetzterDatensatz(event);

		if (PlatformUI.getWorkbench().getDisplay() != null) {
			PlatformUI.getWorkbench().getDisplay().asyncExec(new Runnable() {

				@Override
				public void run() {
					handleDatensatzAktualisiert((PdRdsMeldung.Daten) event
							.getDatum());
					editPart.refreshLocation();
				}

			});
		}
	}

	private void handleDatensatzAktualisiert(final PdRdsMeldung.Daten datum) {
		if (datum.dContainsDaten()) {
			streckenabschnitt = getStreckenabschnitt(datum);
			
			// Ergänzung RDS-Symbole
			RdsCode priosymb = RdsCode.DEFAULT;
			
			if (datum.getVersion().getVerkehrsInformationen()
					.getEreignis().getEreignisDaten().size() > 0) {
				for (int lauf = 0; lauf < datum.getVersion()
						.getVerkehrsInformationen().getEreignis()
						.getEreignisDaten().size(); lauf++) {
					final int code = datum.getVersion()
							.getVerkehrsInformationen().getEreignis()
							.getEreignisDaten().get(lauf)
							.getEreignisCode().intValue();
					RdsCode symb;
					if (code != 0) {
						symb = getTabelle().getZuordnung().get(code);
					} else {
						symb = RdsCode.RDSMELDUNG;
					}
					if (symb.compareTo(priosymb) < 0) {
						priosymb = symb;
					}
				}
			}
			figure.setImage(getTabelle().getImageByRdsCode(priosymb));
			// Ende Ergänzung
		} else {
			streckenabschnitt = new PointList();
		}
	}
	// Ende Ergänzung

}
