/*
 * Rahmenwerk-Plug-in Darstellungsobjekt "RDS-Symbole"
 * Copyright (C) 2009 BitCtrl Systems GmbH
 * Copyright (C) 2014 Regierungspräsidium Tübingen, Landesstelle für Straßentechnik, Thomas Bräuner
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 * 
 * Landesstelle für Straßentechnik
 * Heilbronner Straße 300 - 302
 * 70469 Stuttgart
 * Phone: +49 0711 8910-200
 * mailto: abteilung9@rpt.bwl.de
 */
package de.bsvrz.buv.plugin.netz.rdssymbole;

import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.buv.plugin.netz.rdssymbole.util.RdsCode;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateEvent;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AtlRdsEreignisQuantitaet;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.parameter.PdRdsMeldung;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Darstellung einer <em>RDS-Meldung</em> als Symbol mit Länge.
 * 
 * @author RPT, LST, 95-31, Thomas Bräuner
 */
class RdsMeldungSymbolMitLaengeEmpfaenger extends RdsMeldungSymbolEmpfaenger {

	// Die Figure, um das Symbol ändern zu können
	private final RdsMeldungSymbolMitLaengeFigure figure;
	
	RdsMeldungSymbolMitLaengeEmpfaenger(final RdsMeldungEditPart<?> editPart, RdsMeldungSymbolMitLaengeFigure figure) {
		super(editPart);
		this.figure = figure;
	}
	
	@Override
	public final void datensatzAktualisiert(final DatensatzUpdateEvent event) {
		if (!editPart.isActive()) {
			return;
		}

		updateZeitstempelLetzterDatensatz(event);

		if (PlatformUI.getWorkbench().getDisplay() != null) {
			PlatformUI.getWorkbench().getDisplay().asyncExec(new Runnable() {

				@Override
				public void run() {
					handleDatensatzAktualisiert((PdRdsMeldung.Daten) event
							.getDatum());
					editPart.refreshLocation();
				}

			});
		}
	}

	private void handleDatensatzAktualisiert(final PdRdsMeldung.Daten datum) {
		if (datum.dContainsDaten()) {
			streckenabschnitt = getStreckenabschnitt(datum);
			
			RdsCode priosymb = RdsCode.DEFAULT;
			Long laenge = null;
			
			if (datum.getVersion().getVerkehrsInformationen()
					.getEreignis().getEreignisDaten().size() > 0) {
				for (int lauf = 0; lauf < datum.getVersion()
						.getVerkehrsInformationen().getEreignis()
						.getEreignisDaten().size(); lauf++) {
					final int code = datum.getVersion()
							.getVerkehrsInformationen().getEreignis()
							.getEreignisDaten().get(lauf)
							.getEreignisCode().intValue();
					RdsCode symb;
					if (code != 0) {
						symb = getTabelle().getZuordnung().get(code); 
					} else {
						symb = RdsCode.RDSMELDUNG;
					}
					if (symb.compareTo(priosymb) < 0) {
						priosymb = symb;
						// Gegebenenfalls die Länge eines Staus berechnen.
						if (priosymb == RdsCode.STAU) {
							for (final AtlRdsEreignisQuantitaet quantitaet : datum
									.getVersion().getVerkehrsInformationen()
									.getEreignis().getEreignisDaten().get(lauf)
									.getEreignisQuantitaet()) {
								if ("rdsQuantität.LEN".equals(quantitaet.getQuantitaetsKennung().getPid())) {
									try {
										laenge = Long.parseLong(quantitaet.getQuantitaetsWert());
									} catch (final NumberFormatException e) {
										Debug.getLogger().error("Falsches Zahlenformat!", e);
									}
								}
							}

							if (laenge == null) {
								laenge = parseCodeStr(datum
										.getVersion().getVerkehrsInformationen()
										.getEreignis().getEreignisDaten().get(lauf)
										.getEreignisCode().toString());
							}
						}
					}
				}
			}
			figure.setImage(getTabelle().getImageByRdsCode(priosymb), laenge);
		} else {
			streckenabschnitt = new PointList();
		}
	}
	
	/**
	 * die Funktion analysiert den übergebenen String und versucht eine
	 * Länge in Kilometern zu ermitteln. Diese Funktion wird verwendet,
	 * wenn die Quantität in der RDS-Meldung nicht geliefert wurde.
	 * übergeben wird in der Regel die Bezeichnung des
	 * RDS-Meldungscodes, z.B. "Stau 2 km". Der text wird über die
	 * Leerzeichen in Teile aufgeteilt. Wenn einer der Teile "km" ist
	 * wird der vorherige als Zahl interpretiert.
	 * 
	 * @param string
	 *            der zu analysierende String
	 * @return der Wert oder <code>null</code>, wenn keine Länge
	 *         ermittelt werden konnte
	 */
	private Long parseCodeStr(final String string) {
		Long result = null;
		if (string != null) {
			final String[] parts = string.split(" ");
			for (int idx = 0; idx < parts.length; idx++) {
				if (parts[idx].equalsIgnoreCase("km")) {
					if (idx > 0) {
						try {
							result = Long.parseLong(parts[idx - 1]);
						} catch (final NumberFormatException e) {
							// Eintrag wird ignoriert
							Debug.getLogger().finest("", e);
						}
					}
				}
			}
		}
		return result;
	}

}
