/*
 * Rahmenwerk-Plug-in "Benutzerverwaltung/Zugriffsrechte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.benutzervew.data;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import de.bsvrz.buv.plugin.benutzervew.internal.RahmenwerkService;
import de.bsvrz.buv.rw.basislib.Rahmenwerk;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.Data.Array;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.daf.main.ReceiverRole;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.DynamicObject;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.puk.param.lib.daten.UrlasserInfo;

public class Benutzer extends SystemObjectWrapper {

	private static final String ATG_BENUTZER_EIGENSCHAFTEN = "atg.benutzerEigenschaften";
	private static final String ATG_BENUTZER_PARAMETER = "atg.benutzerParameter";
	private static final String ATG_BERECHTIGUNGSKLASSEN = "atg.berechtigungsklassen";

	private final BenutzerInfo info = new BenutzerInfo();
	private final Set<BerechtigungsKlasse> berechtigungsKlassen = new LinkedHashSet<>();

	private final DataDescription berechtigungsDesc;
	private final DataDescription berechtigungsDescVorgabe;

	public Benutzer(final SystemObject systemObject) {
		super(systemObject);

		final ClientDavInterface dav = RahmenwerkService.getService().getRahmenWerk().getDavVerbindung();
		final DataModel dataModel = systemObject.getDataModel();

		readBenutzerInfo(((DynamicObject) systemObject)
				.getConfigurationData(dataModel.getAttributeGroup(ATG_BENUTZER_EIGENSCHAFTEN)));

		if (RahmenwerkService.getService().getRahmenWerk().usesBerechtigungenNeu()) {
			berechtigungsDesc = new DataDescription(dataModel.getAttributeGroup(ATG_BERECHTIGUNGSKLASSEN),
					getAspectParameterSoll());
			berechtigungsDescVorgabe = new DataDescription(dataModel.getAttributeGroup(ATG_BERECHTIGUNGSKLASSEN),
					getAspectParameterVorgabe());
		} else {
			berechtigungsDesc = new DataDescription(dataModel.getAttributeGroup(ATG_BENUTZER_PARAMETER),
					getAspectParameterSoll());
			berechtigungsDescVorgabe = new DataDescription(dataModel.getAttributeGroup(ATG_BENUTZER_PARAMETER),
					getAspectParameterVorgabe());
		}
		updateBerechtigungsKlassen(dav.getData(systemObject, berechtigungsDesc, 0));
		dav.subscribeReceiver((results) -> updateBerechtigungsKlassen(results), systemObject, berechtigungsDesc,
				ReceiveOptions.normal(), ReceiverRole.receiver());
	}

	private void readBenutzerInfo(final Data data) {
		if (data == null) {
			return;
		}

		info.setVorname(data.getTextValue("vorname").getText());
		info.setZweiterVorname(data.getTextValue("zweiterVorname").getText());
		info.setNachname(data.getTextValue("nachname").getText());
		info.setOrganisation(data.getTextValue("organisation").getText());
		info.setEmailAdresse(data.getTextValue("emailAdresse").getText());
	}

	private void updateBerechtigungsKlassen(final ResultData... results) {
		for (final ResultData result : results) {
			berechtigungsKlassen.clear();
			if (result.hasData()) {
				if (ATG_BERECHTIGUNGSKLASSEN.equals(result.getDataDescription().getAttributeGroup().getPid())) {
					final Array array = result.getData().getArray("Berechtigungsklassen");
					for (int idx = 0; idx < array.getLength(); idx++) {
						final BerechtigungsKlasse berechtigungsklasse = RahmenwerkService.getService()
								.getBenutzerverwaltung()
								.getBerechtigungsklasse(array.getReferenceValue(idx).getSystemObjectPid());
						if (berechtigungsklasse != null) {
							berechtigungsKlassen.add(berechtigungsklasse);
						}
					}
				} else if (ATG_BENUTZER_PARAMETER.equals(result.getDataDescription().getAttributeGroup().getPid())) {
					final BerechtigungsKlasse berechtigungsklasse = RahmenwerkService.getService()
							.getBenutzerverwaltung().getBerechtigungsklasse(
									result.getData().getReferenceValue("berechtigungsklasse").getSystemObjectPid());
					if (berechtigungsklasse != null) {
						berechtigungsKlassen.add(berechtigungsklasse);
					}
				}

				RahmenwerkService.getService().getBenutzerverwaltung().fireBenutzerChanged(this);
			}
		}
	}

	public void sendeBerechtigungsKlassen(Collection<BerechtigungsKlasse> klassen, UrlasserInfo urlasser) {

		final Rahmenwerk rahmenWerk = RahmenwerkService.getService().getRahmenWerk();
		if (!rahmenWerk.isOnline()) {
			return;
		}

		final ClientDavInterface dav = rahmenWerk.getDavVerbindung();
		final AttributeGroup atg = berechtigungsDescVorgabe.getAttributeGroup();
		final Data data = dav.createData(atg);

		if (ATG_BENUTZER_PARAMETER.equals(atg.getPid())) {
			if (!klassen.isEmpty()) {
				data.getReferenceValue("berechtigungsklasse")
						.setSystemObject(klassen.iterator().next().getSystemObject());
			}
		} else if (ATG_BERECHTIGUNGSKLASSEN.equals(atg.getPid())) {
			final Array array = data.getArray("Berechtigungsklassen");
			array.setLength(klassen.size());
			int idx = 0;
			for (final BerechtigungsKlasse klasse : klassen) {
				array.getReferenceValue(idx).setSystemObject(klasse.getSystemObject());
				idx++;
			}
		}

		if (urlasser != null) {
			final Data urlasserItem = data.getItem("Urlasser");
			urlasserItem.getReferenceValue("BenutzerReferenz").setSystemObject(urlasser.getBenutzer());
			urlasserItem.getTextValue("Ursache").setText(urlasser.getUrsache());
			urlasserItem.getTextValue("Veranlasser").setText(urlasser.getVeranlasser());
		}

		RahmenwerkService.getService().getBenutzerverwaltung().getSender()
				.sende(new ResultData(getSystemObject(), berechtigungsDescVorgabe, 0, data));
	}

	public Set<BerechtigungsKlasse> getBerechtigungsKlassen() {
		return berechtigungsKlassen;
	}

	public String getNachname() {
		return info.getNachname();
	}

	public String getVorname() {
		return info.getVorname();
	}

	public String getZweiterVorname() {
		return info.getZweiterVorname();
	}

	public String getOrganisation() {
		return info.getOrganisation();
	}

	public String getEmailAdresse() {
		return info.getEmailAdresse();
	}

	public String getBerechtigungsKlassenAsString() {
		final List<BerechtigungsKlasse> bkListe = new ArrayList<>(berechtigungsKlassen);
		Collections.sort(bkListe, (o1, o2) -> (o1.getName().compareTo(o2.getName())));
		return Arrays.toString(bkListe.toArray()).substring(1).replace("]", "");
	}
}
