/*
 * Rahmenwerk-Plug-in "Benutzerverwaltung/Zugriffsrechte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.benutzervew.data;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.CopyOnWriteArrayList;

import de.bsvrz.buv.plugin.benutzervew.internal.RahmenwerkService;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataAndATGUsageInformation;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.ConfigurationArea;
import de.bsvrz.dav.daf.main.config.ConfigurationChangeException;
import de.bsvrz.dav.daf.main.config.ConfigurationTaskException;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.DynamicObject;
import de.bsvrz.dav.daf.main.config.DynamicObjectType;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.dav.daf.main.config.management.UserAdministration;
import de.bsvrz.puk.param.lib.daten.UrlasserInfo;
import de.bsvrz.sys.funclib.bitctrl.daf.DavTools;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.dynobj.DynObjektException;
import de.bsvrz.sys.funclib.dynobj.DynamischeObjekte;

/**
 * Verwaltet die Benutzer des Datenverteilers.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 */
public final class Benutzerverwaltung {

	/** Der Standardpräfix für die PID eines neuen Benutzer. */
	public static final String PRAEFIX_PID = "benutzer.";

	/** PID der Berechtigungsklasse ohne jede Zugriffsrechte. */
	public static final String PID_KLASSE_KEIN_ZUGRIFF = "berechtigungsklasse.keinZugriff";

	/** PID der Berechtigungsklasse mit allen Zugriffsrechten. */
	public static final String PID_KLASSE_ADMINISTRATOR = "berechtigungsklasse.administrator";

	private static final String BENUTZER_TYP_PID = "typ.benutzer";

	private static final Debug LOGGER = Debug.getLogger();

	private ClientDavInterface dav;

	private final Collection<BenutzerListener> listeners = new CopyOnWriteArrayList<>();

	private final Map<String, Benutzer> benutzerMap = new LinkedHashMap<>();

	private final Map<String, BerechtigungsKlasse> berechtigungsKlassenMap = new LinkedHashMap<>();

	private DatenVerteiler datenVerteiler;

	private final Map<String, Region> regionenMap = new LinkedHashMap<>();

	private final Map<String, Rolle> rollenMap = new LinkedHashMap<>();

	private final Sender sender = new Sender();

	/**
	 * Initialisierung.
	 *
	 * @param factory die zum erzeugen von Systemobjekten verwendete ObjektFactory
	 */
	public void connect(final ClientDavInterface connection) {

		if (connection == null) {
			disconnect();
			return;
		}

		this.dav = connection;
		sender.setDav(connection);

		datenVerteiler = new DatenVerteiler(connection.getLocalDav());

		SystemObjectType berechtigungsKlassenObjectTyp;
		SystemObjectType regionObjectTyp;
		SystemObjectType rolleObjectTyp;

		if (RahmenwerkService.getService().getRahmenWerk().usesBerechtigungenNeu()) {
			berechtigungsKlassenObjectTyp = connection.getDataModel().getType("typ.berechtigungsklasseNeu");
			regionObjectTyp = connection.getDataModel().getType("typ.zugriffsRegionNeu");
			rolleObjectTyp = connection.getDataModel().getType("typ.zugriffsRolleNeu");
		} else {
			berechtigungsKlassenObjectTyp = connection.getDataModel().getType("typ.berechtigungsklasse");
			regionObjectTyp = connection.getDataModel().getType("typ.zugriffsRegion");
			rolleObjectTyp = connection.getDataModel().getType("typ.zugriffsRolle");
		}
		regionObjectTyp.getElements().stream().forEach(o -> regionenMap.put(o.getPid(), new Region(o)));
		rolleObjectTyp.getElements().stream().forEach(o -> rollenMap.put(o.getPid(), new Rolle(o)));
		berechtigungsKlassenObjectTyp.getElements().stream()
				.forEach(o -> berechtigungsKlassenMap.put(o.getPid(), new BerechtigungsKlasse(o)));

		// Listener auf Änderungen der Menge der Benutzer
		final DynamicObjectType benutzerObjectTyp = (DynamicObjectType) connection.getDataModel()
				.getType(BENUTZER_TYP_PID);

		/* Benutzerliste initialisieren. */
		benutzerObjectTyp.getElements().stream().forEach(o -> benutzerAngelegt((DynamicObject) o));
		benutzerObjectTyp.addObjectCreationListener(o -> benutzerAngelegt(o));
		benutzerObjectTyp.addInvalidationListener(o -> benutzerEntfernt(o));

		LOGGER.info("Schnittstelle zur Nutzerverwaltung initialisiert.");
	}

	public void disconnect() {
		sender.setDav(null);
		benutzerMap.clear();
	}

	private void benutzerAngelegt(final DynamicObject neuesBenutzerObjekt) {
		final Benutzer benutzer = new Benutzer(neuesBenutzerObjekt);
		benutzerMap.put(neuesBenutzerObjekt.getPid(), benutzer);
		fireBenutzerAdded(benutzer);
	}

	private void benutzerEntfernt(final DynamicObject zuentfernendesBenutzerObjekt) {
		final Benutzer benutzer = benutzerMap.remove(zuentfernendesBenutzerObjekt.getPid());
		if (benutzer != null) {
			fireBenutzerRemoved(benutzer);
		}
	}

	/**
	 * Registriert einen Listener an.
	 *
	 * @param l ein Listener.
	 */
	public void addBenutzerListener(final BenutzerListener l) {
		listeners.add(l);
	}

	/**
	 * Meldet einen Listener wieder ab.
	 *
	 * @param l ein Listener.
	 */
	public void removeBenutzerListener(final BenutzerListener l) {
		listeners.remove(l);
	}

	/**
	 * Benachrichtigt die angmeldeten Listener darüber, dass ein neuer Benutzer
	 * angelegt wurde.
	 *
	 * @param benutzer der betroffene Benutzer.
	 */
	private void fireBenutzerAdded(final Benutzer benutzer) {
		for (final BenutzerListener l : listeners) {
			l.benutzerAdded(benutzer);
		}
	}

	/**
	 * Benachrichtigt die angmeldeten Listener darüber, dass ein Benutzer gelöscht
	 * wurde.
	 *
	 * @param benutzer der betroffene Benutzer.
	 */
	protected synchronized void fireBenutzerRemoved(final Benutzer benutzer) {
		for (final BenutzerListener l : listeners) {
			l.benutzerRemoved(benutzer);
		}
	}

	/**
	 * Benachrichtigt die angmeldeten Listener darüber, dass die Berechtigungsklasse
	 * eines Benutzer geändert wurde.
	 *
	 * @param benutzer            der betroffene Benutzer.
	 * @param berechtigungsklasse die Berechtigungsklasse des Benutzers.
	 */
	protected synchronized void fireBenutzerChanged(final Benutzer benutzer) {

		for (final BenutzerListener l : listeners) {
			l.benutzerChanged(benutzer);
		}
	}

	public List<AngemeldeteApplikation> getAnmeldungen(final Benutzer benutzer) {
		return datenVerteiler.getAngemeldeteApplikationenFor(benutzer);
	}

	public boolean isAngemeldet(final Benutzer benutzer) {
		return !datenVerteiler.getAngemeldeteApplikationenFor(benutzer).isEmpty();
	}

	/**
	 * Gibt den lokal angemeldeten Benutzer zurück.
	 *
	 * @return der angemeldete Benutzer.
	 */
	public Benutzer getAngemeldetenBenutzer() {
		return benutzerMap.get(RahmenwerkService.getService().getRahmenWerk().getBenutzer().getPid());
	}

	public Collection<Benutzer> getBenutzer() {
		return Collections.unmodifiableCollection(benutzerMap.values());
	}

	/**
	 * Gibt eine Liste aller Benutzer zurück, die einer bestimmten
	 * Berechtigungsklasse angehören.
	 *
	 * @param klasse eine Berechtigungsklasse.
	 * @return die Benutzerliste.
	 */
	public List<Benutzer> getBenutzer(final BerechtigungsKlasse klasse) {
		final List<Benutzer> benutzerListe = new ArrayList<>();

		for (final Benutzer benutzer : benutzerMap.values()) {
			if (isBerechtigungsklasse(benutzer, klasse)) {
				benutzerListe.add(benutzer);
			}
		}

		return benutzerListe;
	}

	/**
	 * Gibt eine Liste aller Berechtigungsklassen im System zurück.
	 *
	 * @return die Liste der Berechtigungsklassen.
	 */
	public Collection<BerechtigungsKlasse> getBerechtigungsklassen() {
		return berechtigungsKlassenMap.values();
	}

	/**
	 * Pr&uuml;ft, ob ein Benutzer Administratorrechte am DAV besitzt.
	 * Administratoren d&uuml;rfen u. a. Benutzer l&ouml;schen, Passworte anderer
	 * Benuzter &auml;ndern und Autorisierungspassworte f&uuml;r andere Benuzter
	 * anlegen.<br>
	 * Da sich der Zustand des Admin-Flags derzeit nicht direkt &uuml;ber das
	 * DAV-API ermitteln l&auml;&szlig;t, hilft folgender Trick: Man versucht den
	 * Benuzter mit sich selbst zum Admin zu machen, klappt dies, so ist er bereits
	 * Admin gewesen, ansonsten ist er kein Admin.
	 *
	 * @param loginname ein Nutzername
	 * @param passwort  das Passwort des Benutzers
	 * @return {@code true}, wenn der Bennutzer Administratorrechte besitzt.
	 */
	public boolean isDAVAdmin(final String loginname, final String passwort) {
		final DataModel modell = dav.getDataModel();
		final UserAdministration userAdmin = modell.getUserAdministration();
		try {
			return userAdmin.isUserAdmin(loginname, passwort, loginname);
		} catch (final ConfigurationTaskException e) {
			LOGGER.warning(e.getLocalizedMessage());
		}

		return false;
	}

	/**
	 * Prüft ob ein bestimmter Benutzer zu einer bestimmten Berechtigungsklasse
	 * gehört.
	 *
	 * @param benutzer ein Benutzer.
	 * @param klasse   eine Berechtuigungsklasse.
	 * @return {@code true}, wenn der Benutzer zu der Berechtigungsklasse gehört.
	 */
	private boolean isBerechtigungsklasse(final Benutzer benutzer, final BerechtigungsKlasse klasse) {
		return benutzer.getBerechtigungsKlassen().contains(klasse);
	}

	/**
	 * Bestimmt die aktuelle Berechtigungsklasse des Benutzers.
	 *
	 * @param benutzer ein Benutzer.
	 * @return die Berechtigungsklasse des Benutzers.
	 */
	public BerechtigungsKlasse getBerechtigungsklasse(final Benutzer benutzer) {
		// TODO Wie mit mehreren Klassen umgehen?
		final Set<BerechtigungsKlasse> klassen = benutzer.getBerechtigungsKlassen();
		if (klassen.isEmpty()) {
			return null;
		}

		return klassen.iterator().next();
	}

	/**
	 * Prüft ob ein bestimmter Benutzer eine bestimmte Rolle in einer Region hat.
	 *
	 * @param benutzer ein Benutzer.
	 * @param rolle    eine Zugriffsrolle.
	 * @param region   eine Zugriffsregion. Wenn {@code null}, wird die Region
	 *                 ignoriert und nur die Rolle geprüpft.
	 * @return {@code true}, wenn der Benutzer in der Region die angegebene Rolle
	 *         hat.
	 */
	public boolean isRolleUndRegion(final Benutzer benutzer, final Rolle rolle, final Region region) {
		if (benutzer == null) {
			throw new IllegalArgumentException("Benutzer darf nicht null sein.");
		}
		if (rolle == null) {
			throw new IllegalArgumentException("Zugriffsrolle darf nicht null sein.");
		}

		final Set<BerechtigungsKlasse> klassen = benutzer.getBerechtigungsKlassen();
		if (klassen.isEmpty()) {
			LOGGER.info("Dem Benutzer " + benutzer + " ist keine Berechtigungsklasse zugeordnet.");
			return false;
		}

		for (final BerechtigungsKlasse klasse : klassen) {
			for (final RolleRegionPaar paar : klasse.getRolleRegionenPaare()) {
				if (region != null) {
					if (region.equals(paar.getRegion()) && rolle.equals(paar.getRolle())) {
						return true;
					}
				} else {
					if (rolle.equals(paar.getRolle())) {
						return true;
					}
				}
			}
		}
		return false;
	}

	/**
	 * Sucht eine bestimmte Berechtigungsklasse.
	 *
	 * @param pid die PID einer Berechtigungsklasse.
	 * @return die Berechtigungsklasse oder {@code null}, wenn zu der PID keine
	 *         existiert.
	 */
	public BerechtigungsKlasse getBerechtigungsklasse(final String pid) {
		return berechtigungsKlassenMap.get(pid);
	}

	/**
	 * Gibt eine Liste aller Zugriffsrollen im System zurück.
	 *
	 * @return die Liste der Zugriffsrollen.
	 */
	public Collection<Rolle> getRollen() {
		return Collections.unmodifiableCollection(rollenMap.values());
	}

	/**
	 * Prüft ob ein bestimmter Benutzer existiert und gibt ihn zurück.
	 *
	 * @param loginname der eindeutige Benutzername (Loginname).
	 * @return der Benutzer oder {@code null}, wenn kein Benutzer mit dem
	 *         angegebenen Namen existiert.
	 */
	public Benutzer getBenutzer(final String loginname) {
		if ((loginname == null) || (loginname.length() == 0)) {
			throw new IllegalArgumentException("Der Benutzername darf weder null noch ein Leerstring sein.");
		}

		for (final Benutzer benutzer : benutzerMap.values()) {
			if (benutzer.getName().equals(loginname)) {
				return benutzer;
			}
		}

		return null;
	}

	public Benutzer getBenutzerWithPid(String benutzerPid) {
		return benutzerMap.get(benutzerPid);
	}

	/**
	 * Sucht alle Benutzer auf die bestimmte Kriterien zutreffen. Die Suchkriterien
	 * dürfen auch {@code null} sein, dies wird als Wildcard "alle" gedeutet. Die
	 * einzelnen Kriterien werden "oder"-verknüpft.
	 *
	 * @param nachname       der Nachname der gesuchten Benutzer.
	 * @param vorname        der Vorname der gesuchten Benutzer.
	 * @param zweiterVorname der zweite Vorname der gesuchten Benutzer.
	 * @param organisation   die Organisation der gesuchten Benutzer.
	 * @param email          die E-Mail-Adresse der gesuchten Benutzer.
	 * @return die Liste der gefundenen Benutzer, niemals {@code null}.
	 */
	public List<Benutzer> sucheBenutzer(final String nachname, final String vorname, final String zweiterVorname,
			final String organisation, final String email) {
		final List<Benutzer> benutzerListe = new ArrayList<>();

		for (final Benutzer benutzer : getBenutzer()) {
			boolean ok = false;

			if ((nachname == null) || nachname.equals(benutzer.getNachname())) {
				ok = true;
			} else if ((vorname == null) || vorname.equals(benutzer.getVorname())) {
				ok = true;
			} else if ((zweiterVorname == null) || zweiterVorname.equals(benutzer.getZweiterVorname())) {
				ok = true;
			} else if ((organisation == null) || organisation.equals(benutzer.getOrganisation())) {
				ok = true;
			} else if ((email == null) || email.equals(benutzer.getEmailAdresse())) {
				ok = true;
			}

			if (ok) {
				benutzerListe.add(benutzer);
			}
		}

		return benutzerListe;
	}

	/**
	 * Legt einen neuen Benutzer an. Das Anlegen erfolgt unabh&auml;ngig von
	 * Berechtigungsklassen nur unter Pr&uuml;fung der entsprechenden Rechte am DAV.
	 *
	 * @param adminLoginname der Name des Administrators der die Aktion ausführt.
	 * @param adminPasswort  das Anmeldekennwort des Administrators der die Aktion
	 *                       ausführt.
	 * @param benutzerInfo   die Eigenschaften des neuen Benutzers.
	 * @param adminRechte    <code>true</code>, wenn der neue Benutzer die Rechte
	 *                       eines Administrators besitzen soll; <code>false</code>,
	 *                       wenn der Benutzer keine speziellen Rechte besitzen
	 *                       soll.
	 * @return der neue Benutzer
	 * @throws KeineRechteException    wenn die Benutzerrechte für diese Aktion
	 *                                 nicht ausreichen.
	 * @throws BenutzerChangeException wenn beim Anlegen des Benutzers ein Fehler
	 *                                 eintrat.
	 */
	public Benutzer anlegenBenutzer(String adminLoginname, String adminPasswort, String loginName,
			BenutzerInfo benutzerInfo, boolean adminRechte) throws KeineRechteException, BenutzerChangeException {
		if (!isDAVAdmin(adminLoginname, adminPasswort)) {
			throw new KeineRechteException(
					"Sie verfügen nicht über ausreichend Rechte zum Anlegen eines neuen Benutzer.");
		}

		final String pid;
		Benutzer benutzer = null;

		pid = DavTools.generierePID(loginName, PRAEFIX_PID);
		try {
			benutzer = benutzerObjektAnlegen(pid, loginName, benutzerInfo);

			final DataModel modell = dav.getDataModel();
			final UserAdministration userAdmin = modell.getUserAdministration();

			final DynamicObjectType type = (DynamicObjectType) dav.getDataModel().getType(BENUTZER_TYP_PID);

			final ConfigurationArea konfigurationsBereich = DynamischeObjekte.getInstanz(dav)
					.getKonfigurationsBereich(type);

			userAdmin.createNewUser(adminLoginname, adminPasswort, loginName, pid, benutzerInfo.getPasswort(),
					adminRechte, konfigurationsBereich.getPid());
		} catch (final ConfigurationTaskException ex) {
			throw new BenutzerChangeException("Der Benutzer '" + loginName + "' konnte nicht angelegt werden.", ex);
		} catch (final DynObjektException ex) {
			throw new BenutzerChangeException("Der Benutzer '" + loginName + "' konnte nicht angelegt werden.", ex);
		}

		return benutzer;
	}

	private Benutzer benutzerObjektAnlegen(final String pid, final String name, final BenutzerInfo info)
			throws DynObjektException {

		// Konfigurationsdatensatz erzeugen
		final AttributeGroup benutzerEigenschaftenAtg = dav.getDataModel()
				.getAttributeGroup("atg.benutzerEigenschaften");
		final Aspect eigenschaftenAsp = dav.getDataModel().getAspect("asp.eigenschaften");
		final Data datenSatz = dav.createData(benutzerEigenschaftenAtg);
		datenSatz.getTextValue("vorname").setText(info.getVorname());
		datenSatz.getTextValue("zweiterVorname").setText(info.getZweiterVorname());
		datenSatz.getTextValue("nachname").setText(info.getNachname());
		datenSatz.getTextValue("organisation").setText(info.getOrganisation());
		datenSatz.getTextValue("emailAdresse").setText(info.getEmailAdresse());

		final DataAndATGUsageInformation configData = new DataAndATGUsageInformation(
				benutzerEigenschaftenAtg.getAttributeGroupUsage(eigenschaftenAsp), datenSatz);
		final SystemObjectType type = dav.getDataModel().getType(BENUTZER_TYP_PID);
		return new Benutzer(DynamischeObjekte.getInstanz(dav).erzeugeObjekt((DynamicObjectType) type, name, pid,
				Collections.singleton(configData)));
	}

	/**
	 * Deaktiviert und löscht einen Benutzer. Das entsprechende Systemobjekt wird
	 * invalidiert.
	 *
	 * @param adminLoginname der Name des Administrators der die Aktion ausführt.
	 * @param adminPasswort  das Anmeldekennwort des Administrators der die Aktion
	 *                       ausführt. <em>Wird derzeit ignoriert, da es nicht
	 *                       benötigt wird!</em>
	 * @param benutzer       der zu löschende Benutzers.
	 * @throws KeineRechteException    wenn die Benutzerrechte für diese Aktion
	 *                                 nicht ausreichen.
	 * @throws BenutzerChangeException wenn beim Löschen des Benutzers ein Fehler
	 *                                 eintrat.
	 */
	public void entfernenBenutzer(final String adminLoginname, final String adminPasswort, final Benutzer benutzer)
			throws KeineRechteException, BenutzerChangeException {
		if (!isDAVAdmin(adminLoginname, adminPasswort)) {
			throw new KeineRechteException("Sie verfügen nicht über ausreichend Rechte zum Löschen eines Benutzer.");
		}

		try {
			benutzer.getSystemObject().invalidate();
			final UserAdministration userAdministration = dav.getDataModel().getUserAdministration();
			userAdministration.deleteUser(adminLoginname, adminPasswort, benutzer.getName());
		} catch (final ConfigurationChangeException ex) {
			throw new BenutzerChangeException(
					"Das Benutzerobjekt " + benutzer.getName() + " konnte nicht gelöscht werden.", ex);
		} catch (final ConfigurationTaskException ex) {
			throw new BenutzerChangeException(
					"Die Benutzerdaten " + benutzer.getName() + " konnten nicht gelöscht werden.", ex);
		}
	}

	/**
	 * Ändert die Berechtigungsklasse eines Benutzer.
	 *
	 * @param adminLoginname der Name des Administrators der die Aktion ausführt.
	 * @param adminPasswort  das Anmeldekennwort des Administrators der die Aktion
	 *                       ausführt.
	 * @param benutzer       der zu deaktivierende Benutzers.
	 * @param klasse         die zu setzende Berechtigungsklasse.
	 * @param urlasser       der Nutzername des Urlassers.
	 * @param ursache        Ursache zur Urlasserinformation
	 * @param veranlasser    Veranlasser zur Urlasserinformation
	 * @throws KeineRechteException    wenn die Benutzerrechte für diese Aktion
	 *                                 nicht ausreichen.
	 * @throws BenutzerChangeException wenn beim Ändern der Benutzerrechte ein
	 *                                 Fehler eintrat.
	 */
	public void setBerechtigungsklassen(final String adminLoginname, final String adminPasswort,
			final Benutzer benutzer, Collection<BerechtigungsKlasse> klassen, UrlasserInfo urlasserInfo)
			throws KeineRechteException, BenutzerChangeException {
		if (!isDAVAdmin(adminLoginname, adminPasswort)) {
			throw new KeineRechteException(
					"Sie verfügen nicht über ausreichend Rechte zum Ändern der Berechtigungsklasse eines Benutzers.");
		}
		benutzer.sendeBerechtigungsKlassen(klassen, urlasserInfo);
	}

	/**
	 * Ändert die Berechtigungsklasse eines Benutzer.
	 *
	 * @param adminLoginname der Name des Administrators der die Aktion ausführt.
	 * @param adminPasswort  das Anmeldekennwort des Administrators der die Aktion
	 *                       ausführt.
	 * @param benutzer       der zu deaktivierende Benutzers.
	 * @param passworte      Liste der Einmalpassworte
	 * @throws KeineRechteException    wenn die Benutzerrechte für diese Aktion
	 *                                 nicht ausreichen.
	 * @throws BenutzerChangeException wenn beim Ändern der Benutzerrechte ein
	 *                                 Fehler eintrat.
	 */
	public SortedMap<Integer, String> setEinmalPassworte(final String adminLoginname, final String adminPasswort,
			final String loginName, final List<String> passworte) throws KeineRechteException, BenutzerChangeException {
		if (!isDAVAdmin(adminLoginname, adminPasswort)) {
			throw new KeineRechteException(
					"Sie verfügen nicht über ausreichend Rechte zum Ändern der Berechtigungsklasse eines Benutzers.");
		}

		try {
			final DataModel modell = dav.getDataModel();
			final UserAdministration userAdmin = modell.getUserAdministration();

			final String[] passwortArray = passworte.toArray(new String[passworte.size()]);
			final int firstId = userAdmin.createOneTimePasswords(adminLoginname, adminPasswort, loginName,
					passwortArray);

			final SortedMap<Integer, String> result = new TreeMap<>();
			for (int idx = 0; idx < passwortArray.length; idx++) {
				result.put(firstId + idx, passwortArray[idx]);
			}
			return result;

		} catch (final ConfigurationChangeException ex) {
			throw new BenutzerChangeException(
					"Die Einmalpassworte für " + loginName + " konnten nicht angelegt werden.", ex);
		} catch (final ConfigurationTaskException ex) {
			throw new BenutzerChangeException(
					"Die Einmalpassworte für " + loginName + " konnten nicht angelegt werden.", ex);
		}
	}

	/**
	 * Prüft ein neues Passwort auf seine Sicherheit. Die Funktion wertet die für
	 * die Vergabe von Passworten definierten Parameter aus und prüft, ob das
	 * Passwort diesen Kriterien entspricht.
	 *
	 * @param passwort     das Passwort.
	 * @param benutzer     der Nutzer, für den das Passwort verwendet werden soll.
	 * @param passwortInfo die Sicherheitskriterien.
	 * @return {@code null}, wenn das Passwort sicher ist, sonst eine
	 *         Fehlerbeschreibung.
	 *
	 */
	public String checkPasswort(final String passwort, final Benutzer benutzer, final PasswortInfo passwortInfo) {
		if (passwort == null) {
			throw new IllegalArgumentException("Passwort darf nicht null sein.");
		}

		final long minLaenge = passwortInfo.getMinLaenge();
		if (minLaenge > 0) {
			if (passwort.length() < minLaenge) {
				return "Das Passwort muss mindestens " + minLaenge + " Zeichen lang sein.";
			}
		}

		final String lowerPasswort = passwort.toLowerCase(Locale.getDefault());
		if (passwortInfo.isGemischteZeichen()) {
			final int laenge = lowerPasswort.length();
			int anzahl = 0;
			for (int i = 0; i < laenge; i++) {
				if (lowerPasswort.substring(i, i + 1).matches("[a-z]")) {
					anzahl++;
				}
			}

			if ((anzahl == 0) || (anzahl == laenge)) {
				return "Das Passwort muss außer Buchstaben auch Zahlen oder Sonderzeichen enthalten.";
			}
		}

		if ((benutzer != null) && passwortInfo.isVergleicheBenutzerdaten()) {
			final String loginname = benutzer.getName().toLowerCase(Locale.getDefault());

			final String nachname = benutzer.getNachname().toLowerCase(Locale.getDefault());
			final String vorname = benutzer.getVorname().toLowerCase(Locale.getDefault());

			if (loginname.contains(lowerPasswort.subSequence(0, lowerPasswort.length() - 1))) {
				return "Das Passwort darf nicht im Loginnamen enthalten sein.";
			}
			if (lowerPasswort.contains(loginname.subSequence(0, loginname.length() - 1))) {
				return "Das Passwort darf nicht den Loginnamen enthalten.";
			}

			if (nachname.contains(lowerPasswort.subSequence(0, lowerPasswort.length() - 1))) {
				return "Das Passwort darf nicht im Nachnamen enthalten sein.";
			}
			if (lowerPasswort.contains(nachname.subSequence(0, nachname.length() - 1))) {
				return "Das Passwort darf nicht den Nachnamen enthalten.";
			}

			if (lowerPasswort.contains(vorname.subSequence(0, vorname.length() - 1))) {
				return "Das Passwort darf nicht im Vornamen enthalten sein.";
			}
			if (vorname.contains(lowerPasswort.subSequence(0, lowerPasswort.length() - 1))) {
				return "Das Passwort darf nicht den Vornamen enthalten.";
			}
		}

		// Alle Tests bestanden
		return null;
	}

	/**
	 * &Auml;ndert das Anmeldekennwort eines Benutzer. Das &Auml;ndern erfolgt
	 * unabh&auml;ngig von Berechtigungsklassen nur unter Pr&uuml;fung der
	 * entsprechenden Rechte am DAV.
	 *
	 * @param adminLoginname der Name des Administrators der die Aktion ausführt.
	 * @param adminPasswort  das Anmeldekennwort des Administrators der die Aktion
	 *                       ausführt.
	 * @param benutzer       der Benutzer, dessen Passwort geändert werden soll.
	 * @param neuesPasswort  das neue Passwort des Benutzer.
	 * @return der neue Benutzer
	 * @throws KeineRechteException    wenn die Benutzerrechte für diese Aktion
	 *                                 nicht ausreichen.
	 * @throws BenutzerChangeException wenn beim Anlegen des Benutzers ein Fehler
	 *                                 eintrat.
	 */
	public Benutzer aendernPasswort(final String adminLoginname, final String adminPasswort, final Benutzer benutzer,
			final String neuesPasswort) throws KeineRechteException, BenutzerChangeException {
		// wenn nicht das eigene Passwort geändert werden soll, prüfe ob der
		// Anfordernde ein Administrator ist, um eine ordentliche Fehlermeldung
		// zu erzeugen
		if (!benutzer.equals(getAngemeldetenBenutzer())) {
			if (!isDAVAdmin(adminLoginname, adminPasswort)) {
				throw new KeineRechteException(
						"Sie verfügen nicht über ausreichend Rechte zum Ändern des Passwortes eines Benutzer.");
			}
		}

		try {
			final DataModel modell = dav.getDataModel();
			final UserAdministration userAdmin = modell.getUserAdministration();
			userAdmin.changeUserPassword(adminLoginname, adminPasswort, benutzer.getName(), neuesPasswort);
		} catch (final ConfigurationTaskException ex) {
			throw new BenutzerChangeException(
					"Das Passwort des Benutzers " + benutzer.getName() + " konnte nicht geändert werden.", ex);
		}

		return benutzer;
	}

	public Collection<BerechtigungsKlasse> getBerechtigungsKlassen() {
		return Collections.unmodifiableCollection(berechtigungsKlassenMap.values());
	}

	public Collection<Region> getRegionen() {
		return Collections.unmodifiableCollection(regionenMap.values());
	}

	public Rolle getRolle(String pid) {
		return rollenMap.get(pid);
	}

	public Region getRegion(String pid) {
		return regionenMap.get(pid);
	}

	public Sender getSender() {
		return sender;
	}
}
