/*
 * Rahmenwerk-Plug-in "Benutzerverwaltung/Zugriffsrechte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.benutzervew.data;

import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import de.bsvrz.buv.plugin.benutzervew.internal.RahmenwerkService;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.Data.Array;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.daf.main.ReceiverRole;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;

public class DatenVerteiler extends SystemObjectWrapper {

	private static final String ATG_ANGEMELDETE_APPLIKATIONEN = "atg.angemeldeteApplikationen";
	private static final String ASP_STANDARD = "asp.standard";

	private final DataDescription anmeldungenDesc;

	private final Map<SystemObject, AngemeldeteApplikation> angemeldeteApplikationen = new LinkedHashMap<>();

	public DatenVerteiler(final SystemObject systemObject) {
		super(systemObject);

		final ClientDavInterface dav = RahmenwerkService.getService().getRahmenWerk().getDavVerbindung();
		final DataModel dataModel = systemObject.getDataModel();

		anmeldungenDesc = new DataDescription(dataModel.getAttributeGroup(ATG_ANGEMELDETE_APPLIKATIONEN),
				dataModel.getAspect(ASP_STANDARD));
		updateAngemeldeteApplikationen(dav.getData(systemObject, anmeldungenDesc, 0));
		dav.subscribeReceiver((results) -> updateAngemeldeteApplikationen(results), systemObject, anmeldungenDesc,
				ReceiveOptions.normal(), ReceiverRole.receiver());

	}

	private void updateAngemeldeteApplikationen(final ResultData... results) {
		final Set<SystemObject> vorherigeApplikationen = new LinkedHashSet<>(angemeldeteApplikationen.keySet());
		final Set<String> geaenderteBenutzerPid = new LinkedHashSet<>();
		for (final ResultData result : results) {
			if (result.hasData()) {
				final Array array = result.getData().getArray("angemeldeteApplikation");
				for (int idx = 0; idx < array.getLength(); idx++) {
					final Data item = array.getItem(idx);
					final SystemObject appObj = item.getReferenceValue("applikation").getSystemObject();
					if (appObj == null) {
						continue;
					}

					AngemeldeteApplikation angemeldeteApplikation = angemeldeteApplikationen.get(appObj);
					if (angemeldeteApplikation == null) {
						angemeldeteApplikation = new AngemeldeteApplikation(appObj);
						angemeldeteApplikationen.put(appObj, angemeldeteApplikation);
						geaenderteBenutzerPid.add(item.getReferenceValue("benutzer").getSystemObjectPid());
					}

					angemeldeteApplikation.setBenutzerPid(item.getReferenceValue("benutzer").getSystemObjectPid());
					angemeldeteApplikation.setAnmeldeZeit(item.getTimeValue("seit").getMillis());

					vorherigeApplikationen.remove(appObj);
				}
				vorherigeApplikationen.stream().forEach(so -> {
					final AngemeldeteApplikation applikation = angemeldeteApplikationen.remove(so);
					if (applikation != null) {
						geaenderteBenutzerPid.add(applikation.getUserPid());
					}
				});
			}
		}

		final Benutzerverwaltung benutzerverwaltung = RahmenwerkService.getService().getBenutzerverwaltung();
		for (final String benutzerPid : geaenderteBenutzerPid) {
			final Benutzer benutzer = benutzerverwaltung.getBenutzerWithPid(benutzerPid);
			if (benutzer != null) {
				benutzerverwaltung.fireBenutzerChanged(benutzer);
			}
		}
	}

	public List<AngemeldeteApplikation> getAngemeldeteApplikationenFor(final Benutzer benutzer) {
		return angemeldeteApplikationen.values().stream().filter(app -> benutzer.getPid().equals(app.getUserPid()))
				.collect(Collectors.toList());
	}
}
