/*
 * Rahmenwerk-Plug-in "Benutzerverwaltung/Zugriffsrechte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.benutzervew.data;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import de.bsvrz.buv.plugin.benutzervew.PluginBenutzerVew;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientSenderInterface;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.DataNotSubscribedException;
import de.bsvrz.dav.daf.main.OneSubscriptionPerSendData;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.SendSubscriptionNotConfirmed;
import de.bsvrz.dav.daf.main.SenderRole;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;

public class Sender implements ClientSenderInterface {

	private static class Anmeldung {
		private final SystemObject systemObject;
		private final AttributeGroup atg;
		private final Aspect asp;

		Anmeldung(SystemObject object, DataDescription dataDescription) {
			systemObject = object;
			atg = dataDescription.getAttributeGroup();
			asp = dataDescription.getAspect();
		}

		@Override
		public int hashCode() {
			return Objects.hash(systemObject, atg, asp);
		}

		@Override
		public boolean equals(Object obj) {

			if (this == obj) {
				return true;
			}
			if (obj == null) {
				return false;
			}
			if (getClass() != obj.getClass()) {
				return false;
			}

			final Anmeldung other = (Anmeldung) obj;
			boolean result = Objects.equals(other.systemObject, systemObject);
			result &= Objects.equals(other.atg, atg);
			result &= Objects.equals(other.asp, asp);

			return result;
		}
	}

	private ClientDavInterface dav;
	private final List<ResultData> requests = new ArrayList<>();
	private final Set<Anmeldung> anmeldungen = new LinkedHashSet<>();

	@Override
	public void dataRequest(SystemObject object, DataDescription dataDescription, byte state) {
		if (state == ClientSenderInterface.START_SENDING) {
			anmeldungen.add(new Anmeldung(object, dataDescription));
			synchronized (requests) {
				final Iterator<ResultData> iterator = requests.iterator();
				while (iterator.hasNext()) {
					final ResultData data = iterator.next();
					if (!Objects.equals(data.getObject(), object)) {
						continue;
					}
					if (!Objects.equals(data.getDataDescription().getAttributeGroup(),
							dataDescription.getAttributeGroup())) {
						continue;
					}
					if (!Objects.equals(data.getDataDescription().getAspect(), dataDescription.getAspect())) {
						continue;
					}

					if (dav != null) {
						try {
							dav.sendData(data);
						} catch (DataNotSubscribedException | SendSubscriptionNotConfirmed e) {
							PluginBenutzerVew.getDefault().getLog()
									.log(new Status(IStatus.ERROR, PluginBenutzerVew.PLUGIN_ID,
											"Das Versenden von Benutzerdaten ist fehlgeschlagen.", e));
						}
					}

					iterator.remove();
				}
			}
		} else {
			anmeldungen.remove(new Anmeldung(object, dataDescription));
		}
	}

	@Override
	public boolean isRequestSupported(SystemObject object, DataDescription dataDescription) {
		return true;
	}

	public void setDav(ClientDavInterface dav) {
		if (Objects.equals(dav, this.dav)) {
			return;
		}

		if (dav == null) {
			reset();
		}

		this.dav = dav;
	}

	private void reset() {
		if (dav == null) {
			return;
		}

		synchronized (requests) {
			requests.clear();
		}

		synchronized (anmeldungen) {
			anmeldungen.stream().forEach(anmeldung -> dav.unsubscribeSender(this, anmeldung.systemObject,
					new DataDescription(anmeldung.atg, anmeldung.asp)));
			anmeldungen.clear();
		}
	}

	public void sende(ResultData resultData) {

		if (dav == null) {
			return;
		}

		if (anmeldungen.contains(new Anmeldung(resultData.getObject(), resultData.getDataDescription()))) {
			try {
				dav.sendData(resultData);
			} catch (DataNotSubscribedException | SendSubscriptionNotConfirmed e) {
				PluginBenutzerVew.getDefault().getLog().log(new Status(IStatus.ERROR, PluginBenutzerVew.PLUGIN_ID,
						"Das Versenden von Benutzerdaten ist fehlgeschlagen.", e));
			}
		} else {
			synchronized (requests) {
				requests.add(resultData);
				try {
					dav.subscribeSender(this, resultData.getObject(), resultData.getDataDescription(),
							SenderRole.sender());
				} catch (final OneSubscriptionPerSendData e) {
					PluginBenutzerVew.getDefault().getLog().log(new Status(IStatus.ERROR, PluginBenutzerVew.PLUGIN_ID,
							"Die Anmeldung als Sender für Benutzerdaten ist fehlgeschlagen.", e));
				}
			}
		}
	}

}
