/*
 * Rahmenwerk-Plug-in "Benutzerverwaltung/Zugriffsrechte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.benutzervew.editor;

import java.util.Collection;
import java.util.Collections;
import java.util.TreeSet;
import java.util.stream.Collectors;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.ui.plugin.AbstractUIPlugin;

import de.bsvrz.buv.plugin.benutzervew.PluginBenutzerVew;
import de.bsvrz.buv.plugin.benutzervew.internal.RahmenwerkService;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;

/**
 * {@link WizardPage} für die Auswahl einzelner {@link SystemObject}e.
 *
 * @author BitCtrl Systems GmbH, ChHoesel
 *
 */
public class ObjekteWizardPage extends WizardPage {

	private static final class SystemObjekteTreeContentProvider implements ITreeContentProvider {
		@Override
		public boolean hasChildren(Object element) {
			return getChildren(element).length > 0;
		}

		@Override
		public Object getParent(Object element) {
			return null;
		}

		@Override
		public Object[] getElements(Object inputElement) {
			if (inputElement instanceof Collection) {
				return ((Collection<?>) inputElement).toArray();
			}
			return new Object[0];
		}

		@Override
		public Object[] getChildren(Object parentElement) {
			if (parentElement instanceof SystemObjectType) {
				return ((SystemObjectType) parentElement).getElements().toArray();
			}
			return new Object[0];
		}
	}

	private final Collection<SystemObject> verwendeteObjekte = new TreeSet<>();
	private TreeViewer listVerfuegbareTypen;
	private TreeViewer listVerwendeteAtg;

	private final ObjektFactory objektFactory;

	public ObjekteWizardPage() {
		super(ObjekteWizardPage.class.getName());
		setImageDescriptor(AbstractUIPlugin.imageDescriptorFromPlugin(PluginBenutzerVew.PLUGIN_ID,
				"icons/objekte_wizard_page.gif"));
		setTitle("Auswahl der Objekte");
		setDescription(
				"Über dieses Array können beliebige Konfigurationsobjekte angegeben werden.\nBei einem leeren Array sind alle Konfigurationsobjekte ausgewählt.");
		objektFactory = RahmenwerkService.getService().getObjektFactory();
	}

	@Override
	public void createControl(Composite parent) {
		final Composite container = new Composite(parent, SWT.NONE);

		setControl(container);
		container.setLayout(new GridLayout(3, false));

		final Group groupVerfuegbareAtg = new Group(container, SWT.NONE);
		groupVerfuegbareAtg.setText("Verfügbare Objekte");
		groupVerfuegbareAtg.setLayout(new FillLayout());
		groupVerfuegbareAtg
				.setLayoutData(GridDataFactory.fillDefaults().grab(true, true).span(1, 1).hint(300, 400).create());

		listVerfuegbareTypen = new TreeViewer(groupVerfuegbareAtg,
				SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL | SWT.MULTI);
		listVerfuegbareTypen.setContentProvider(new SystemObjekteTreeContentProvider());
		listVerfuegbareTypen.setFilters(new ViewerFilter() {

			@Override
			public boolean select(Viewer viewer, Object parentElement, Object element) {
				if (element instanceof SystemObjectType) {
					return !((SystemObjectType) element).getElements().isEmpty();
				}
				return !verwendeteObjekte.contains(element);
			}
		});

		listVerfuegbareTypen.setInput(objektFactory.getDav().getDataModel().getTypeTypeObject().getElements());

		final Composite buttonComposite = new Composite(container, SWT.NONE);
		buttonComposite.setLayout(new GridLayout(1, false));
		buttonComposite.setLayoutData(GridDataFactory.fillDefaults().align(SWT.FILL, SWT.CENTER).grab(false, true)
				.hint(100, SWT.DEFAULT).create());

		final Button btnEinfuegen = new Button(buttonComposite, SWT.NONE);
		btnEinfuegen.setLayoutData(new GridData(SWT.FILL, SWT.BOTTOM, true, true, 1, 1));
		btnEinfuegen.setText(">>");
		btnEinfuegen.addSelectionListener(new SelectionAdapter() {

			@SuppressWarnings("unchecked")
			@Override
			public void widgetSelected(SelectionEvent e) {
				final IStructuredSelection selection = (IStructuredSelection) listVerfuegbareTypen.getSelection();
				verwendeteObjekte.addAll(selection.toList());
				listVerfuegbareTypen.refresh();
				listVerwendeteAtg.refresh();
			}

		});

		final Button btnEntfernen = new Button(buttonComposite, SWT.NONE);
		btnEntfernen.setLayoutData(GridDataFactory.fillDefaults().align(SWT.FILL, SWT.CENTER).grab(true, true)
				.hint(100, SWT.DEFAULT).create());
		btnEntfernen.setText("<<");
		btnEntfernen.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(SelectionEvent e) {
				final IStructuredSelection selection = (IStructuredSelection) listVerwendeteAtg.getSelection();
				verwendeteObjekte.removeAll(selection.toList());
				listVerfuegbareTypen.refresh();
				listVerwendeteAtg.refresh();
			}

		});

		final Group groupVerwendeteKB = new Group(container, SWT.NONE);
		groupVerwendeteKB.setText("Verwendete Objekte");
		groupVerwendeteKB.setLayout(new FillLayout());
		groupVerwendeteKB
				.setLayoutData(GridDataFactory.fillDefaults().grab(true, true).span(1, 1).hint(300, 400).create());

		listVerwendeteAtg = new TreeViewer(groupVerwendeteKB, SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL | SWT.MULTI);
		listVerwendeteAtg.setContentProvider(new SystemObjekteTreeContentProvider());
		listVerwendeteAtg.setInput(verwendeteObjekte);

	}

	public void setVerwendeteObjekte(Collection<SystemObjekt> feld) {
		verwendeteObjekte.addAll(feld.stream().map(o -> o.getSystemObject()).collect(Collectors.toList()));
	}

	public Collection<SystemObjekt> getVerwendeteObjekte() {
		return Collections.unmodifiableCollection(verwendeteObjekte.stream().filter(o -> o instanceof SystemObject)
				.map(o -> objektFactory.getModellobjekt(o)).collect(Collectors.toList()));
	}
}
