/*
 * Rahmenwerk-Plug-in "Benutzerverwaltung/Zugriffsrechte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.benutzervew.editor.region;

import java.util.Collection;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.layout.TreeColumnLayout;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.editor.FormPage;
import org.eclipse.ui.forms.events.ExpansionAdapter;
import org.eclipse.ui.forms.events.ExpansionEvent;
import org.eclipse.ui.forms.widgets.ExpandableComposite;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;

import de.bsvrz.buv.plugin.benutzervew.PluginBenutzerVew;
import de.bsvrz.buv.plugin.benutzervew.editor.NamedFeld;
import de.bsvrz.buv.plugin.benutzervew.internal.RahmenwerkService;
import de.bsvrz.buv.rw.basislib.urlasser.UrlasserInfoDatenDialog;
import de.bsvrz.sys.funclib.bitctrl.modell.AnmeldeException;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensendeException;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Feld;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.attribute.AtlAusgeschlosseneObjekte;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.attribute.AtlBereiche;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.attribute.AtlEnthalteneObjekte;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.attribute.AtlObjekte;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.attribute.AtlRegionen;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.objekte.Benutzer;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.objekte.ZugriffsRegionNeu;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.parameter.PdRegion.Daten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmvewbetriebglobal.attribute.AtlUrlasser;

/**
 * FormPage im Editor zur Berabeitung der Region - Parameter.
 *
 * @author BitCtrl Systems GmbH, ChHoesel
 *
 */
public class RegionPage extends FormPage {

	private static final String REGION = "Region";
	private static final String TYPEN = "Typen";
	private static final String KONFIGURATIONSBEREICHE = "Konfigurationsbereiche";
	private static final String KONFIGURATIONSVERANTWORTLICHE = "Konfigurationsverantwortliche";
	private static final String OBJEKTE = "Objekte";
	private static final String BEREICHE = "Bereiche";
	private static final String REGIONEN = "Regionen";

	private final class AusgeschlosseneObjekteEntfernenAdapter extends SelectionAdapter {
		@Override
		public void widgetSelected(SelectionEvent e) {
			final IStructuredSelection selection = (IStructuredSelection) ausgeschlosseneObjekteViewer.getSelection();
			final Object firstElement = selection.getFirstElement();

			if (firstElement instanceof AtlBereiche) {
				regionParameterDaten.getAusgeschlosseneObjekte().stream()
						.forEach(a -> a.getBereich().remove(firstElement));
			} else if (firstElement instanceof AtlRegionen) {
				regionParameterDaten.getAusgeschlosseneObjekte().stream()
						.forEach(a -> a.getRegion().remove(firstElement));
			} else if (firstElement instanceof AtlObjekte) {
				regionParameterDaten.getAusgeschlosseneObjekte().stream()
						.forEach(a -> a.getObjekte().remove(firstElement));
			} else if (firstElement instanceof AtlAusgeschlosseneObjekte) {
				regionParameterDaten.getAusgeschlosseneObjekte().remove(firstElement);
			}
			setDirty(true);
			ausgeschlosseneObjekteViewer.refresh();

		}
	}

	private final class AusgeschlosseneObjekteBearbeitenAdapter extends SelectionAdapter {
		@Override
		public void widgetSelected(SelectionEvent e) {
			final IStructuredSelection selection = (IStructuredSelection) ausgeschlosseneObjekteViewer.getSelection();
			final Object firstElement = selection.getFirstElement();

			if (firstElement instanceof NamedFeld<?>) {
				final NamedFeld<?> feld = (NamedFeld<?>) firstElement;
				if ("Bereiche".equals(feld.getName())) {
					final BereicheWizard wiz = new BereicheWizard((AtlAusgeschlosseneObjekte) feld.getParent());
					final WizardDialog dialog = new WizardDialog(ausgeschlosseneObjekteViewer.getControl().getShell(),
							wiz);
					if (Window.OK == dialog.open()) {
						setDirty(true);
					}
				}

			} else if (firstElement instanceof AtlBereiche) {

				final BereicheWizard wiz = new BereicheWizard((AtlBereiche) firstElement);
				final WizardDialog dialog = new WizardDialog(ausgeschlosseneObjekteViewer.getControl().getShell(), wiz);
				if (Window.OK == dialog.open()) {
					setDirty(true);
				}

			} else if (firstElement instanceof AtlRegionen) {
				final RegionWizard wiz = new RegionWizard((AtlRegionen) firstElement);
				final WizardDialog dialog = new WizardDialog(ausgeschlosseneObjekteViewer.getControl().getShell(), wiz);
				if (Window.OK == dialog.open()) {
					setDirty(true);
				}

			} else if (firstElement instanceof AtlObjekte) {
				final ObjekteWizard wiz = new ObjekteWizard((AtlObjekte) firstElement);
				final WizardDialog dialog = new WizardDialog(ausgeschlosseneObjekteViewer.getControl().getShell(), wiz);
				if (Window.OK == dialog.open()) {
					setDirty(true);
				}
			}

			ausgeschlosseneObjekteViewer.refresh();
		}
	}

	private final class AusgeschlosseneObjekteAnlegenAdapter extends SelectionAdapter {
		@Override
		public void widgetSelected(SelectionEvent e) {
			final IStructuredSelection selection = (IStructuredSelection) ausgeschlosseneObjekteViewer.getSelection();
			final Object firstElement = selection.getFirstElement();

			if (firstElement instanceof NamedFeld<?>) {
				final NamedFeld<?> feld = (NamedFeld<?>) firstElement;
				if ("Bereiche".equals(feld.getName())) {
					final BereicheWizard wiz = new BereicheWizard((AtlAusgeschlosseneObjekte) feld.getParent());
					final WizardDialog dialog = new WizardDialog(ausgeschlosseneObjekteViewer.getControl().getShell(),
							wiz);
					if (Window.OK == dialog.open()) {
						setDirty(true);
					}
				} else if ("Regionen".equals(feld.getName())) {
					final RegionWizard wiz = new RegionWizard((AtlAusgeschlosseneObjekte) feld.getParent());
					final WizardDialog dialog = new WizardDialog(ausgeschlosseneObjekteViewer.getControl().getShell(),
							wiz);
					if (Window.OK == dialog.open()) {
						setDirty(true);
					}
				} else if ("Objekte".equals(feld.getName())) {
					final ObjekteWizard wiz = new ObjekteWizard((AtlAusgeschlosseneObjekte) feld.getParent());
					final WizardDialog dialog = new WizardDialog(ausgeschlosseneObjekteViewer.getControl().getShell(),
							wiz);
					if (Window.OK == dialog.open()) {
						setDirty(true);
					}
				}
			} else {
				regionParameterDaten.getAusgeschlosseneObjekte().add(new AtlAusgeschlosseneObjekte());
				setDirty(true);
			}
			ausgeschlosseneObjekteViewer.refresh();
		}
	}

	private final class EnthalteneObjekteEntfernenAdapter extends SelectionAdapter {
		@Override
		public void widgetSelected(SelectionEvent e) {
			final IStructuredSelection selection = (IStructuredSelection) enthalteneObjekteViewer.getSelection();
			final Object firstElement = selection.getFirstElement();

			if (firstElement instanceof AtlBereiche) {
				regionParameterDaten.getEnthalteneObjekte().stream().forEach(a -> a.getBereich().remove(firstElement));
			} else if (firstElement instanceof AtlRegionen) {
				regionParameterDaten.getEnthalteneObjekte().stream().forEach(a -> a.getRegion().remove(firstElement));
			} else if (firstElement instanceof AtlObjekte) {
				regionParameterDaten.getEnthalteneObjekte().stream().forEach(a -> a.getObjekte().remove(firstElement));
			} else if (firstElement instanceof AtlEnthalteneObjekte) {
				regionParameterDaten.getEnthalteneObjekte().remove(firstElement);
			}
			setDirty(true);
			enthalteneObjekteViewer.refresh();

		}
	}

	private final class EnthalteneObjekteBearbeitenAdapter extends SelectionAdapter {
		@Override
		public void widgetSelected(SelectionEvent e) {
			final IStructuredSelection selection = (IStructuredSelection) enthalteneObjekteViewer.getSelection();
			final Object firstElement = selection.getFirstElement();

			if (firstElement instanceof NamedFeld<?>) {
				final NamedFeld<?> feld = (NamedFeld<?>) firstElement;
				if ("Bereiche".equals(feld.getName())) {
					final BereicheWizard wiz = new BereicheWizard((AtlEnthalteneObjekte) feld.getParent());
					final WizardDialog dialog = new WizardDialog(enthalteneObjekteViewer.getControl().getShell(), wiz);
					if (Window.OK == dialog.open()) {
						setDirty(true);
					}
				}

			} else if (firstElement instanceof AtlBereiche) {

				final BereicheWizard wiz = new BereicheWizard((AtlBereiche) firstElement);
				final WizardDialog dialog = new WizardDialog(enthalteneObjekteViewer.getControl().getShell(), wiz);
				if (Window.OK == dialog.open()) {
					setDirty(true);
				}

			} else if (firstElement instanceof AtlRegionen) {
				final RegionWizard wiz = new RegionWizard((AtlRegionen) firstElement);
				final WizardDialog dialog = new WizardDialog(enthalteneObjekteViewer.getControl().getShell(), wiz);
				if (Window.OK == dialog.open()) {
					setDirty(true);
				}

			} else if (firstElement instanceof AtlObjekte) {
				final ObjekteWizard wiz = new ObjekteWizard((AtlObjekte) firstElement);
				final WizardDialog dialog = new WizardDialog(enthalteneObjekteViewer.getControl().getShell(), wiz);
				if (Window.OK == dialog.open()) {
					setDirty(true);
				}
			}

			enthalteneObjekteViewer.refresh();
		}
	}

	private final class EnthalteneObjekteAnlegenAdapter extends SelectionAdapter {
		@Override
		public void widgetSelected(SelectionEvent e) {
			final IStructuredSelection selection = (IStructuredSelection) enthalteneObjekteViewer.getSelection();
			final Object firstElement = selection.getFirstElement();

			if (firstElement instanceof NamedFeld<?>) {
				final NamedFeld<?> feld = (NamedFeld<?>) firstElement;
				if ("Bereiche".equals(feld.getName())) {
					final BereicheWizard wiz = new BereicheWizard((AtlEnthalteneObjekte) feld.getParent());
					final WizardDialog dialog = new WizardDialog(enthalteneObjekteViewer.getControl().getShell(), wiz);
					if (Window.OK == dialog.open()) {
						setDirty(true);
					}
				} else if ("Regionen".equals(feld.getName())) {
					final RegionWizard wiz = new RegionWizard((AtlEnthalteneObjekte) feld.getParent());
					final WizardDialog dialog = new WizardDialog(enthalteneObjekteViewer.getControl().getShell(), wiz);
					if (Window.OK == dialog.open()) {
						setDirty(true);
					}
				} else if ("Objekte".equals(feld.getName())) {
					final ObjekteWizard wiz = new ObjekteWizard((AtlEnthalteneObjekte) feld.getParent());
					final WizardDialog dialog = new WizardDialog(enthalteneObjekteViewer.getControl().getShell(), wiz);
					if (Window.OK == dialog.open()) {
						setDirty(true);
					}
				}
			} else {
				regionParameterDaten.getEnthalteneObjekte().add(new AtlEnthalteneObjekte());
				setDirty(true);
			}
			enthalteneObjekteViewer.refresh();
		}
	}

	private static final class ObjekteLabelProvider extends ColumnLabelProvider {

		@Override
		public String getText(Object element) {
			if (element instanceof AtlEnthalteneObjekte) {
				final AtlEnthalteneObjekte objekte = (AtlEnthalteneObjekte) element;
				return "Enthaltene Objekte " + "(" + objekte.getBereich().size() + " Bereiche, "
						+ objekte.getRegion().size() + " Regionen, " + objekte.getObjekte().size() + " Objekte)";
			} else if (element instanceof AtlAusgeschlosseneObjekte) {
				final AtlAusgeschlosseneObjekte objekte = (AtlAusgeschlosseneObjekte) element;
				return "Ausgeschlossene Objekte " + "(" + objekte.getBereich().size() + " Bereiche, "
						+ objekte.getRegion().size() + " Regionen, " + objekte.getObjekte().size() + " Objekte)";
			} else if (element instanceof AtlBereiche) {
				final AtlBereiche bereich = (AtlBereiche) element;
				return "Bereich (" + bereich.getKonfigurationsverantwortlicher().size() + " KV, "
						+ bereich.getKonfigurationsbereich().size() + " KB, " + bereich.getTyp().size()
						+ " Typen, Mengenbez.: " + bereich.getMengenbezeichnung() + ")";
			} else if (element instanceof AtlRegionen) {
				final AtlRegionen region = (AtlRegionen) element;
				return "Region (" + region.getRegion().size() + " Regionen, " + region.getTyp().size()
						+ " Typen, Mengenbez.: " + region.getMengenbezeichnung();
			} else if (element instanceof AtlObjekte) {
				final AtlObjekte objekte = (AtlObjekte) element;
				return "Objekt (" + objekte.getObjekt().size() + "Objekte, Mengenbez.: "
						+ objekte.getMengenbezeichnung() + ")";
			} else if (element instanceof NamedFeld) {
				final NamedFeld<?> feld = (NamedFeld<?>) element;
				return feld.getName();
			} else if (element instanceof SystemObjekt) {
				return ((SystemObjekt) element).getName() + "( " + ((SystemObjekt) element).getPid() + ")";
			}
			return super.getText(element);
		}
	}

	private static final class ObjekteTreeContentProvider implements ITreeContentProvider {

		@Override
		public boolean hasChildren(Object element) {
			return getChildren(element).length > 0;
		}

		@Override
		public Object getParent(Object element) {
			return null;
		}

		@SuppressWarnings("rawtypes")
		@Override
		public Object[] getElements(Object inputElement) {
			if (inputElement instanceof Collection<?>) {
				return ((Collection) inputElement).toArray();
			}
			return new Object[0];
		}

		@Override
		public Object[] getChildren(Object parentElement) {
			if (parentElement instanceof AtlEnthalteneObjekte) {
				final AtlEnthalteneObjekte objekte = (AtlEnthalteneObjekte) parentElement;
				return new Object[] { new NamedFeld<>(BEREICHE, objekte, objekte.getBereich()),
						new NamedFeld<>(REGIONEN, objekte, objekte.getRegion()),
						new NamedFeld<>(OBJEKTE, objekte, objekte.getObjekte()) };
			} else if (parentElement instanceof AtlAusgeschlosseneObjekte) {
				final AtlAusgeschlosseneObjekte objekte = (AtlAusgeschlosseneObjekte) parentElement;
				return new Object[] { new NamedFeld<>(BEREICHE, objekte, objekte.getBereich()),
						new NamedFeld<>(REGIONEN, objekte, objekte.getRegion()),
						new NamedFeld<>(OBJEKTE, objekte, objekte.getObjekte()) };
			} else if (parentElement instanceof AtlBereiche) {
				final AtlBereiche atlBereiche = (AtlBereiche) parentElement;
				return new Object[] {
						new NamedFeld<>(KONFIGURATIONSVERANTWORTLICHE, parentElement,
								atlBereiche.getKonfigurationsverantwortlicher()),
						new NamedFeld<>(KONFIGURATIONSBEREICHE, parentElement,
								((AtlBereiche) parentElement).getKonfigurationsbereich()),
						new NamedFeld<>(TYPEN, parentElement, ((AtlBereiche) parentElement).getTyp()) };
			} else if (parentElement instanceof AtlRegionen) {
				final AtlRegionen region = (AtlRegionen) parentElement;
				return new Object[] { new NamedFeld<>(REGION, parentElement, region.getRegion()),
						new NamedFeld<>(TYPEN, parentElement, region.getTyp()) };
			} else if (parentElement instanceof AtlObjekte) {
				return ((AtlObjekte) parentElement).getObjekt().toArray();
			} else if (parentElement instanceof Feld) {
				return ((Feld<?>) parentElement).toArray();
			}
			return new Object[0];
		}
	}

	private boolean dirty;
	private TreeViewer enthalteneObjekteViewer;
	private TreeViewer ausgeschlosseneObjekteViewer;
	private Daten regionParameterDaten;

	public RegionPage(FormEditor editor, String id, String title) {
		super(editor, id, title);
	}

	@Override
	protected void createFormContent(IManagedForm managedForm) {
		managedForm.getForm()
				.setImage(PluginBenutzerVew.getDefault().getImageRegistry().get(PluginBenutzerVew.ICONS_REGION_GIF));
		super.createFormContent(managedForm);
		final ZugriffsRegionNeu region = getEditorInput().getAdapter(ZugriffsRegionNeu.class);
		Assert.isNotNull(region);

		final FormToolkit toolkit = managedForm.getToolkit();
		final ScrolledForm form = managedForm.getForm();
		form.setText("Regionen parametrieren");

		final Composite body = form.getBody();
		toolkit.decorateFormHeading(form.getForm());
		toolkit.paintBordersFor(body);

		form.getBody().setLayout(GridLayoutFactory.fillDefaults().numColumns(2).equalWidth(true).create());

		createEnthalteneObjekteSection(toolkit, form);
		createAusgeschlosseneObjekteSection(toolkit, form);

		regionParameterDaten = region.getPdRegion().getDatum().clone();

		enthalteneObjekteViewer.setInput(regionParameterDaten.getEnthalteneObjekte());
		ausgeschlosseneObjekteViewer.setInput(regionParameterDaten.getAusgeschlosseneObjekte());

	}

	private void createEnthalteneObjekteSection(FormToolkit toolkit, ScrolledForm form) {
		final Section section = toolkit.createSection(form.getBody(), ExpandableComposite.FOCUS_TITLE
				| Section.DESCRIPTION | ExpandableComposite.TITLE_BAR | ExpandableComposite.EXPANDED);
		section.setLayoutData(GridDataFactory.fillDefaults().grab(true, true).create());

		section.addExpansionListener(new ExpansionAdapter() {
			@Override
			public void expansionStateChanged(final ExpansionEvent event) {
				form.reflow(true);
			}
		});
		section.setText("Enthaltene Objekte");
		section.setDescription("Über diese Vorgaben werden die in der Region enthaltenen Objekte spezifiziert.");

		final Composite sectionClient = toolkit.createComposite(section);
		sectionClient.setLayout(new GridLayout(1, false));

		final Composite buttonPanel = toolkit.createComposite(sectionClient);
		buttonPanel.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		buttonPanel.setLayout(GridLayoutFactory.fillDefaults().numColumns(3).equalWidth(false).create());

		final Button anlegenButton = toolkit.createButton(buttonPanel, "Anlegen", SWT.PUSH);
		anlegenButton.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		anlegenButton.addSelectionListener(new EnthalteneObjekteAnlegenAdapter());

		final Button bearbeitenButton = toolkit.createButton(buttonPanel, "Bearbeiten", SWT.PUSH);
		bearbeitenButton.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		bearbeitenButton.setEnabled(false);
		bearbeitenButton.addSelectionListener(new EnthalteneObjekteBearbeitenAdapter());

		final Button entfernenButton = toolkit.createButton(buttonPanel, "Entfernen", SWT.PUSH);
		entfernenButton.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		entfernenButton.setEnabled(false);
		entfernenButton.addSelectionListener(new EnthalteneObjekteEntfernenAdapter());

		final Composite treeComposite = toolkit.createComposite(sectionClient, SWT.NONE);
		treeComposite.setLayoutData(GridDataFactory.fillDefaults().grab(true, true).minSize(100, 200).create());
		final TreeColumnLayout treeLayout = new TreeColumnLayout();
		treeComposite.setLayout(treeLayout);

		enthalteneObjekteViewer = new TreeViewer(treeComposite, SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);
		enthalteneObjekteViewer.setContentProvider(new ObjekteTreeContentProvider());

		enthalteneObjekteViewer.addSelectionChangedListener(event -> {
			final Object firstElement = event.getStructuredSelection().getFirstElement();
			final boolean datenSelected = (firstElement instanceof AtlBereiche) | (firstElement instanceof AtlRegionen)
					| (firstElement instanceof AtlObjekte);
			bearbeitenButton.setEnabled(datenSelected);
			entfernenButton.setEnabled(datenSelected | (firstElement instanceof AtlEnthalteneObjekte));
		});

		final TreeViewerColumn nameCloumn = new TreeViewerColumn(enthalteneObjekteViewer, SWT.NONE);
		treeLayout.setColumnData(nameCloumn.getColumn(), new ColumnWeightData(50, 100));

		nameCloumn.setLabelProvider(new ObjekteLabelProvider());

		section.setClient(sectionClient);

	}

	private void createAusgeschlosseneObjekteSection(FormToolkit toolkit, ScrolledForm form) {
		final Section section = toolkit.createSection(form.getBody(), ExpandableComposite.FOCUS_TITLE
				| Section.DESCRIPTION | ExpandableComposite.TITLE_BAR | ExpandableComposite.EXPANDED);
		section.setLayoutData(GridDataFactory.fillDefaults().grab(true, true).create());

		section.addExpansionListener(new ExpansionAdapter() {
			@Override
			public void expansionStateChanged(final ExpansionEvent event) {
				form.reflow(true);
			}
		});
		section.setText("Ausgeschlossene Objekte");
		section.setDescription(
				"Über diese Vorgaben werden die in der Region explizit ausgeschlossenen Objekte spezifiziert.");

		final Composite sectionClient = toolkit.createComposite(section);
		sectionClient.setLayout(new GridLayout(1, false));

		final Composite buttonPanel = toolkit.createComposite(sectionClient);
		buttonPanel.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		buttonPanel.setLayout(GridLayoutFactory.fillDefaults().numColumns(3).equalWidth(false).create());

		final Button anlegenButton = toolkit.createButton(buttonPanel, "Anlegen", SWT.PUSH);
		anlegenButton.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		anlegenButton.addSelectionListener(new AusgeschlosseneObjekteAnlegenAdapter());

		final Button bearbeitenButton = toolkit.createButton(buttonPanel, "Bearbeiten", SWT.PUSH);
		bearbeitenButton.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		bearbeitenButton.setEnabled(false);
		bearbeitenButton.addSelectionListener(new AusgeschlosseneObjekteBearbeitenAdapter());

		final Button entfernenButton = toolkit.createButton(buttonPanel, "Entfernen", SWT.PUSH);
		entfernenButton.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		entfernenButton.setEnabled(false);
		entfernenButton.addSelectionListener(new AusgeschlosseneObjekteEntfernenAdapter());

		final Composite treeComposite = toolkit.createComposite(sectionClient, SWT.NONE);
		treeComposite.setLayoutData(GridDataFactory.fillDefaults().grab(true, true).minSize(100, 200).create());
		final TreeColumnLayout treeLayout = new TreeColumnLayout();
		treeComposite.setLayout(treeLayout);

		ausgeschlosseneObjekteViewer = new TreeViewer(treeComposite, SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);
		ausgeschlosseneObjekteViewer.setContentProvider(new ObjekteTreeContentProvider());

		ausgeschlosseneObjekteViewer.addSelectionChangedListener(event -> {
			final Object firstElement = event.getStructuredSelection().getFirstElement();
			final boolean datenSelected = (firstElement instanceof AtlBereiche) | (firstElement instanceof AtlRegionen)
					| (firstElement instanceof AtlObjekte);
			bearbeitenButton.setEnabled(datenSelected);
			entfernenButton.setEnabled(datenSelected | (firstElement instanceof AtlAusgeschlosseneObjekte));
		});

		final TreeViewerColumn nameCloumn = new TreeViewerColumn(ausgeschlosseneObjekteViewer, SWT.NONE);
		treeLayout.setColumnData(nameCloumn.getColumn(), new ColumnWeightData(50, 100));

		nameCloumn.setLabelProvider(new ObjekteLabelProvider());

		section.setClient(sectionClient);

	}

	@Override
	public RegionEditor getEditor() {
		return (RegionEditor) super.getEditor();
	}

	@Override
	public boolean isDirty() {
		return dirty;
	}

	/**
	 * Setzt das lokale Dirty - Flag.
	 *
	 * @param newDirtyFlag der neue dirty - Zustand des Editors.
	 */
	protected void setDirty(final boolean newDirtyFlag) {
		dirty = newDirtyFlag;
		firePropertyChange(PROP_DIRTY);
		if (getManagedForm() != null) {
			getManagedForm().getForm().getDisplay().asyncExec(() -> getManagedForm().dirtyStateChanged());

		}
	}

	@Override
	public void doSave(IProgressMonitor monitor) {

		final UrlasserInfoDatenDialog dialog = new UrlasserInfoDatenDialog(getSite().getShell(),
				(verbindung, urlasserInfo) -> {
					final ZugriffsRegionNeu regionNeu = getEditorInput().getAdapter(ZugriffsRegionNeu.class);

					final ObjektFactory objektFactory = RahmenwerkService.getService().getObjektFactory();
					try {
						monitor.beginTask("Speichern", 5);
						regionNeu.getPdRegion().anmeldenSender();
						monitor.worked(1);
						final AtlUrlasser urlasser = new AtlUrlasser();
						urlasser.setBenutzerReferenz(
								(Benutzer) objektFactory.getModellobjekt(urlasserInfo.getBenutzer()));
						urlasser.setUrsache(urlasserInfo.getUrsache());
						urlasser.setVeranlasser(urlasserInfo.getVeranlasser());
						monitor.worked(1);
						regionParameterDaten.setUrlasser(urlasser);
						monitor.worked(1);
						regionNeu.getPdRegion().sendeDatum(regionParameterDaten);
						monitor.worked(1);
						regionNeu.getPdRegion().abmeldenSender();
						setDirty(false);
					} catch (AnmeldeException | DatensendeException e) {
						monitor.setCanceled(true);
						throw new IllegalStateException(
								"Das Speichern neuer (enthaltener & ausgeschlossener) Objekte für die Region \""
										+ regionNeu + "\" ist fehlgeschlagen.",
								e);
					}
				});

		dialog.open();
		monitor.done();

		super.doSave(monitor);
	}
}
