/*
 * Rahmenwerk-Plug-in "Benutzerverwaltung/Zugriffsrechte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.benutzervew.editor.rolle;

import java.util.Arrays;
import java.util.Collection;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.layout.TreeColumnLayout;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.editor.FormPage;
import org.eclipse.ui.forms.events.ExpansionAdapter;
import org.eclipse.ui.forms.events.ExpansionEvent;
import org.eclipse.ui.forms.widgets.ExpandableComposite;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;

import de.bsvrz.buv.plugin.benutzervew.PluginBenutzerVew;
import de.bsvrz.buv.plugin.benutzervew.editor.NamedFeld;
import de.bsvrz.buv.plugin.benutzervew.internal.RahmenwerkService;
import de.bsvrz.buv.rw.basislib.urlasser.UrlasserInfoDatenDialog;
import de.bsvrz.sys.funclib.bitctrl.modell.AnmeldeException;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensendeException;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Feld;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.attribute.AtlAktivitaetDaten;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.attribute.AtlAktivitaetMengen;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.attribute.AtlAktivitaetObjekteNeu;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.objekte.Benutzer;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.objekte.ZugriffsRolleNeu;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.parameter.PdRollenAktivitaet;
import de.bsvrz.sys.funclib.bitctrl.modell.tmvewbetriebglobal.attribute.AtlUrlasser;

/**
 * {@link FormPage} für den Rollen Parametereditor (ZugriffsRolleNeu).
 *
 * @author BitCtrl Systems GmbH, ChHoesel
 *
 */
public class RollenPage extends FormPage {

	private static final class AktivitaetDatenTreeProvider implements ITreeContentProvider {
		@Override
		public Object[] getElements(Object inputElement) {
			if (inputElement instanceof Feld) {
				return ((Feld<?>) inputElement).toArray();
			}
			return new Object[0];
		}

		@Override
		public Object[] getChildren(Object parentElement) {
			if (parentElement instanceof Feld) {
				return ((Feld<?>) parentElement).toArray();
			} else if (parentElement instanceof AtlAktivitaetDaten) {
				final AtlAktivitaetDaten aktivitaet = (AtlAktivitaetDaten) parentElement;
				return new Object[] { new NamedFeld<>("Konfigurationsbereiche", aktivitaet.getKonfigurationsbereich()),
						new NamedFeld<>("Attributgruppen", aktivitaet.getAttributgruppe()),
						new NamedFeld<>("Aspekte", aktivitaet.getAspekt()) };
			} else if (parentElement instanceof AtlAktivitaetObjekteNeu) {
				final AtlAktivitaetObjekteNeu objekte = (AtlAktivitaetObjekteNeu) parentElement;
				return new Object[] { new NamedFeld<>("Konfigurationsbereiche", objekte.getKonfigurationsbereich()),
						new NamedFeld<>("Objekt-Typen", objekte.getObjekttyp()) };
			} else if (parentElement instanceof AtlAktivitaetMengen) {
				final AtlAktivitaetMengen mengen = (AtlAktivitaetMengen) parentElement;
				return new Object[] { new NamedFeld<>("Konfigurationsbereiche", mengen.getKonfigurationsbereich()),
						new NamedFeld<>("Mengen-Typen", mengen.getMengentyp()) };
			}

			return new Object[0];
		}

		@Override
		public Object getParent(Object element) {
			return null;
		}

		@Override
		public boolean hasChildren(Object element) {
			return getChildren(element).length > 0;
		}
	}

	private TreeViewer datenTabelle;
	private PdRollenAktivitaet.Daten rollenAktivitaetParameterDaten;
	private TreeViewer objekteTabelle;
	private boolean dirty;
	private TreeViewer mengenTabelle;
	private TableViewer rollenTabelle;

	/**
	 * Create the form page.
	 *
	 * @param id
	 * @param title
	 */
	public RollenPage(String id, String title) {
		super(id, title);
	}

	public RollenPage(FormEditor editor, String id, String title) {
		super(editor, id, title);
	}

	@Override
	protected void createFormContent(IManagedForm managedForm) {
		managedForm.getForm()
				.setImage(PluginBenutzerVew.getDefault().getImageRegistry().get(PluginBenutzerVew.ICONS_ROLLE_GIF));
		super.createFormContent(managedForm);
		final ZugriffsRolleNeu rolleNeu = getEditorInput().getAdapter(ZugriffsRolleNeu.class);
		Assert.isNotNull(rolleNeu);

		final FormToolkit toolkit = managedForm.getToolkit();
		final ScrolledForm form = managedForm.getForm();
		form.setText("Rollen parametrieren");

		final Composite body = form.getBody();
		toolkit.decorateFormHeading(form.getForm());
		toolkit.paintBordersFor(body);

		form.getBody().setLayout(GridLayoutFactory.fillDefaults().numColumns(2).equalWidth(true).create());

		createAktivitaetenSection(toolkit, form);
		createObjekteSection(toolkit, form);
		createMengenSection(toolkit, form);
		createRollenSection(toolkit, form);

		rollenAktivitaetParameterDaten = rolleNeu.getPdRollenAktivitaet().getDatum().clone();
		datenTabelle.setInput(rollenAktivitaetParameterDaten.getAktivitaetDaten());
		objekteTabelle.setInput(rollenAktivitaetParameterDaten.getAktivitaetObjekte());
		mengenTabelle.setInput(rollenAktivitaetParameterDaten.getAktivitaetMengen());
		rollenTabelle.setInput(rollenAktivitaetParameterDaten.getRolle());

		form.reflow(true);
	}

	private void createObjekteSection(FormToolkit toolkit, ScrolledForm form) {
		final Section section = toolkit.createSection(form.getBody(), ExpandableComposite.FOCUS_TITLE
				| Section.DESCRIPTION | ExpandableComposite.TITLE_BAR | ExpandableComposite.EXPANDED);
		section.setLayoutData(GridDataFactory.fillDefaults().grab(true, true).create());

		section.addExpansionListener(new ExpansionAdapter() {
			@Override
			public void expansionStateChanged(final ExpansionEvent event) {
				form.reflow(true);
			}
		});
		section.setText("Aktivität Objekte");
		section.setDescription(
				"Attributliste zur Beschreibung, welche Objekte neu angelegt, geändert oder gelöscht werden dürfen.");

		final Composite sectionClient = toolkit.createComposite(section);
		sectionClient.setLayout(new GridLayout(1, false));

		final Composite buttonPanel = toolkit.createComposite(sectionClient);
		buttonPanel.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		buttonPanel.setLayout(GridLayoutFactory.fillDefaults().numColumns(3).equalWidth(false).create());

		final Button anlegenButton = toolkit.createButton(buttonPanel, "Anlegen", SWT.PUSH);
		anlegenButton.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		anlegenButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(SelectionEvent e) {
				final AktivitaetObjekteWizard wiz = new AktivitaetObjekteWizard(rollenAktivitaetParameterDaten);
				final WizardDialog dialog = new WizardDialog(anlegenButton.getShell(), wiz);
				if (Window.OK == dialog.open()) {
					setDirty(true);
				}
				objekteTabelle.refresh();
			}

		});

		final Button bearbeitenButton = toolkit.createButton(buttonPanel, "Bearbeiten", SWT.PUSH);
		bearbeitenButton.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		bearbeitenButton.setEnabled(false);
		bearbeitenButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(SelectionEvent e) {
				final IStructuredSelection selection = (IStructuredSelection) objekteTabelle.getSelection();
				final Object firstElement = selection.getFirstElement();
				if (firstElement instanceof AtlAktivitaetObjekteNeu) {
					final AktivitaetObjekteWizard wiz = new AktivitaetObjekteWizard(rollenAktivitaetParameterDaten,
							(AtlAktivitaetObjekteNeu) firstElement);
					final WizardDialog dialog = new WizardDialog(anlegenButton.getShell(), wiz);
					if (Window.OK == dialog.open()) {
						setDirty(true);
					}
					objekteTabelle.refresh();
				}
			}

		});

		final Button entfernenButton = toolkit.createButton(buttonPanel, "Entfernen", SWT.PUSH);
		entfernenButton.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		entfernenButton.setEnabled(false);
		entfernenButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(SelectionEvent e) {
				final IStructuredSelection selection = (IStructuredSelection) objekteTabelle.getSelection();
				final Object firstElement = selection.getFirstElement();
				if (firstElement instanceof AtlAktivitaetObjekteNeu) {
					rollenAktivitaetParameterDaten.getAktivitaetObjekte().remove(firstElement);
					setDirty(true);
					objekteTabelle.refresh();
				}
			}

		});

		final Composite treeComposite = toolkit.createComposite(sectionClient, SWT.NONE);
		treeComposite.setLayoutData(GridDataFactory.fillDefaults().grab(true, true).minSize(100, 200).create());
		final TreeColumnLayout treeLayout = new TreeColumnLayout();
		treeComposite.setLayout(treeLayout);

		objekteTabelle = new TreeViewer(treeComposite, SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);
		objekteTabelle.setContentProvider(new AktivitaetDatenTreeProvider());

		objekteTabelle.addSelectionChangedListener(event -> {
			final boolean datenSelected = event.getStructuredSelection()
					.getFirstElement() instanceof AtlAktivitaetObjekteNeu;
			bearbeitenButton.setEnabled(datenSelected);
			entfernenButton.setEnabled(datenSelected);
		});

		final TreeViewerColumn nameCloumn = new TreeViewerColumn(objekteTabelle, SWT.NONE);
		treeLayout.setColumnData(nameCloumn.getColumn(), new ColumnWeightData(50, 100));

		nameCloumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				if (element instanceof AtlAktivitaetObjekteNeu) {

					final AtlAktivitaetObjekteNeu aktObjekte = (AtlAktivitaetObjekteNeu) element;

					return "Aktivität Objekte " + "(" + aktObjekte.getKonfigurationsbereich().size() + " KB, "
							+ aktObjekte.getObjekttyp().size() + " Typen) Erzeugen/Ändern/Löschen="
							+ aktObjekte.getObjekteErzeugenAendernLoeschen();
				} else if (element instanceof NamedFeld) {
					final NamedFeld<?> feld = (NamedFeld<?>) element;
					return feld.getName();
				}
				return super.getText(element);
			}
		});

		section.setClient(sectionClient);

	}

	private void createAktivitaetenSection(FormToolkit toolkit, ScrolledForm form) {
		final Section sctnAktivittDaten = toolkit.createSection(form.getBody(), ExpandableComposite.FOCUS_TITLE
				| ExpandableComposite.EXPANDED | Section.DESCRIPTION | ExpandableComposite.TITLE_BAR);
		sctnAktivittDaten.setLayoutData(GridDataFactory.fillDefaults().grab(true, true).create());

		sctnAktivittDaten.addExpansionListener(new ExpansionAdapter() {
			@Override
			public void expansionStateChanged(final ExpansionEvent event) {
				form.reflow(true);
			}
		});
		sctnAktivittDaten.setText("Aktivität Daten");
		sctnAktivittDaten.setDescription(
				"Über die Attributliste AktivitätDaten wird spezifziert, wie die Zugriffrechte beim Datenaustausch für die entsprechende Rolle gesetzt sind.\n Es können verschiedenen Aktivitäten definiert und der Rolle zugeordnet werden. ");

		final Composite sectionClient = toolkit.createComposite(sctnAktivittDaten);
		sectionClient.setLayout(new GridLayout(1, false));

		final Composite buttonPanel = toolkit.createComposite(sectionClient);
		buttonPanel.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		buttonPanel.setLayout(GridLayoutFactory.fillDefaults().numColumns(3).equalWidth(false).create());

		final Button anlegenButton = toolkit.createButton(buttonPanel, "Anlegen", SWT.PUSH);
		anlegenButton.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		anlegenButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(SelectionEvent e) {

				final AktivitaetDatenWizard wiz = new AktivitaetDatenWizard(rollenAktivitaetParameterDaten);
				final WizardDialog dialog = new WizardDialog(anlegenButton.getShell(), wiz);
				if (Window.OK == dialog.open()) {
					setDirty(true);
				}
				datenTabelle.refresh();
			}

		});

		final Button bearbeitenButton = toolkit.createButton(buttonPanel, "Bearbeiten", SWT.PUSH);
		bearbeitenButton.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		bearbeitenButton.setEnabled(false);
		bearbeitenButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(SelectionEvent e) {
				final IStructuredSelection selection = (IStructuredSelection) datenTabelle.getSelection();
				final Object firstElement = selection.getFirstElement();
				if (firstElement instanceof AtlAktivitaetDaten) {
					final AktivitaetDatenWizard wiz = new AktivitaetDatenWizard(rollenAktivitaetParameterDaten,
							(AtlAktivitaetDaten) firstElement);
					final WizardDialog dialog = new WizardDialog(anlegenButton.getShell(), wiz);
					if (Window.OK == dialog.open()) {
						setDirty(true);
					}
					datenTabelle.refresh();
				}
			}

		});

		final Button entfernenButton = toolkit.createButton(buttonPanel, "Entfernen", SWT.PUSH);
		entfernenButton.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		entfernenButton.setEnabled(false);
		entfernenButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(SelectionEvent e) {
				final IStructuredSelection selection = (IStructuredSelection) datenTabelle.getSelection();
				final Object firstElement = selection.getFirstElement();
				if (firstElement instanceof AtlAktivitaetDaten) {
					rollenAktivitaetParameterDaten.getAktivitaetDaten().remove(firstElement);
					setDirty(true);
					datenTabelle.refresh();
				}

			}

		});

		final Composite treeComposite = toolkit.createComposite(sectionClient, SWT.NONE);
		treeComposite.setLayoutData(GridDataFactory.fillDefaults().grab(true, true).minSize(300, 200).create());
		final TreeColumnLayout treeLayout = new TreeColumnLayout();
		treeComposite.setLayout(treeLayout);

		datenTabelle = new TreeViewer(treeComposite, SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);
		datenTabelle.setContentProvider(new AktivitaetDatenTreeProvider());
		datenTabelle.addSelectionChangedListener(event -> {
			final boolean datenSelected = event.getStructuredSelection()
					.getFirstElement() instanceof AtlAktivitaetDaten;
			bearbeitenButton.setEnabled(datenSelected);
			entfernenButton.setEnabled(datenSelected);
		});

		final TreeViewerColumn nameCloumn = new TreeViewerColumn(datenTabelle, SWT.NONE);
		treeLayout.setColumnData(nameCloumn.getColumn(), new ColumnWeightData(50, 100));

		nameCloumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				if (element instanceof AtlAktivitaetDaten) {
					return ((AtlAktivitaetDaten) element).getName();
				} else if (element instanceof NamedFeld) {
					final NamedFeld<?> feld = (NamedFeld<?>) element;
					return feld.getName();
				}
				return super.getText(element);
			}
		});

		sctnAktivittDaten.setClient(sectionClient);
	}

	private void createMengenSection(FormToolkit toolkit, ScrolledForm form) {
		final Section section = toolkit.createSection(form.getBody(), ExpandableComposite.FOCUS_TITLE
				| Section.DESCRIPTION | ExpandableComposite.TITLE_BAR | ExpandableComposite.EXPANDED);
		section.setLayoutData(GridDataFactory.fillDefaults().grab(true, true).create());

		section.addExpansionListener(new ExpansionAdapter() {
			@Override
			public void expansionStateChanged(final ExpansionEvent event) {
				form.reflow(true);
			}
		});
		section.setText("Aktivität Mengen");
		section.setDescription("Attributliste zur Beschreibung, welche Mengen geändert werden dürfen.");

		final Composite sectionClient = toolkit.createComposite(section);
		sectionClient.setLayout(new GridLayout(1, false));

		final Composite buttonPanel = toolkit.createComposite(sectionClient);
		buttonPanel.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		buttonPanel.setLayout(GridLayoutFactory.fillDefaults().numColumns(3).equalWidth(false).create());

		final Button anlegenButton = toolkit.createButton(buttonPanel, "Anlegen", SWT.PUSH);
		anlegenButton.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		anlegenButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(SelectionEvent e) {
				final AktivitaetMengenWizard wiz = new AktivitaetMengenWizard(rollenAktivitaetParameterDaten);
				final WizardDialog dialog = new WizardDialog(anlegenButton.getShell(), wiz);
				if (Window.OK == dialog.open()) {
					setDirty(true);
				}
				mengenTabelle.refresh();
			}

		});

		final Button bearbeitenButton = toolkit.createButton(buttonPanel, "Bearbeiten", SWT.PUSH);
		bearbeitenButton.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		bearbeitenButton.setEnabled(false);
		bearbeitenButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(SelectionEvent e) {
				final IStructuredSelection selection = (IStructuredSelection) mengenTabelle.getSelection();
				final Object firstElement = selection.getFirstElement();
				if (firstElement instanceof AtlAktivitaetMengen) {
					final AktivitaetMengenWizard wiz = new AktivitaetMengenWizard(rollenAktivitaetParameterDaten,
							(AtlAktivitaetMengen) firstElement);
					final WizardDialog dialog = new WizardDialog(anlegenButton.getShell(), wiz);
					if (Window.OK == dialog.open()) {
						setDirty(true);
					}
					mengenTabelle.refresh();
				}
			}

		});

		final Button entfernenButton = toolkit.createButton(buttonPanel, "Entfernen", SWT.PUSH);
		entfernenButton.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		entfernenButton.setEnabled(false);
		entfernenButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(SelectionEvent e) {
				final IStructuredSelection selection = (IStructuredSelection) mengenTabelle.getSelection();
				final Object firstElement = selection.getFirstElement();
				if (firstElement instanceof AtlAktivitaetMengen) {
					rollenAktivitaetParameterDaten.getAktivitaetMengen().remove(firstElement);
					setDirty(true);
					mengenTabelle.refresh();
				}
			}

		});

		final Composite treeComposite = toolkit.createComposite(sectionClient, SWT.NONE);
		treeComposite.setLayoutData(GridDataFactory.fillDefaults().grab(true, true).minSize(100, 200).create());
		final TreeColumnLayout treeLayout = new TreeColumnLayout();
		treeComposite.setLayout(treeLayout);

		mengenTabelle = new TreeViewer(treeComposite, SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);
		mengenTabelle.setContentProvider(new AktivitaetDatenTreeProvider());

		mengenTabelle.addSelectionChangedListener(event -> {
			final boolean datenSelected = event.getStructuredSelection()
					.getFirstElement() instanceof AtlAktivitaetMengen;
			bearbeitenButton.setEnabled(datenSelected);
			entfernenButton.setEnabled(datenSelected);
		});

		final TreeViewerColumn nameCloumn = new TreeViewerColumn(mengenTabelle, SWT.NONE);
		treeLayout.setColumnData(nameCloumn.getColumn(), new ColumnWeightData(50, 100));

		nameCloumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				if (element instanceof AtlAktivitaetMengen) {

					final AtlAktivitaetMengen aktObjekte = (AtlAktivitaetMengen) element;

					return "Mengen Objekte " + "(" + aktObjekte.getKonfigurationsbereich().size() + " KB, "
							+ aktObjekte.getMengentyp().size() + " Mengen-Typen)";
				} else if (element instanceof NamedFeld) {
					final NamedFeld<?> feld = (NamedFeld<?>) element;
					return feld.getName();
				}
				return super.getText(element);
			}
		});

		section.setClient(sectionClient);

	}

	private void createRollenSection(FormToolkit toolkit, ScrolledForm form) {
		final Section section = toolkit.createSection(form.getBody(), ExpandableComposite.FOCUS_TITLE
				| Section.DESCRIPTION | ExpandableComposite.TITLE_BAR | ExpandableComposite.EXPANDED);
		section.setLayoutData(GridDataFactory.fillDefaults().grab(true, true).create());

		section.addExpansionListener(new ExpansionAdapter() {
			@Override
			public void expansionStateChanged(final ExpansionEvent event) {
				form.reflow(true);
			}
		});
		section.setText("Rollen");
		section.setDescription("Attribut zur Referenzierung des Objekttyps: typ.zugriffsRolleNeu");

		final Composite sectionClient = toolkit.createComposite(section);
		sectionClient.setLayout(new GridLayout(1, false));

		final Composite buttonPanel = toolkit.createComposite(sectionClient);
		buttonPanel.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		buttonPanel.setLayout(GridLayoutFactory.fillDefaults().numColumns(2).equalWidth(true).create());

		final Button hinzufuegenButton = toolkit.createButton(buttonPanel, "Hinzufügen", SWT.PUSH);
		hinzufuegenButton.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		hinzufuegenButton.addSelectionListener(new SelectionAdapter() {

			@SuppressWarnings("unchecked")
			@Override
			public void widgetSelected(SelectionEvent e) {
				final ElementListSelectionDialog dialog = new ElementListSelectionDialog(hinzufuegenButton.getShell(),
						new LabelProvider());
				dialog.setMultipleSelection(true);
				dialog.setTitle("Auswahl der Rollen");
				dialog.setMessage("Wählen Sie eine oder mehrere Rollen aus.");

				final ZugriffsRolleNeu rolleNeu = getEditorInput().getAdapter(ZugriffsRolleNeu.class);

				final Object[] rollen = RahmenwerkService.getService().getObjektFactory()
						.bestimmeModellobjekte(ZugriffsRolleNeu.PID).stream()
						.filter(r -> !rollenAktivitaetParameterDaten.getRolle().contains(r))
						.filter(r -> !rolleNeu.equals(r)).toArray();

				dialog.setElements(rollen);
				if (Window.OK == dialog.open()) {
					rollenAktivitaetParameterDaten.getRolle()
							.addAll((Collection<? extends ZugriffsRolleNeu>) Arrays.asList(dialog.getResult()));
					setDirty(true);
				}
				rollenTabelle.setInput(rollenAktivitaetParameterDaten.getRolle());
			}

		});

		final Button entfernenButton = toolkit.createButton(buttonPanel, "Entfernen", SWT.PUSH);
		entfernenButton.setLayoutData(GridDataFactory.fillDefaults().grab(true, false).create());
		entfernenButton.setEnabled(false);
		entfernenButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(SelectionEvent e) {
				final IStructuredSelection selection = (IStructuredSelection) rollenTabelle.getSelection();
				final Object firstElement = selection.getFirstElement();
				if (firstElement instanceof ZugriffsRolleNeu) {
					rollenAktivitaetParameterDaten.getRolle().remove(firstElement);
					setDirty(true);
					rollenTabelle.setInput(rollenAktivitaetParameterDaten.getRolle());
				}
			}

		});

		final Composite treeComposite = toolkit.createComposite(sectionClient, SWT.NONE);
		treeComposite.setLayoutData(GridDataFactory.fillDefaults().grab(true, true).minSize(100, 200).create());
		final TableColumnLayout layout = new TableColumnLayout();
		treeComposite.setLayout(layout);

		rollenTabelle = new TableViewer(treeComposite, SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);
		rollenTabelle.setContentProvider(new ArrayContentProvider());

		rollenTabelle.addSelectionChangedListener(event -> {
			final boolean datenSelected = event.getStructuredSelection().getFirstElement() instanceof ZugriffsRolleNeu;
			entfernenButton.setEnabled(datenSelected);
		});

		final TableViewerColumn nameCloumn = new TableViewerColumn(rollenTabelle, SWT.NONE);
		layout.setColumnData(nameCloumn.getColumn(), new ColumnWeightData(50, 100));

		nameCloumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				if (element instanceof ZugriffsRolleNeu) {
					final ZugriffsRolleNeu rolle = (ZugriffsRolleNeu) element;
					return rolle.getName();
				}
				return super.getText(element);
			}
		});

		section.setClient(sectionClient);

	}

	@Override
	public RollenEditor getEditor() {
		return (RollenEditor) super.getEditor();
	}

	@Override
	public boolean isDirty() {
		return dirty;
	}

	/**
	 * Setzt das lokale Dirty - Flag.
	 *
	 * @param newDirtyFlag der neue dirty - Zustand des Editors.
	 */
	protected void setDirty(final boolean newDirtyFlag) {
		dirty = newDirtyFlag;
		firePropertyChange(PROP_DIRTY);
		if (getManagedForm() != null) {
			getManagedForm().getForm().getDisplay().asyncExec(() -> getManagedForm().dirtyStateChanged());

		}
	}

	@Override
	public void doSave(IProgressMonitor monitor) {
		super.doSave(monitor);

		final UrlasserInfoDatenDialog dialog = new UrlasserInfoDatenDialog(getSite().getShell(),
				(verbindung, urlasserInfo) -> {
					final ZugriffsRolleNeu rolleNeu = getEditorInput().getAdapter(ZugriffsRolleNeu.class);

					final ObjektFactory objektFactory = RahmenwerkService.getService().getObjektFactory();
					try {
						monitor.beginTask("Speichern", 5);
						rolleNeu.getPdRollenAktivitaet().anmeldenSender();
						monitor.worked(1);
						final AtlUrlasser urlasser = new AtlUrlasser();
						urlasser.setBenutzerReferenz(
								(Benutzer) objektFactory.getModellobjekt(urlasserInfo.getBenutzer()));
						urlasser.setUrsache(urlasserInfo.getUrsache());
						urlasser.setVeranlasser(urlasserInfo.getVeranlasser());
						monitor.worked(1);
						rollenAktivitaetParameterDaten.setUrlasser(urlasser);
						monitor.worked(1);
						rolleNeu.getPdRollenAktivitaet().sendeDatum(rollenAktivitaetParameterDaten);
						monitor.worked(1);
						rolleNeu.getPdRollenAktivitaet().abmeldenSender();
						setDirty(false);
					} catch (AnmeldeException | DatensendeException e) {
						monitor.setCanceled(true);
						throw new IllegalStateException("Das Speichern neuer Aktivitäten für die Rolle  \"" + rolleNeu
								+ "\" ist fehlgeschlagen.", e);
					}
				});

		dialog.open();
		monitor.done();
	}

}
