/*
 * Rahmenwerk-Plug-in "Benutzerverwaltung/Zugriffsrechte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.benutzervew.internal;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import de.bsvrz.buv.plugin.benutzervew.PluginBenutzerVew;
import de.bsvrz.buv.plugin.benutzervew.data.Benutzerverwaltung;
import de.bsvrz.buv.rw.basislib.Rahmenwerk;
import de.bsvrz.buv.rw.basislib.berechtigung.Berechtigungen;
import de.bsvrz.buv.rw.basislib.dav.DavVerbindungsEvent;
import de.bsvrz.buv.rw.basislib.dav.DavVerbindungsListener;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Interne Komponente zur Bereitstellung der für das Plug-in erforderlichen
 * Services.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public class RahmenwerkService {

	private static RahmenwerkService service;

	private Rahmenwerk rahmenWerk;
	private Berechtigungen berechtigungen;

	/** Benutzerverwaltungsobjekt. */
	private final Benutzerverwaltung benutzerverwaltung = new Benutzerverwaltung();

	private ObjektFactory objektFactory;

	/** Die minimale Anzahl von Zeichen des Passwortes. */
	public static final int BENUTZERPASSWORT_MIN_LAENGE = 5;

	protected static final Debug LOGGER = Debug.getLogger();

	/**
	 * liefert die globale Instanz der Komponente.
	 *
	 * @return die Komponente
	 */
	public static RahmenwerkService getService() {
		return RahmenwerkService.service;
	}

	/** aktiviert die Komponente. */
	protected void activate() {
		RahmenwerkService.service = this;
	}

	/** deaktiviert die Komponente. */
	protected void deactivate() {
		RahmenwerkService.service = null;
	}

	/**
	 * bindet einen Rahmenwerk-Service an die Komponente.
	 *
	 * @param newRahmenWerk der Service
	 */
	protected void bindRahmenwerk(final Rahmenwerk newRahmenWerk) {
		this.rahmenWerk = newRahmenWerk;
		this.rahmenWerk.addDavVerbindungsListener(new DavVerbindungsListener() {

			@Override
			public void verbindungHergestellt(final DavVerbindungsEvent event) {
				PluginBenutzerVew.getDefault().getLog().log(new Status(IStatus.INFO, PluginBenutzerVew.PLUGIN_ID,
						"Initialisiere Berechtigungsverwaltung Benutzerverwaltung/Zugriffsrechte"));
				benutzerverwaltung.connect(rahmenWerk.getDavVerbindung());
			}

			@Override
			public boolean verbindungHalten(final DavVerbindungsEvent event) {
				return false;
			}

			@Override
			public void verbindungGetrennt(final DavVerbindungsEvent event) {
				PluginBenutzerVew.getDefault().getLogger().info("Benutzer-Plugin: Verbindung abgebaut");
				benutzerverwaltung.disconnect();
			}
		});
	}

	/**
	 * entfernt einen Rahmenwerk-Service von der Komponente.
	 *
	 * @param newRahmenWerk der Service
	 */
	protected void unbindRahmenwerk(final Rahmenwerk newRahmenWerk) {
		this.rahmenWerk = null;
	}

	/**
	 * liefert den {@link Rahmenwerk}-Service.
	 *
	 * @return den Service
	 */
	public Rahmenwerk getRahmenWerk() {
		return rahmenWerk;
	}

	/**
	 * bindet einen {@link Berechtigungen}-Service an die Komponente.
	 *
	 * @param newBerechtigungen der Service
	 */
	protected void bindBerechtigungen(final Berechtigungen newBerechtigungen) {
		this.berechtigungen = newBerechtigungen;
	}

	/**
	 * entfernt einen {@link Berechtigungen}-Service von der Komponente.
	 *
	 * @param newBerechtigungen der Service
	 */
	protected void unbindBerechtigungen(final Berechtigungen newBerechtigungen) {
		this.berechtigungen = null;
	}

	/**
	 * liefert den {@link Berechtigungen}-Service.
	 *
	 * @return den Service
	 */
	public Berechtigungen getBerechtigungen() {
		return berechtigungen;
	}

	/**
	 * Gibt das Benutzerverwaltungsobjekt zur&uuml;ck.
	 *
	 * @return Benutzerverwaltungsobjekt
	 */
	public Benutzerverwaltung getBenutzerverwaltung() {
		return benutzerverwaltung;
	}

	protected void bindObjektFactory(final ObjektFactory objektFactory) {
		this.objektFactory = objektFactory;
	}

	protected void unbindObjektFactory(final ObjektFactory objektFactory) {
		this.objektFactory = null;
	}

	public final ObjektFactory getObjektFactory() {
		return objektFactory;
	}

}
