/*
 * Rahmenwerk-Plug-in "Benutzerverwaltung/Zugriffsrechte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.benutzervew.views;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.ResourceBundle;

import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.nebula.paperclips.core.BigPrint;
import org.eclipse.nebula.paperclips.core.ImagePrint;
import org.eclipse.nebula.paperclips.core.border.LineBorder;
import org.eclipse.nebula.paperclips.core.grid.DefaultGridLook;
import org.eclipse.nebula.paperclips.core.grid.GridColumn;
import org.eclipse.nebula.paperclips.core.grid.GridPrint;
import org.eclipse.nebula.paperclips.core.page.PagePrint;
import org.eclipse.nebula.paperclips.core.text.TextPrint;
import org.eclipse.nebula.paperclips.core.text.TextStyle;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.ViewPart;

import de.bsvrz.buv.plugin.benutzervew.PluginBenutzerVew;
import de.bsvrz.buv.plugin.benutzervew.actions.ViewAktualisierenAktion;
import de.bsvrz.buv.plugin.benutzervew.data.Benutzer;
import de.bsvrz.buv.plugin.benutzervew.data.BenutzerListener;
import de.bsvrz.buv.plugin.benutzervew.data.BerechtigungsKlasse;
import de.bsvrz.buv.plugin.benutzervew.data.DavVerbindungSwitcher;
import de.bsvrz.buv.plugin.benutzervew.data.DavVerbindungView;
import de.bsvrz.buv.plugin.benutzervew.internal.RahmenwerkService;
import de.bsvrz.buv.rw.basislib.printing.RwPrintable;
import de.bsvrz.dav.daf.main.ClientDavInterface;

/**
 * Ansicht zur Anzeige der im System definierten Nutzer als Liste.
 *
 * @author BitCtrl Systems GmbH, Gieseler
 *
 */
public class BenutzerListe extends ViewPart implements BenutzerListener, RwPrintable, DavVerbindungView {

	/** Die View-ID. */
	public static final String VIEW_ID = BenutzerListe.class.getName();

	/** Die Context-Hilfe-ID. */
	private static final String HILFE_ID = PluginBenutzerVew.PLUGIN_ID + "." + BenutzerListe.class.getSimpleName();

	/** ResourceBundle. */
	private ResourceBundle resourceBundle;

	/**
	 * Spalte der Liste.
	 *
	 * @author BitCtrl Systems GmbH, Steffen Gieseler
	 *
	 */
	private static class SpaltenDefinition {
		/**
		 * Konstruktor.
		 *
		 * @param spaltenname   Name
		 * @param tooltip       Tooltip
		 * @param breite        Breite
		 * @param spalteKennung Kennung
		 */
		SpaltenDefinition(final String spaltenname, final String tooltip, final int breite,
				final SpaltenIndex spalteKennung) {
			this.spaltenname = spaltenname;
			this.tooltip = tooltip;
			this.breite = breite;
			this.spaltenindex = spalteKennung;
		}

		/** Name der Spalte. */
		private final String spaltenname;

		/** Tooltip der Spalte. */
		private final String tooltip;

		/** Breite der Spalte. */
		private final int breite;

		/** Index der Spalte. */
		private final SpaltenIndex spaltenindex;
	}

	/**
	 * Spalten der Ansicht.
	 *
	 * @author BitCtrl Systems GmbH, Steffen Gieseler
	 *
	 */
	public enum SpaltenIndex {
		/** Spalte 'Benutzerkennung'. */
		SPALTE_KENNUNG,
		/** Spalte 'Benutzerklassen'. */
		SPALTE_BENUTZERKLASSE,
		/** Spalte 'Name'. */
		SPALTE_NAME,
		/** Spalte 'Vorname'. */
		SPALTE_VORNAME,
		/** Spalte 'Zweiter Vorname'. */
		SPALTE_ZWEITERVORNAME,
		/** Spalte 'Organisation'. */
		SPALTE_ORGANISATION,
		/** Spalte 'E-Mail'. */
		SPALTE_EMAIL
	}

	/**
	 * Die Spalten der Liste.
	 */
	private final SpaltenDefinition[] spalten = {
			new SpaltenDefinition("Benutzerkennung", "Benutzerkennung", 200, SpaltenIndex.SPALTE_KENNUNG),
			new SpaltenDefinition("Berechtigungsklassen", "Berechtigungsklassen", 200,
					SpaltenIndex.SPALTE_BENUTZERKLASSE),
			new SpaltenDefinition("Name", "Name", 200, SpaltenIndex.SPALTE_NAME),
			new SpaltenDefinition("Vorname", "Vorname", 200, SpaltenIndex.SPALTE_VORNAME),
			new SpaltenDefinition("Zweiter Vorname", "Zweiter Vorname", 200, SpaltenIndex.SPALTE_ZWEITERVORNAME),
			new SpaltenDefinition("Organisation", "Organisation", 200, SpaltenIndex.SPALTE_ORGANISATION),
			new SpaltenDefinition("E-Mail", "E-Mail", 200, SpaltenIndex.SPALTE_EMAIL), };

	/** Der Filter für deaktivierte Nutzer ist aktiviert? */
	private boolean filterDeactivated = true;

	/** Der Sorter. */
	private BenutzerListeSorter sorter;

	private TableViewer viewer;

	private DavVerbindungSwitcher verbindungsListener;

	public TableViewer getViewer() {
		return viewer;
	}

	@Override
	public void createPartControl(final Composite parent) {
		resourceBundle = PluginBenutzerVew.getDefault().getResourceBundle();

		viewer = new TableViewer(parent, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION);

		getViewSite().setSelectionProvider(viewer);
		createContextMenu();
		sorter = new BenutzerListeSorter(false);

		viewer.getTable().setHeaderVisible(true);
		viewer.getTable().setLinesVisible(true);

		for (final SpaltenDefinition def : spalten) {
			createTableColumn(viewer.getTable(), SpaltenIndex.valueOf(def.spaltenindex.name()).ordinal(),
					def.spaltenname, def.tooltip, true, SWT.LEFT, def.breite, false, getSorter(def.spaltenindex));
		}

		viewer.setContentProvider(new ArrayContentProvider());
		viewer.setLabelProvider(new BenutzerListeLabelProvider());

		sorter.resetState();

		final IActionBars bars = getViewSite().getActionBars();
		final IToolBarManager toolbar = bars.getToolBarManager();
		toolbar.add(new ViewAktualisierenAktion(viewer));

		// Online-Hilfe hinzufügen
		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, BenutzerListe.HILFE_ID);

		verbindungsListener = new DavVerbindungSwitcher(this);
		RahmenwerkService.getService().getRahmenWerk().addDavVerbindungsListener(verbindungsListener);
		getSite().setSelectionProvider(viewer);

		setDav(RahmenwerkService.getService().getRahmenWerk().getDavVerbindung());
		RahmenwerkService.getService().getBenutzerverwaltung().addBenutzerListener(this);
	}

	/**
	 * Gibt den Sorter zur Spalte zur&uuml;ck.
	 *
	 * @param index Spaltenindex
	 * @return BenutzerSpaltenSorter
	 */
	private BenutzerSpaltenSorter getSorter(final SpaltenIndex index) {
		if (index == SpaltenIndex.SPALTE_KENNUNG) {
			return new BenutzerSpaltenSorter((o1, o2) -> o1.getName().compareToIgnoreCase(o2.getName()));
		} else if (index == SpaltenIndex.SPALTE_BENUTZERKLASSE) {
			return new BenutzerSpaltenSorter((o1, o2) -> o1.getBerechtigungsKlassenAsString()
					.compareToIgnoreCase(o2.getBerechtigungsKlassenAsString()));
		} else if (index == SpaltenIndex.SPALTE_NAME) {
			return new BenutzerSpaltenSorter((o1, o2) -> o1.getNachname().compareToIgnoreCase(o2.getNachname()));
		} else if (index == SpaltenIndex.SPALTE_VORNAME) {
			return new BenutzerSpaltenSorter((o1, o2) -> o1.getVorname().compareToIgnoreCase(o2.getVorname()));
		} else if (index == SpaltenIndex.SPALTE_ZWEITERVORNAME)

		{
			return new BenutzerSpaltenSorter(
					(o1, o2) -> o1.getZweiterVorname().compareToIgnoreCase(o2.getZweiterVorname()));
		} else if (index == SpaltenIndex.SPALTE_ORGANISATION) {
			return new BenutzerSpaltenSorter(
					(o1, o2) -> o1.getOrganisation().compareToIgnoreCase(o2.getOrganisation()));
		} else if (index == SpaltenIndex.SPALTE_EMAIL) {
			return new BenutzerSpaltenSorter(
					(o1, o2) -> o1.getEmailAdresse().compareToIgnoreCase(o2.getEmailAdresse()));
		} else {
			return null;
		}
	}

	/**
	 * Ermittelt, ob der Filter für deaktivierte Nutzer deaktiviert ist.
	 *
	 * @return Filterstatus
	 */
	public boolean isFilterDeactivated() {
		return filterDeactivated;
	}

	/**
	 * Erzeugt eine Spalte.
	 *
	 * @param parent          Parent-Composite
	 * @param spaltenindex    Index der Spalte
	 * @param name            Name der Spalte
	 * @param tooltip         Tooltip der Spalte
	 * @param resizable
	 * @param alignment
	 * @param width
	 * @param createMenuEntry
	 * @param columnSorter
	 * @return TableColumn
	 */
	private TableColumn createTableColumn(final Table parent, final int spaltenindex, final String name,
			final String tooltip, final boolean resizable, final int alignment, final int width,
			final boolean createMenuEntry, final BenutzerSpaltenSorter columnSorter) {

		sorter.addTableColumnSorter(spaltenindex, columnSorter);

		final TableColumn column = new TableColumn(parent, alignment);
		column.setText(name);
		column.setToolTipText(tooltip);
		column.setWidth(width);
		column.setResizable(resizable);
		column.setMoveable(false);

		column.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final Cursor loadCursor = new Cursor(parent.getDisplay(), SWT.CURSOR_WAIT);
				parent.setCursor(loadCursor);
				sorter.setColumn(spaltenindex);
				sortTable();
				parent.setCursor(null);
				loadCursor.dispose();
			}
		});

		return column;
	}

	/**
	 * Erzeugt das Kontextmenü der Ansicht.
	 */
	private void createContextMenu() {
		// Create menu manager.
		final MenuManager menuMgr = new BenutzerContextMenu(viewer);

		// Create menu.
		final Menu menu = menuMgr.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);

		// Register menu for extension.
		getViewSite().registerContextMenu(menuMgr, viewer);
	}

	/**
	 * Sortiert die Tabelle.
	 */
	private void sortTable() {
		// store current selection
		IStructuredSelection selection = (IStructuredSelection) viewer.getSelection();

		// sort viewer content
		final List<?> input = new ArrayList<>((Collection<?>) viewer.getInput());
		Collections.sort(input, sorter);

		// update viewer
		viewer.setInput(input);
		updateSortIndicator();
		if (!input.isEmpty()) {
			if (selection.isEmpty()) {
				selection = new StructuredSelection(input.get(0));
			}
			viewer.setSelection(selection);
		}
	}

	/**
	 * Aktualisiert den Sortier-Indikator.
	 */
	private void updateSortIndicator() {
		viewer.getTable().setSortColumn(viewer.getTable().getColumn(sorter.getColumn()));
		viewer.getTable()
				.setSortDirection((sorter.getDirection(sorter
						.getColumn()) == de.bsvrz.buv.plugin.benutzervew.views.BenutzerListeSorter.Direction.ASCENDING)
								? SWT.UP
								: SWT.DOWN);
	}

	/**
	 * liefert die ausgew&auml;hlte Nutzergruppe.
	 *
	 * @return die aktuell selektierte Nutzergruppe
	 */
	public BerechtigungsKlasse getSelectedBerechtigungsklasse() {
		if (viewer.getSelection() != null) {
			final Object obj = ((StructuredSelection) viewer.getSelection()).getFirstElement();
			if (obj instanceof BerechtigungsKlasse) {
				return (BerechtigungsKlasse) obj;
			}
		}
		return null;
	}

	/**
	 * Liefert den ausgewählten Nutzer.
	 *
	 * @return den aktuell ausgewählten Nutzer.
	 */
	public Benutzer getSelectedBenutzer() {
		if (viewer.getSelection() != null) {
			final Object obj = ((StructuredSelection) viewer.getSelection()).getFirstElement();
			if (obj instanceof Benutzer) {
				return (Benutzer) obj;
			}
		}
		return null;
	}

	/**
	 * Deaktiviert/aktiviert den Filter für deaktivierte Nutzer.
	 *
	 * @param filterDeactivated <code>true</code> == Filter wird deaktiviert.
	 */
	public void setFilterDeactivated(final boolean filterDeactivated) {
		this.filterDeactivated = filterDeactivated;
		viewer.refresh();
	}

	@Override
	public void setFocus() {
		viewer.getControl().setFocus();
	}

	public void aktualisieren() {
		if ((viewer != null) && !viewer.getControl().isDisposed()) {
			Display.getDefault().asyncExec(() -> {
				if ((viewer != null) && !viewer.getControl().isDisposed()
						&& (RahmenwerkService.getService().getBenutzerverwaltung() != null)) {
					viewer.setInput(RahmenwerkService.getService().getBenutzerverwaltung().getBenutzer());
					sortTable();
				}
			});
		}
	}

	@Override
	public PagePrint getDruckAuftrag() {

		final String header = resourceBundle.getString("BenutzerlisteDruck.header");

		final GridPrint grid = new GridPrint();
		grid.addColumn(new GridColumn(GridColumn.DEFAULT_ALIGN, GridColumn.DEFAULT_SIZE, GridColumn.DEFAULT_WEIGHT));
		grid.add(new TextPrint(header, new FontData("Arial", 14, SWT.BOLD)));

		final GridPrint tabelle = new GridPrint();
		final DefaultGridLook look = new DefaultGridLook();
		look.setHeaderBackground(new RGB(200, 200, 200));

		TextStyle textStyle = new TextStyle();
		final TextStyle headerStyle = textStyle.font("Arial", 10, SWT.BOLD);
		textStyle = textStyle.font("Arial", 10, SWT.NORMAL);

		look.setCellBorder(new LineBorder(new RGB(0, 0, 0)));
		tabelle.setLook(look);

		final TableViewer tviewer = viewer;

		for (final TableColumn column : tviewer.getTable().getColumns()) {
			final GridColumn gridColumn = new GridColumn(SWT.LEFT, GridColumn.DEFAULT_SIZE, 1);
			tabelle.addColumn(gridColumn);
			tabelle.addHeader(headerStyle.create(column.getText()));
		}

		for (final TableItem item : tviewer.getTable().getItems()) {
			for (int colIdx = 0; colIdx < tviewer.getTable().getColumns().length; colIdx++) {
				final Image image = item.getImage(colIdx);
				if (image != null) {
					tabelle.add(new ImagePrint(image.getImageData()));
				} else {
					tabelle.add(textStyle.create(item.getText(colIdx)));
				}
			}
		}

		grid.add(tabelle);

		return new PagePrint(new BigPrint(grid));
	}

	@Override
	public String getTitel() {
		return resourceBundle.getString("BenutzerlisteDruck.titel");
	}

	@Override
	public void benutzerAdded(Benutzer benutzer) {
		Display.getDefault().asyncExec(() -> {
			viewer.setInput(RahmenwerkService.getService().getBenutzerverwaltung().getBenutzer());
			sortTable();
		});
	}

	@Override
	public void benutzerRemoved(Benutzer benutzer) {
		Display.getDefault().asyncExec(() -> {
			viewer.setInput(RahmenwerkService.getService().getBenutzerverwaltung().getBenutzer());
			sortTable();
		});
	}

	@Override
	public void benutzerChanged(Benutzer benutzer) {
		Display.getDefault().asyncExec(() -> viewer.refresh(benutzer));
	}

	@Override
	public void dispose() {
		RahmenwerkService.getService().getRahmenWerk().removeDavVerbindungsListener(verbindungsListener);
		super.dispose();
	}

	@Override
	public void setDav(ClientDavInterface dav) {
		if (dav == null) {
			setContentDescription(resourceBundle.getString("OfflineMeldung.label"));
			viewer.setInput(null);
			viewer.getControl().setEnabled(false);
		} else {
			setContentDescription("");
			viewer.setInput(RahmenwerkService.getService().getBenutzerverwaltung().getBenutzer());
			viewer.getControl().setEnabled(true);
		}

		viewer.refresh();
	}
}
